/**
 * <h1>Assertions</h1>
 * <p>
 * If you're migrating from the old <code>&#64;aws-cdk/assert</code> library, first use this migration guide to migrate from <code>&#64;aws-cdk/assert</code> to <code>&#64;aws-cdk/assertions</code> found in
 * <a href="https://github.com/aws/aws-cdk/blob/v1-main/packages/&#64;aws-cdk/assertions/MIGRATING.md">our GitHub repository</a>. Then, you can migrate your application to AWS CDK v2 in order to use this library using <a href="https://docs.aws.amazon.com/cdk/v2/guide/migrating-v2.html">this guide</a>.
 * <p>
 * Functions for writing test asserting against CDK applications, with focus on CloudFormation templates.
 * <p>
 * The <code>Template</code> class includes a set of methods for writing assertions against CloudFormation templates. Use one of the <code>Template.fromXxx()</code> static methods to create an instance of this class.
 * <p>
 * To create <code>Template</code> from CDK stack, start off with:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.assertions.Template;
 * 
 * Stack stack = new Stack();
 * // ...
 * Template template = Template.fromStack(stack);
 * </pre></blockquote>
 * <p>
 * Alternatively, assertions can be run on an existing CloudFormation template -
 * <p>
 * <blockquote><pre>
 * String templateJson = "{ \"Resources\": ... }"; /* The CloudFormation template as JSON serialized string. *&#47;
 * Template template = Template.fromString(templateJson);
 * </pre></blockquote>
 * <p>
 * <strong>Cyclical Resources Note</strong>
 * <p>
 * If allowing cyclical references is desired, for example in the case of unprocessed Transform templates, supply TemplateParsingOptions and
 * set skipCyclicalDependenciesCheck to true. In all other cases, will fail on detecting cyclical dependencies.
 * <p>
 * <h2>Full Template Match</h2>
 * <p>
 * The simplest assertion would be to assert that the template matches a given
 * template.
 * <p>
 * <blockquote><pre>
 * template.templateMatches(Map.of(
 *         "Resources", Map.of(
 *                 "BarLogicalId", Map.of(
 *                         "Type", "Foo::Bar",
 *                         "Properties", Map.of(
 *                                 "Baz", "Qux")))));
 * </pre></blockquote>
 * <p>
 * By default, the <code>templateMatches()</code> API will use the an 'object-like' comparison,
 * which means that it will allow for the actual template to be a superset of the
 * given expectation. See <a href="#special-matchers">Special Matchers</a> for details on how
 * to change this.
 * <p>
 * Snapshot testing is a common technique to store a snapshot of the output and
 * compare it during future changes. Since CloudFormation templates are human readable,
 * they are a good target for snapshot testing.
 * <p>
 * The <code>toJSON()</code> method on the <code>Template</code> can be used to produce a well formatted JSON
 * of the CloudFormation template that can be used as a snapshot.
 * <p>
 * See <a href="https://jestjs.io/docs/snapshot-testing">Snapshot Testing in Jest</a> and <a href="https://json-snapshot.github.io/">Snapshot
 * Testing in Java</a>.
 * <p>
 * <h2>Counting Resources</h2>
 * <p>
 * This module allows asserting the number of resources of a specific type found
 * in a template.
 * <p>
 * <blockquote><pre>
 * template.resourceCountIs("Foo::Bar", 2);
 * </pre></blockquote>
 * <p>
 * You can also count the number of resources of a specific type whose <code>Properties</code>
 * section contains the specified properties:
 * <p>
 * <blockquote><pre>
 * template.resourcePropertiesCountIs("Foo::Bar", Map.of(
 *         "Foo", "Bar",
 *         "Baz", 5,
 *         "Qux", List.of("Waldo", "Fred")), 1);
 * </pre></blockquote>
 * <p>
 * <h2>Resource Matching &amp; Retrieval</h2>
 * <p>
 * Beyond resource counting, the module also allows asserting that a resource with
 * specific properties are present.
 * <p>
 * The following code asserts that the <code>Properties</code> section of a resource of type
 * <code>Foo::Bar</code> contains the specified properties -
 * <p>
 * <blockquote><pre>
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Lorem", "Ipsum",
 *         "Baz", 5,
 *         "Qux", List.of("Waldo", "Fred")));
 * </pre></blockquote>
 * <p>
 * You can also assert that the <code>Properties</code> section of all resources of type
 * <code>Foo::Bar</code> contains the specified properties -
 * <p>
 * <blockquote><pre>
 * template.allResourcesProperties("Foo::Bar", Map.of(
 *         "Lorem", "Ipsum",
 *         "Baz", 5,
 *         "Qux", List.of("Waldo", "Fred")));
 * </pre></blockquote>
 * <p>
 * Alternatively, if you would like to assert the entire resource definition, you
 * can use the <code>hasResource()</code> API.
 * <p>
 * <blockquote><pre>
 * template.hasResource("Foo::Bar", Map.of(
 *         "Properties", Map.of("Lorem", "Ipsum"),
 *         "DependsOn", List.of("Waldo", "Fred")));
 * </pre></blockquote>
 * <p>
 * You can also assert the definitions of all resources of a type using the
 * <code>allResources()</code> API.
 * <p>
 * <blockquote><pre>
 * template.allResources("Foo::Bar", Map.of(
 *         "Properties", Map.of("Lorem", "Ipsum"),
 *         "DependsOn", List.of("Waldo", "Fred")));
 * </pre></blockquote>
 * <p>
 * Beyond assertions, the module provides APIs to retrieve matching resources.
 * The <code>findResources()</code> API is complementary to the <code>hasResource()</code> API, except,
 * instead of asserting its presence, it returns the set of matching resources.
 * <p>
 * By default, the <code>hasResource()</code> and <code>hasResourceProperties()</code> APIs perform deep
 * partial object matching. This behavior can be configured using matchers.
 * See subsequent section on <a href="#special-matchers">special matchers</a>.
 * <p>
 * <h2>Output and Mapping sections</h2>
 * <p>
 * The module allows you to assert that the CloudFormation template contains an Output
 * that matches specific properties. The following code asserts that a template contains
 * an Output with a <code>logicalId</code> of <code>Foo</code> and the specified properties -
 * <p>
 * <blockquote><pre>
 * Map&lt;String, Object&gt; expected = Map.of(
 *         "Value", "Bar",
 *         "Export", Map.of("Name", "ExportBaz"));
 * template.hasOutput("Foo", expected);
 * </pre></blockquote>
 * <p>
 * If you want to match against all Outputs in the template, use <code>*</code> as the <code>logicalId</code>.
 * <p>
 * <blockquote><pre>
 * template.hasOutput("*", Map.of(
 *         "Value", "Bar",
 *         "Export", Map.of("Name", "ExportBaz")));
 * </pre></blockquote>
 * <p>
 * <code>findOutputs()</code> will return a set of outputs that match the <code>logicalId</code> and <code>props</code>,
 * and you can use the <code>'*'</code> special case as well.
 * <p>
 * <blockquote><pre>
 * Map&lt;String, Map&lt;String, Object&gt;&gt; result = template.findOutputs("*", Map.of("Value", "Fred"));
 * expect(result.Foo).toEqual(Map.of("Value", "Fred", "Description", "FooFred"));
 * expect(result.Bar).toEqual(Map.of("Value", "Fred", "Description", "BarFred"));
 * </pre></blockquote>
 * <p>
 * The APIs <code>hasMapping()</code>, <code>findMappings()</code>, <code>hasCondition()</code>, and <code>hasCondtions()</code> provide similar functionalities.
 * <p>
 * <h2>Special Matchers</h2>
 * <p>
 * The expectation provided to the <code>hasXxx()</code>, <code>findXxx()</code> and <code>templateMatches()</code>
 * APIs, besides carrying literal values, as seen in the above examples, also accept
 * special matchers.
 * <p>
 * They are available as part of the <code>Match</code> class.
 * <p>
 * <h3>Object Matchers</h3>
 * <p>
 * The <code>Match.objectLike()</code> API can be used to assert that the target is a superset
 * object of the provided pattern.
 * This API will perform a deep partial match on the target.
 * Deep partial matching is where objects are matched partially recursively. At each
 * level, the list of keys in the target is a subset of the provided pattern.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": {
 * //           "Wobble": "Flob",
 * //           "Bob": "Cat"
 * //         }
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Match.objectLike(Map.of(
 *                 "Wobble", "Flob"))));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Match.objectLike(Map.of(
 *                 "Brew", "Coffee"))));
 * </pre></blockquote>
 * <p>
 * The <code>Match.objectEquals()</code> API can be used to assert a target as a deep exact
 * match.
 * <p>
 * <h3>Presence and Absence</h3>
 * <p>
 * The <code>Match.absent()</code> matcher can be used to specify that a specific
 * value should not exist on the target. This can be used within <code>Match.objectLike()</code>
 * or outside of any matchers.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": {
 * //           "Wobble": "Flob",
 * //         }
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Match.objectLike(Map.of(
 *                 "Bob", Match.absent()))));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Match.objectLike(Map.of(
 *                 "Wobble", Match.absent()))));
 * </pre></blockquote>
 * <p>
 * The <code>Match.anyValue()</code> matcher can be used to specify that a specific value should be found
 * at the location. This matcher will fail if when the target location has null-ish values
 * (i.e., <code>null</code> or <code>undefined</code>).
 * <p>
 * This matcher can be combined with any of the other matchers.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": {
 * //           "Wobble": ["Flob", "Flib"],
 * //         }
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Map.of(
 *                 "Wobble", List.of(Match.anyValue(), Match.anyValue()))));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Map.of(
 *                 "Wimble", Match.anyValue())));
 * </pre></blockquote>
 * <p>
 * <h3>Array Matchers</h3>
 * <p>
 * The <code>Match.arrayWith()</code> API can be used to assert that the target is equal to or a subset
 * of the provided pattern array.
 * This API will perform subset match on the target.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": ["Flob", "Cat"]
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Match.arrayWith(List.of("Flob"))));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Match.objectLike(Map.of(
 *         "Fred", Match.arrayWith(List.of("Wobble")))));
 * </pre></blockquote>
 * <p>
 * <em>Note:</em> The list of items in the pattern array should be in order as they appear in the
 * target array. Out of order will be recorded as a match failure.
 * <p>
 * Alternatively, the <code>Match.arrayEquals()</code> API can be used to assert that the target is
 * exactly equal to the pattern array.
 * <p>
 * <h3>String Matchers</h3>
 * <p>
 * The <code>Match.stringLikeRegexp()</code> API can be used to assert that the target matches the
 * provided regular expression.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Template": "const includeHeaders = true;"
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Template", Match.stringLikeRegexp("includeHeaders = (true|false)")));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Template", Match.stringLikeRegexp("includeHeaders = null")));
 * </pre></blockquote>
 * <p>
 * <h3>Not Matcher</h3>
 * <p>
 * The not matcher inverts the search pattern and matches all patterns in the path that does
 * not match the pattern specified.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": ["Flob", "Cat"]
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", Match.not(List.of("Flob"))));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Match.objectLike(Map.of(
 *         "Fred", Match.not(List.of("Flob", "Cat")))));
 * </pre></blockquote>
 * <p>
 * <h3>Serialized JSON</h3>
 * <p>
 * Often, we find that some CloudFormation Resource types declare properties as a string,
 * but actually expect JSON serialized as a string.
 * For example, the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-buildspec"><code>BuildSpec</code> property of <code>AWS::CodeBuild::Project</code></a>,
 * the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-stepfunctions-statemachine.html#cfn-stepfunctions-statemachine-definition"><code>Definition</code> property of <code>AWS::StepFunctions::StateMachine</code></a>,
 * to name a couple.
 * <p>
 * The <code>Match.serializedJson()</code> matcher allows deep matching within a stringified JSON.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Baz": "{ \"Fred\": [\"Waldo\", \"Willow\"] }"
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * // The following will NOT throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Baz", Match.serializedJson(Map.of(
 *                 "Fred", Match.arrayWith(List.of("Waldo"))))));
 * 
 * // The following will throw an assertion error
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Baz", Match.serializedJson(Map.of(
 *                 "Fred", List.of("Waldo", "Johnny")))));
 * </pre></blockquote>
 * <p>
 * <h2>Capturing Values</h2>
 * <p>
 * The matcher APIs documented above allow capturing values in the matching entry
 * (Resource, Output, Mapping, etc.). The following code captures a string from a
 * matching resource.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": ["Flob", "Cat"],
 * //         "Waldo": ["Qix", "Qux"],
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * Capture fredCapture = new Capture();
 * Capture waldoCapture = new Capture();
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", fredCapture,
 *         "Waldo", (Object[])List.of("Qix", waldoCapture)));
 * 
 * fredCapture.asArray(); // returns ["Flob", "Cat"]
 * waldoCapture.asString();
 * </pre></blockquote>
 * <p>
 * With captures, a nested pattern can also be specified, so that only targets
 * that match the nested pattern will be captured. This pattern can be literals or
 * further Matchers.
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar1": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": ["Flob", "Cat"],
 * //       }
 * //     }
 * //     "MyBar2": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": ["Qix", "Qux"],
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * Capture capture = new Capture(Match.arrayWith(List.of("Cat")));
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", capture));
 * 
 * capture.asArray();
 * </pre></blockquote>
 * <p>
 * When multiple resources match the given condition, each <code>Capture</code> defined in
 * the condition will capture all matching values. They can be paged through using
 * the <code>next()</code> API. The following example illustrates this -
 * <p>
 * <blockquote><pre>
 * // Given a template -
 * // {
 * //   "Resources": {
 * //     "MyBar": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": "Flob",
 * //       }
 * //     },
 * //     "MyBaz": {
 * //       "Type": "Foo::Bar",
 * //       "Properties": {
 * //         "Fred": "Quib",
 * //       }
 * //     }
 * //   }
 * // }
 * 
 * Capture fredCapture = new Capture();
 * template.hasResourceProperties("Foo::Bar", Map.of(
 *         "Fred", fredCapture));
 * 
 * fredCapture.asString(); // returns "Flob"
 * fredCapture.next(); // returns true
 * fredCapture.asString();
 * </pre></blockquote>
 * <p>
 * <h2>Asserting Annotations</h2>
 * <p>
 * In addition to template matching, we provide an API for annotation matching.
 * <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.Annotations.html">Annotations</a>
 * can be added via the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.Aspects.html">Aspects</a>
 * API. You can learn more about Aspects <a href="https://docs.aws.amazon.com/cdk/v2/guide/aspects.html">here</a>.
 * <p>
 * Say you have a <code>MyAspect</code> and a <code>MyStack</code> that uses <code>MyAspect</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.constructs.Construct;
 * import software.constructs.IConstruct;
 * 
 * public class MyAspect implements IAspect {
 *     public void visit(IConstruct node) {
 *         if (node instanceof CfnResource &amp;&amp; node.getCfnResourceType() == "Foo::Bar") {
 *             this.error(node, "we do not want a Foo::Bar resource");
 *         }
 *     }
 * 
 *     public void error(IConstruct node, String message) {
 *         Annotations.of(node).addError(message);
 *     }
 * }
 * 
 * public class MyStack extends Stack {
 *     public MyStack(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         Stack stack = new Stack();
 *         CfnResource.Builder.create(stack, "Foo")
 *                 .type("Foo::Bar")
 *                 .properties(Map.of(
 *                         "Fred", "Thud"))
 *                 .build();
 *         Aspects.of(stack).add(new MyAspect());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * We can then assert that the stack contains the expected Error:
 * <p>
 * <blockquote><pre>
 * // import { Annotations } from '&#64;aws-cdk/assertions';
 * 
 * Annotations.fromStack(stack).hasError("/Default/Foo", "we do not want a Foo::Bar resource");
 * </pre></blockquote>
 * <p>
 * Here are the available APIs for <code>Annotations</code>:
 * <p>
 * <ul>
 * <li><code>hasError()</code>, <code>hasNoError()</code>, and <code>findError()</code></li>
 * <li><code>hasWarning()</code>, <code>hasNoWarning()</code>, and <code>findWarning()</code></li>
 * <li><code>hasInfo()</code>, <code>hasNoInfo()</code>, and <code>findInfo()</code></li>
 * </ul>
 * <p>
 * The corresponding <code>findXxx()</code> API is complementary to the <code>hasXxx()</code> API, except instead
 * of asserting its presence, it returns the set of matching messages.
 * <p>
 * In addition, this suite of APIs is compatible with <code>Matchers</code> for more fine-grained control.
 * For example, the following assertion works as well:
 * <p>
 * <blockquote><pre>
 * Annotations.fromStack(stack).hasError("/Default/Foo", Match.stringLikeRegexp(".*Foo::Bar.*"));
 * </pre></blockquote>
 * <p>
 * <h2>Asserting Stack tags</h2>
 * <p>
 * Tags applied to a <code>Stack</code> are not part of the rendered template: instead, they
 * are included as properties in the Cloud Assembly Manifest. To test that stacks
 * are tagged as expected, simple assertions can be written.
 * <p>
 * Given the following setup:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.App;
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.assertions.Tags;
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "MyStack")
 *         .tags(Map.of(
 *                 "tag-name", "tag-value"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * It is possible to test against these values:
 * <p>
 * <blockquote><pre>
 * Tags tags = Tags.fromStack(stack);
 * 
 * // using a default 'objectLike' Matcher
 * tags.hasValues(Map.of(
 *         "tag-name", "tag-value"));
 * 
 * // ... with Matchers embedded
 * tags.hasValues(Map.of(
 *         "tag-name", Match.stringLikeRegexp("value")));
 * 
 * // or another object Matcher at the top level
 * tags.hasValues(Match.objectEquals(Map.of(
 *         "tag-name", Match.anyValue())));
 * </pre></blockquote>
 * <p>
 * When tags are not defined on the stack, it is represented as an empty object
 * rather than <code>undefined</code>. To make this more obvious, there is a <code>hasNone()</code>
 * method that can be used in place of <code>Match.exactly({})</code>. If <code>Match.absent()</code> is
 * passed, an error will result.
 * <p>
 * <blockquote><pre>
 * // no tags present
 * Tags.fromStack(stack).hasNone();
 * 
 * // don't use absent() at the top level, it won't work
 * expect(() =&gt; { Tags.fromStack(stack).hasValues(Match.absent()); }).toThrow(/will never match/i);
 * </pre></blockquote>
 */
package software.amazon.awscdk.assertions;
