/**
 * <h1>AWS::GlobalAccelerator Construct Library</h1>
 * <p>
 * <h2>Introduction</h2>
 * <p>
 * AWS Global Accelerator (AGA) is a service that improves the availability and
 * performance of your applications with local or global users.
 * <p>
 * It intercepts your user's network connection at an edge location close to
 * them, and routes it to one of potentially multiple, redundant backends across
 * the more reliable and less congested AWS global network.
 * <p>
 * AGA can be used to route traffic to Application Load Balancers, Network Load
 * Balancers, EC2 Instances and Elastic IP Addresses.
 * <p>
 * For more information, see the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/AWS_GlobalAccelerator.html">AWS Global
 * Accelerator Developer Guide</a>.
 * <p>
 * <h2>Example</h2>
 * <p>
 * Here's an example that sets up a Global Accelerator for two Application Load
 * Balancers in two different AWS Regions:
 * <p>
 * <blockquote><pre>
 * // Create an Accelerator
 * Accelerator accelerator = new Accelerator(this, "Accelerator");
 * 
 * // Create a Listener
 * Listener listener = accelerator.addListener("Listener", ListenerOptions.builder()
 *         .portRanges(List.of(PortRange.builder().fromPort(80).build(), PortRange.builder().fromPort(443).build()))
 *         .build());
 * 
 * // Import the Load Balancers
 * INetworkLoadBalancer nlb1 = NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(this, "NLB1", NetworkLoadBalancerAttributes.builder()
 *         .loadBalancerArn("arn:aws:elasticloadbalancing:us-west-2:111111111111:loadbalancer/app/my-load-balancer1/e16bef66805b")
 *         .build());
 * INetworkLoadBalancer nlb2 = NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(this, "NLB2", NetworkLoadBalancerAttributes.builder()
 *         .loadBalancerArn("arn:aws:elasticloadbalancing:ap-south-1:111111111111:loadbalancer/app/my-load-balancer2/5513dc2ea8a1")
 *         .build());
 * 
 * // Add one EndpointGroup for each Region we are targeting
 * listener.addEndpointGroup("Group1", EndpointGroupOptions.builder()
 *         .endpoints(List.of(new NetworkLoadBalancerEndpoint(nlb1)))
 *         .build());
 * listener.addEndpointGroup("Group2", EndpointGroupOptions.builder()
 *         // Imported load balancers automatically calculate their Region from the ARN.
 *         // If you are load balancing to other resources, you must also pass a `region`
 *         // parameter here.
 *         .endpoints(List.of(new NetworkLoadBalancerEndpoint(nlb2)))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Create an Accelerator with IP addresses and IP address type</h3>
 * <p>
 * <blockquote><pre>
 * Accelerator accelerator = Accelerator.Builder.create(this, "Accelerator")
 *         .ipAddresses(List.of("1.1.1.1", "2.2.2.2"))
 *         .ipAddressType(IpAddressType.IPV4)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Concepts</h2>
 * <p>
 * The <strong>Accelerator</strong> construct defines a Global Accelerator resource.
 * <p>
 * An Accelerator includes one or more <strong>Listeners</strong> that accepts inbound
 * connections on one or more ports.
 * <p>
 * Each Listener has one or more <strong>Endpoint Groups</strong>, representing multiple
 * geographically distributed copies of your application. There is one Endpoint
 * Group per Region, and user traffic is routed to the closest Region by default.
 * <p>
 * An Endpoint Group consists of one or more <strong>Endpoints</strong>, which is where the
 * user traffic coming in on the Listener is ultimately sent. The Endpoint port
 * used is the same as the traffic came in on at the Listener, unless overridden.
 * <p>
 * <h2>Types of Endpoints</h2>
 * <p>
 * There are 4 types of Endpoints, and they can be found in the
 * <code>aws-cdk-lib/aws-globalaccelerator-endpoints</code> package:
 * <p>
 * <ul>
 * <li>Application Load Balancers</li>
 * <li>Network Load Balancers</li>
 * <li>EC2 Instances</li>
 * <li>Elastic IP Addresses</li>
 * </ul>
 * <p>
 * <h3>Application Load Balancers</h3>
 * <p>
 * <blockquote><pre>
 * ApplicationLoadBalancer alb;
 * Listener listener;
 * 
 * 
 * listener.addEndpointGroup("Group", EndpointGroupOptions.builder()
 *         .endpoints(List.of(
 *             ApplicationLoadBalancerEndpoint.Builder.create(alb)
 *                     .weight(128)
 *                     .preserveClientIp(true)
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Network Load Balancers</h3>
 * <p>
 * <blockquote><pre>
 * NetworkLoadBalancer nlb;
 * Listener listener;
 * 
 * 
 * listener.addEndpointGroup("Group", EndpointGroupOptions.builder()
 *         .endpoints(List.of(
 *             NetworkLoadBalancerEndpoint.Builder.create(nlb)
 *                     .weight(128)
 *                     .preserveClientIp(true)
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>EC2 Instances</h3>
 * <p>
 * <blockquote><pre>
 * Listener listener;
 * Instance instance;
 * 
 * 
 * listener.addEndpointGroup("Group", EndpointGroupOptions.builder()
 *         .endpoints(List.of(
 *             InstanceEndpoint.Builder.create(instance)
 *                     .weight(128)
 *                     .preserveClientIp(true)
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Elastic IP Addresses</h3>
 * <p>
 * <blockquote><pre>
 * Listener listener;
 * CfnEIP eip;
 * 
 * 
 * listener.addEndpointGroup("Group", EndpointGroupOptions.builder()
 *         .endpoints(List.of(
 *             CfnEipEndpoint.Builder.create(eip)
 *                     .weight(128)
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Client IP Address Preservation and Security Groups</h2>
 * <p>
 * When using the <code>preserveClientIp</code> feature, AGA creates
 * <strong>Elastic Network Interfaces</strong> (ENIs) in your AWS account, that are
 * associated with a Security Group AGA creates for you. You can use the
 * security group created by AGA as a source group in other security groups
 * (such as those for EC2 instances or Elastic Load Balancers), if you want to
 * restrict incoming traffic to the AGA security group rules.
 * <p>
 * AGA creates a specific security group called <code>GlobalAccelerator</code> for each VPC
 * it has an ENI in (this behavior can not be changed). CloudFormation doesn't
 * support referencing the security group created by AGA, but this construct
 * library comes with a custom resource that enables you to reference the AGA
 * security group.
 * <p>
 * Call <code>endpointGroup.connectionsPeer()</code> to obtain a reference to the Security Group
 * which you can use in connection rules. You must pass a reference to the VPC in whose
 * context the security group will be looked up. Example:
 * <p>
 * <blockquote><pre>
 * Listener listener;
 * 
 * // Non-open ALB
 * ApplicationLoadBalancer alb;
 * 
 * // Remember that there is only one AGA security group per VPC.
 * Vpc vpc;
 * 
 * 
 * EndpointGroup endpointGroup = listener.addEndpointGroup("Group", EndpointGroupOptions.builder()
 *         .endpoints(List.of(
 *             ApplicationLoadBalancerEndpoint.Builder.create(alb)
 *                     .preserveClientIp(true)
 *                     .build()))
 *         .build());
 * IPeer agaSg = endpointGroup.connectionsPeer("GlobalAcceleratorSG", vpc);
 * 
 * // Allow connections from the AGA to the ALB
 * alb.connections.allowFrom(agaSg, Port.tcp(443));
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.globalaccelerator;
