/**
 * <h1>Event Targets for Amazon EventBridge</h1>
 * <p>
 * This library contains integration classes to send Amazon EventBridge to any
 * number of supported AWS Services. Instances of these classes should be passed
 * to the <code>rule.addTarget()</code> method.
 * <p>
 * Currently supported are:
 * <p>
 * <ul>
 * <li><a href="#start-a-codebuild-build">Start a CodeBuild build</a></li>
 * <li><a href="#start-a-codepipeline-pipeline">Start a CodePipeline pipeline</a></li>
 * <li>Run an ECS task</li>
 * <li><a href="#invoke-a-lambda-function">Invoke a Lambda function</a></li>
 * <li><a href="#invoke-an-api-gateway-rest-api">Invoke a API Gateway REST API</a></li>
 * <li>Publish a message to an SNS topic</li>
 * <li>Send a message to an SQS queue</li>
 * <li><a href="#start-a-stepfunctions-state-machine">Start a StepFunctions state machine</a></li>
 * <li><a href="#queue-a-batch-job">Queue a Batch job</a></li>
 * <li>Make an AWS API call</li>
 * <li>Put a record to a Kinesis stream</li>
 * <li><a href="#log-an-event-into-a-loggroup">Log an event into a LogGroup</a></li>
 * <li>Put a record to a Kinesis Data Firehose stream</li>
 * <li><a href="#put-an-event-on-an-eventbridge-bus">Put an event on an EventBridge bus</a></li>
 * <li><a href="#invoke-an-api-destination">Send an event to EventBridge API Destination</a></li>
 * </ul>
 * <p>
 * See the README of the <code>&#64;aws-cdk/aws-events</code> library for more information on
 * EventBridge.
 * <p>
 * <h2>Event retry policy and using dead-letter queues</h2>
 * <p>
 * The Codebuild, CodePipeline, Lambda, StepFunctions, LogGroup, SQSQueue, SNSTopic and ECSTask targets support attaching a <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue and setting retry policies</a>. See the <a href="#invoke-a-lambda-function">lambda example</a>.
 * Use <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html">escape hatches</a> for the other target types.
 * <p>
 * <h2>Invoke a Lambda function</h2>
 * <p>
 * Use the <code>LambdaFunction</code> target to invoke a lambda function.
 * <p>
 * The code snippet below creates an event rule with a Lambda function as a target
 * triggered for every events from <code>aws.ec2</code> source. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunc")
 *         .runtime(Runtime.NODEJS_14_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = handler.toString()"))
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "rule")
 *         .eventPattern(EventPattern.builder()
 *                 .source(List.of("aws.ec2"))
 *                 .build())
 *         .build();
 * 
 * Queue queue = new Queue(this, "Queue");
 * 
 * rule.addTarget(LambdaFunction.Builder.create(fn)
 *         .deadLetterQueue(queue) // Optional: add a dead letter queue
 *         .maxEventAge(Duration.hours(2)) // Optional: set the maxEventAge retry policy
 *         .retryAttempts(2)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Log an event into a LogGroup</h2>
 * <p>
 * Use the <code>LogGroup</code> target to log your events in a CloudWatch LogGroup.
 * <p>
 * For example, the following code snippet creates an event rule with a CloudWatch LogGroup as a target.
 * Every events sent from the <code>aws.ec2</code> source will be sent to the CloudWatch LogGroup.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * LogGroup logGroup = LogGroup.Builder.create(this, "MyLogGroup")
 *         .logGroupName("MyLogGroup")
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "rule")
 *         .eventPattern(EventPattern.builder()
 *                 .source(List.of("aws.ec2"))
 *                 .build())
 *         .build();
 * 
 * rule.addTarget(new CloudWatchLogGroup(logGroup));
 * </pre></blockquote>
 * <p>
 * A rule target input can also be specified to modify the event that is sent to the log group.
 * Unlike other event targets, CloudWatchLogs requires a specific input template format.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.logs.*;
 * LogGroup logGroup;
 * Rule rule;
 * 
 * 
 * rule.addTarget(CloudWatchLogGroup.Builder.create(logGroup)
 *         .logEvent(LogGroupTargetInput(Map.of(
 *                 "timestamp", EventField.from("$.time"),
 *                 "message", EventField.from("$.detail-type"))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you want to use static values to overwrite the <code>message</code> make sure that you provide a <code>string</code>
 * value.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.logs.*;
 * LogGroup logGroup;
 * Rule rule;
 * 
 * 
 * rule.addTarget(CloudWatchLogGroup.Builder.create(logGroup)
 *         .logEvent(LogGroupTargetInput(Map.of(
 *                 "message", JSON.stringify(Map.of(
 *                         "CustomField", "CustomValue")))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodeBuild build</h2>
 * <p>
 * Use the <code>CodeBuildProject</code> target to trigger a CodeBuild project.
 * <p>
 * The code snippet below creates a CodeCommit repository that triggers a CodeBuild project
 * on commit to the master branch. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codebuild.*;
 * import software.amazon.awscdk.services.codecommit.*;
 * 
 * 
 * Repository repo = Repository.Builder.create(this, "MyRepo")
 *         .repositoryName("aws-cdk-codebuild-events")
 *         .build();
 * 
 * Project project = Project.Builder.create(this, "MyProject")
 *         .source(Source.codeCommit(CodeCommitSourceProps.builder().repository(repo).build()))
 *         .build();
 * 
 * Queue deadLetterQueue = new Queue(this, "DeadLetterQueue");
 * 
 * // trigger a build when a commit is pushed to the repo
 * Rule onCommitRule = repo.onCommit("OnCommit", OnCommitOptions.builder()
 *         .target(CodeBuildProject.Builder.create(project)
 *                 .deadLetterQueue(deadLetterQueue)
 *                 .build())
 *         .branches(List.of("master"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodePipeline pipeline</h2>
 * <p>
 * Use the <code>CodePipeline</code> target to trigger a CodePipeline pipeline.
 * <p>
 * The code snippet below creates a CodePipeline pipeline that is triggered every hour
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codepipeline.*;
 * 
 * 
 * Pipeline pipeline = new Pipeline(this, "Pipeline");
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.expression("rate(1 hour)"))
 *         .build();
 * 
 * rule.addTarget(new CodePipeline(pipeline));
 * </pre></blockquote>
 * <p>
 * <h2>Start a StepFunctions state machine</h2>
 * <p>
 * Use the <code>SfnStateMachine</code> target to trigger a State Machine.
 * <p>
 * The code snippet below creates a Simple StateMachine that is triggered every minute with a
 * dummy object as input.
 * You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>
 * to the target.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.minutes(1)))
 *         .build();
 * 
 * Queue dlq = new Queue(this, "DeadLetterQueue");
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("events.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "SM")
 *         .definition(Wait.Builder.create(this, "Hello").time(WaitTime.duration(Duration.seconds(10))).build())
 *         .build();
 * 
 * rule.addTarget(SfnStateMachine.Builder.create(stateMachine)
 *         .input(RuleTargetInput.fromObject(Map.of("SomeParam", "SomeValue")))
 *         .deadLetterQueue(dlq)
 *         .role(role)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Queue a Batch job</h2>
 * <p>
 * Use the <code>BatchJob</code> target to queue a Batch job.
 * <p>
 * The code snippet below creates a Simple JobQueue that is triggered every hour with a
 * dummy object as input.
 * You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>
 * to the target.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.batch.*;
 * import software.amazon.awscdk.services.ecs.ContainerImage;
 * 
 * 
 * Object jobQueue = JobQueue.Builder.create(this, "MyQueue")
 *         .computeEnvironments(List.of(Map.of(
 *                 "computeEnvironment", ComputeEnvironment.Builder.create(this, "ComputeEnvironment")
 *                         .managed(false)
 *                         .build(),
 *                 "order", 1)))
 *         .build();
 * 
 * Object jobDefinition = JobDefinition.Builder.create(this, "MyJob")
 *         .container(Map.of(
 *                 "image", ContainerImage.fromRegistry("test-repo")))
 *         .build();
 * 
 * Queue queue = new Queue(this, "Queue");
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(BatchJob.Builder.create(jobQueue.getJobQueueArn(), jobQueue, jobDefinition.getJobDefinitionArn(), jobDefinition)
 *         .deadLetterQueue(queue)
 *         .event(RuleTargetInput.fromObject(Map.of("SomeParam", "SomeValue")))
 *         .retryAttempts(2)
 *         .maxEventAge(Duration.hours(2))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Invoke an API Gateway REST API</h2>
 * <p>
 * Use the <code>ApiGateway</code> target to trigger a REST API.
 * <p>
 * The code snippet below creates a Api Gateway REST API that is invoked every hour.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigateway.*;
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.minutes(1)))
 *         .build();
 * 
 * Function fn = Function.Builder.create(this, "MyFunc")
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_14_X)
 *         .code(Code.fromInline("exports.handler = e =&gt; {}"))
 *         .build();
 * 
 * LambdaRestApi restApi = LambdaRestApi.Builder.create(this, "MyRestAPI").handler(fn).build();
 * 
 * Queue dlq = new Queue(this, "DeadLetterQueue");
 * 
 * rule.addTarget(
 * ApiGateway.Builder.create(restApi)
 *         .path("/*&#47;test")
 *         .method("GET")
 *         .stage("prod")
 *         .pathParameterValues(List.of("path-value"))
 *         .headerParameters(Map.of(
 *                 "Header1", "header1"))
 *         .queryStringParameters(Map.of(
 *                 "QueryParam1", "query-param-1"))
 *         .deadLetterQueue(dlq)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Invoke an API Destination</h2>
 * <p>
 * Use the <code>targets.ApiDestination</code> target to trigger an external API. You need to
 * create an <code>events.Connection</code> and <code>events.ApiDestination</code> as well.
 * <p>
 * The code snippet below creates an external destination that is invoked every hour.
 * <p>
 * <blockquote><pre>
 * Connection connection = Connection.Builder.create(this, "Connection")
 *         .authorization(Authorization.apiKey("x-api-key", SecretValue.secretsManager("ApiSecretName")))
 *         .description("Connection with API Key x-api-key")
 *         .build();
 * 
 * ApiDestination destination = ApiDestination.Builder.create(this, "Destination")
 *         .connection(connection)
 *         .endpoint("https://example.com")
 *         .description("Calling example.com with API key x-api-key")
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.minutes(1)))
 *         .targets(List.of(new ApiDestination(destination)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put an event on an EventBridge bus</h2>
 * <p>
 * Use the <code>EventBus</code> target to route event to a different EventBus.
 * <p>
 * The code snippet below creates the scheduled event rule that route events to an imported event bus.
 * <p>
 * <blockquote><pre>
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .build();
 * 
 * rule.addTarget(new EventBus(EventBus.fromEventBusArn(this, "External", "arn:aws:events:eu-west-1:999999999999:event-bus/test-bus")));
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.events.targets;
