/**
 * <h1>AWS Systems Manager Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Using existing SSM Parameters in your CDK app</h2>
 * <p>
 * You can reference existing SSM Parameter Store values that you want to use in
 * your CDK app by using <code>ssm.StringParameter.fromStringParameterAttributes</code>:
 * <p>
 * <blockquote><pre>
 * // Retrieve the latest value of the non-secret parameter
 * // with name "/My/String/Parameter".
 * String stringValue = StringParameter.fromStringParameterAttributes(this, "MyValue", StringParameterAttributes.builder()
 *         .parameterName("/My/Public/Parameter")
 *         .build()).getStringValue();
 * String stringValueVersionFromToken = StringParameter.fromStringParameterAttributes(this, "MyValueVersionFromToken", StringParameterAttributes.builder()
 *         .parameterName("/My/Public/Parameter")
 *         // parameter version from token
 *         .version(parameterVersion)
 *         .build()).getStringValue();
 * 
 * // Retrieve a specific version of the secret (SecureString) parameter.
 * // 'version' is always required.
 * IStringParameter secretValue = StringParameter.fromSecureStringParameterAttributes(this, "MySecureValue", SecureStringParameterAttributes.builder()
 *         .parameterName("/My/Secret/Parameter")
 *         .version(5)
 *         .build());
 * IStringParameter secretValueVersionFromToken = StringParameter.fromSecureStringParameterAttributes(this, "MySecureValueVersionFromToken", SecureStringParameterAttributes.builder()
 *         .parameterName("/My/Secret/Parameter")
 *         // parameter version from token
 *         .version(parameterVersion)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Lookup existing parameters</h3>
 * <p>
 * You can also use an existing parameter by looking up the parameter from the AWS environment.
 * This method uses AWS API calls to lookup the value from SSM during synthesis.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * String stringValue = StringParameter.valueFromLookup(stack, "/My/Public/Parameter");
 * </pre></blockquote>
 * <p>
 * When using <code>valueFromLookup</code> an initial value of 'dummy-value-for-${parameterName}'
 * (<code>dummy-value-for-/My/Public/Parameter</code> in the above example)
 * is returned prior to the lookup being performed. This can lead to errors if you are using this
 * value in places that require a certain format. For example if you have stored the ARN for a SNS
 * topic in a SSM Parameter which you want to lookup and provide to <code>Topic.fromTopicArn()</code>
 * <p>
 * <blockquote><pre>
 * String arnLookup = StringParameter.valueFromLookup(this, "/my/topic/arn");
 * Topic.fromTopicArn(this, "Topic", arnLookup);
 * </pre></blockquote>
 * <p>
 * Initially <code>arnLookup</code> will be equal to <code>dummy-value-for-/my/topic/arn</code> which will cause
 * <code>Topic.fromTopicArn</code> to throw an error indicating that the value is not in <code>arn</code> format.
 * <p>
 * For these use cases you need to handle the <code>dummy-value</code> in your code. For example:
 * <p>
 * <blockquote><pre>
 * String arnLookup = StringParameter.valueFromLookup(this, "/my/topic/arn");
 * String arnLookupValue;
 * if (arnLookup.includes("dummy-value")) {
 *     arnLookupValue = this.formatArn(ArnComponents.builder()
 *             .service("sns")
 *             .resource("topic")
 *             .resourceName(arnLookup)
 *             .build());
 * } else {
 *     arnLookupValue = arnLookup;
 * }
 * 
 * Topic.fromTopicArn(this, "Topic", arnLookupValue);
 * </pre></blockquote>
 * <p>
 * Alternatively, if the property supports tokens you can convert the parameter value into a token
 * to be resolved <em>after</em> the lookup has been completed.
 * <p>
 * <blockquote><pre>
 * String arnLookup = StringParameter.valueFromLookup(this, "/my/role/arn");
 * Role.fromRoleArn(this, "role", Lazy.string(Map.of("produce", () =&gt; arnLookup)));
 * </pre></blockquote>
 * <p>
 * <h2>Creating new SSM Parameters in your CDK app</h2>
 * <p>
 * You can create either <code>ssm.StringParameter</code> or <code>ssm.StringListParameter</code>s in
 * a CDK app. These are public (not secret) values. Parameters of type
 * <em>SecureString</em> cannot be created directly from a CDK application; if you want
 * to provision secrets automatically, use Secrets Manager Secrets (see the
 * <code>&#64;aws-cdk/aws-secretsmanager</code> package).
 * <p>
 * <blockquote><pre>
 * StringParameter.Builder.create(this, "Parameter")
 *         .allowedPattern(".*")
 *         .description("The value Foo")
 *         .parameterName("FooParameter")
 *         .stringValue("Foo")
 *         .tier(ParameterTier.ADVANCED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Create a new SSM Parameter holding a String
 * StringParameter param = StringParameter.Builder.create(stack, "StringParameter")
 *         // description: 'Some user-friendly description',
 *         // name: 'ParameterName',
 *         .stringValue("Initial parameter value")
 *         .build();
 * 
 * // Grant read access to some Role
 * param.grantRead(role);
 * 
 * // Create a new SSM Parameter holding a StringList
 * StringListParameter listParameter = StringListParameter.Builder.create(stack, "StringListParameter")
 *         // description: 'Some user-friendly description',
 *         // name: 'ParameterName',
 *         .stringListValue(List.of("Initial parameter value A", "Initial parameter value B"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When specifying an <code>allowedPattern</code>, the values provided as string literals
 * are validated against the pattern and an exception is raised if a value
 * provided does not comply.
 */
package software.amazon.awscdk.services.ssm;
