/**
 * <h2>AWS Lambda Event Sources</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * An event source mapping is an AWS Lambda resource that reads from an event source and invokes a Lambda function.
 * You can use event source mappings to process items from a stream or queue in services that don't invoke Lambda
 * functions directly. Lambda provides event source mappings for the following services. Read more about lambda
 * event sources <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html">here</a>.
 * <p>
 * This module includes classes that allow using various AWS services as event
 * sources for AWS Lambda via the high-level <code>lambda.addEventSource(source)</code> API.
 * <p>
 * NOTE: In most cases, it is also possible to use the resource APIs to invoke an
 * AWS Lambda function. This library provides a uniform API for all Lambda event
 * sources regardless of the underlying mechanism they use.
 * <p>
 * The following code sets up a lambda function with an SQS queue event source -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var fn = Function.Builder.create(this, "MyFunction").build();
 * 
 * var queue = new Queue(this, "MyQueue");
 * var eventSource = lambda.addEventSource(new SqsEventSource(queue));
 * 
 * var eventSourceId = eventSource.getEventSourceId();
 * </pre></blockquote>
 * <p>
 * The <code>eventSourceId</code> property contains the event source id. This will be a
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/tokens.html">token</a> that will resolve to the final value at the time of
 * deployment.
 * <p>
 * <h3>SQS</h3>
 * <p>
 * Amazon Simple Queue Service (Amazon SQS) allows you to build asynchronous
 * workflows. For more information about Amazon SQS, see Amazon Simple Queue
 * Service. You can configure AWS Lambda to poll for these messages as they arrive
 * and then pass the event to a Lambda function invocation. To view a sample event,
 * see <a href="https://docs.aws.amazon.com/lambda/latest/dg/eventsources.html#eventsources-sqs">Amazon SQS Event</a>.
 * <p>
 * To set up Amazon Simple Queue Service as an event source for AWS Lambda, you
 * first create or update an Amazon SQS queue and select custom values for the
 * queue parameters. The following parameters will impact Amazon SQS's polling
 * behavior:
 * <p>
 * <ul>
 * <li><strong>visibilityTimeout</strong>: May impact the period between retries.</li>
 * <li><strong>receiveMessageWaitTime</strong>: Will determine <a href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-long-polling.html">long
 * poll</a>
 * duration. The default value is 20 seconds.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.sqs.*;
 * import software.amazon.awscdk.services.lambda.eventsources.SqsEventSource;
 * import software.amazon.awscdk.core.Duration;
 * 
 * Queue queue = new Queue(this, "MyQueue", new QueueProps()
 *         .visibilityTimeout(Duration.seconds(30))// default,
 *         .receiveMessageWaitTime(Duration.seconds(20)));
 * 
 * lambda.addEventSource(new SqsEventSource(queue, new SqsEventSourceProps()
 *         .batchSize(10)));
 * </pre></blockquote>
 * <p>
 * <h3>S3</h3>
 * <p>
 * You can write Lambda functions to process S3 bucket events, such as the
 * object-created or object-deleted events. For example, when a user uploads a
 * photo to a bucket, you might want Amazon S3 to invoke your Lambda function so
 * that it reads the image and creates a thumbnail for the photo.
 * <p>
 * You can use the bucket notification configuration feature in Amazon S3 to
 * configure the event source mapping, identifying the bucket events that you want
 * Amazon S3 to publish and which Lambda function to invoke.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.services.lambda.eventsources.S3EventSource;
 * 
 * Bucket bucket = new Bucket(...);
 * 
 * lambda.addEventSource(new S3EventSource(bucket, new S3EventSourceProps()
 *         .events(asList(s3.EventType.getOBJECT_CREATED(), s3.EventType.getOBJECT_REMOVED()))
 *         .filters(asList(new NotificationKeyFilter().prefix("subdir/")))));
 * </pre></blockquote>
 * <p>
 * <h3>SNS</h3>
 * <p>
 * You can write Lambda functions to process Amazon Simple Notification Service
 * notifications. When a message is published to an Amazon SNS topic, the service
 * can invoke your Lambda function by passing the message payload as a parameter.
 * Your Lambda function code can then process the event, for example publish the
 * message to other Amazon SNS topics, or send the message to other AWS services.
 * <p>
 * This also enables you to trigger a Lambda function in response to Amazon
 * CloudWatch alarms and other AWS services that use Amazon SNS.
 * <p>
 * For an example event, see <a href="https://docs.aws.amazon.com/sns/latest/dg/json-formats.html">Appendix: Message and JSON
 * Formats</a> and
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/eventsources.html#eventsources-sns">Amazon SNS Sample
 * Event</a>.
 * For an example use case, see <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-sns.html">Using AWS Lambda with Amazon SNS from Different
 * Accounts</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.sns.*;
 * import software.amazon.awscdk.services.lambda.eventsources.SnsEventSource;
 * 
 * Topic topic = new Topic(...);
 * 
 * lambda.addEventSource(new SnsEventSource(topic));
 * </pre></blockquote>
 * <p>
 * When a user calls the SNS Publish API on a topic that your Lambda function is
 * subscribed to, Amazon SNS will call Lambda to invoke your function
 * asynchronously. Lambda will then return a delivery status. If there was an error
 * calling Lambda, Amazon SNS will retry invoking the Lambda function up to three
 * times. After three tries, if Amazon SNS still could not successfully invoke the
 * Lambda function, then Amazon SNS will send a delivery status failure message to
 * CloudWatch.
 * <p>
 * <h3>DynamoDB Streams</h3>
 * <p>
 * You can write Lambda functions to process change events from a DynamoDB Table. An event is emitted to a DynamoDB stream (if configured) whenever a write (Put, Delete, Update)
 * operation is performed against the table. See <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html">Using AWS Lambda with Amazon DynamoDB</a> for more information about configuring Lambda function event sources with DynamoDB.
 * <p>
 * To process events with a Lambda function, first create or update a DynamoDB table and enable a <code>stream</code> specification. Then, create a <code>DynamoEventSource</code>
 * and add it to your Lambda function. The following parameters will impact Amazon DynamoDB's polling behavior:
 * <p>
 * <ul>
 * <li><strong>batchSize</strong>: Determines how many records are buffered before invoking your lambda function - could impact your function's memory usage (if too high) and ability to keep up with incoming data velocity (if too low).</li>
 * <li><strong>bisectBatchOnError</strong>: If a batch encounters an error, this will cause the batch to be split in two and have each new smaller batch retried, allowing the records in error to be isolated.</li>
 * <li><strong>maxBatchingWindow</strong>: The maximum amount of time to gather records before invoking the lambda. This increases the likelihood of a full batch at the cost of delayed processing.</li>
 * <li><strong>maxRecordAge</strong>: The maximum age of a record that will be sent to the function for processing. Records that exceed the max age will be treated as failures.</li>
 * <li><strong>onFailure</strong>: In the event a record fails after all retries or if the record age has exceeded the configured value, the record will be sent to SQS queue or SNS topic that is specified here</li>
 * <li><strong>parallelizationFactor</strong>: The number of batches to concurrently process on each shard.</li>
 * <li><strong>retryAttempts</strong>: The maximum number of times a record should be retried in the event of failure.</li>
 * <li><strong>startingPosition</strong>: Will determine where to being consumption, either at the most recent ('LATEST') record or the oldest record ('TRIM_HORIZON'). 'TRIM_HORIZON' will ensure you process all available data, while 'LATEST' will ignore all reocrds that arrived prior to attaching the event source.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.dynamodb.*;
 * import software.amazon.awscdk.services.lambda.*;
 * import software.amazon.awscdk.services.sqs.*;
 * import software.amazon.awscdk.services.lambda.eventsources.DynamoEventSource;
 * import software.amazon.awscdk.services.lambda.eventsources.SqsDlq;
 * 
 * Table table = Table.Builder.create(...)
 *         .partitionKey()...
 *         .stream(dynamodb.StreamViewType.getNEW_IMAGE())
 *         .build();
 * 
 * Queue deadLetterQueue = new Queue(this, "deadLetterQueue");public void () {
 * }new Function(...);
 * public void () {
 * }addEventSource(new DynamoEventSource(table, new DynamoEventSourceProps()
 *         .startingPosition(lambda.StartingPosition.getTRIM_HORIZON())
 *         .batchSize(5)
 *         .bisectBatchOnError(true)
 *         .onFailure(new SqsDlq(deadLetterQueue))
 *         .retryAttempts(10)));
 * </pre></blockquote>
 * <p>
 * <h3>Kinesis</h3>
 * <p>
 * You can write Lambda functions to process streaming data in Amazon Kinesis Streams. For more information about Amazon Kinesis, see <a href="https://aws.amazon.com/kinesis/data-streams/">Amazon Kinesis
 * Service</a>. To learn more about configuring Lambda function event sources with kinesis and view a sample event,
 * see <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html">Amazon Kinesis Event</a>.
 * <p>
 * To set up Amazon Kinesis as an event source for AWS Lambda, you
 * first create or update an Amazon Kinesis stream and select custom values for the
 * event source parameters. The following parameters will impact Amazon Kinesis's polling
 * behavior:
 * <p>
 * <ul>
 * <li><strong>batchSize</strong>: Determines how many records are buffered before invoking your lambda function - could impact your function's memory usage (if too high) and ability to keep up with incoming data velocity (if too low).</li>
 * <li><strong>bisectBatchOnError</strong>: If a batch encounters an error, this will cause the batch to be split in two and have each new smaller batch retried, allowing the records in error to be isolated.</li>
 * <li><strong>maxBatchingWindow</strong>: The maximum amount of time to gather records before invoking the lambda. This increases the likelihood of a full batch at the cost of possibly delaying processing.</li>
 * <li><strong>maxRecordAge</strong>: The maximum age of a record that will be sent to the function for processing. Records that exceed the max age will be treated as failures.</li>
 * <li><strong>onFailure</strong>: In the event a record fails and consumes all retries, the record will be sent to SQS queue or SNS topic that is specified here</li>
 * <li><strong>parallelizationFactor</strong>: The number of batches to concurrently process on each shard.</li>
 * <li><strong>retryAttempts</strong>: The maximum number of times a record should be retried in the event of failure.</li>
 * <li><strong>startingPosition</strong>: Will determine where to being consumption, either at the most recent ('LATEST') record or the oldest record ('TRIM_HORIZON'). 'TRIM_HORIZON' will ensure you process all available data, while 'LATEST' will ignore all reocrds that arrived prior to attaching the event source.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.lambda.*;
 * import software.amazon.awscdk.services.kinesis.*;
 * import software.amazon.awscdk.services.lambda.eventsources.KinesisEventSource;
 * 
 * Stream stream = new Stream(this, "MyStream");
 * 
 * myFunction.addEventSource(new KinesisEventSource(queue, new KinesisEventSourceProps()
 *         .batchSize(100)// default
 *         .startingPosition(lambda.StartingPosition.getTRIM_HORIZON())));
 * </pre></blockquote>
 * <p>
 * <h2>Roadmap</h2>
 * <p>
 * Eventually, this module will support all the event sources described under
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html">Supported Event
 * Sources</a>
 * in the AWS Lambda Developer Guide.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.lambda.eventsources;
