/*
 *   Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */

package software.amazon.cloudwatchlogs.emf.environment;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.Map;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import software.amazon.cloudwatchlogs.emf.Constants;
import software.amazon.cloudwatchlogs.emf.config.Configuration;
import software.amazon.cloudwatchlogs.emf.config.SystemWrapper;
import software.amazon.cloudwatchlogs.emf.model.MetricsContext;
import software.amazon.cloudwatchlogs.emf.util.StringUtils;

@Slf4j
class ECSEnvironment extends AgentBasedEnvironment {
    private Configuration config;
    private ECSMetadata metadata;
    private ResourceFetcher fetcher;
    private String fluentBitEndpoint;
    private String hostname;

    private static final String ECS_CONTAINER_METADATA_URI = "ECS_CONTAINER_METADATA_URI";
    private static final String FLUENT_HOST = "FLUENT_HOST";
    private static final String ENVIRONMENT_TYPE = "AWS::ECS::Container";

    ECSEnvironment(Configuration config, ResourceFetcher fetcher) {
        super(config);
        this.config = config;
        this.fetcher = fetcher;
    }

    @Override
    public boolean probe() {
        String uri = getEnv(ECS_CONTAINER_METADATA_URI);

        if (uri == null) {
            return false;
        }

        checkAndSetFluentHost();

        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(MapperFeature.ACCEPT_CASE_INSENSITIVE_PROPERTIES, true);
        URI parsedURI = null;
        try {
            parsedURI = new URI(uri);
            metadata = fetcher.fetch(parsedURI, objectMapper, ECSMetadata.class);
            formatImageName();
            return true;
        } catch (Exception ex) {
            log.debug("Failed to get response from: " + parsedURI, ex);
        }

        return false;
    }

    @Override
    public String getName() {
        if (config.getServiceName().isPresent()) {
            return config.getServiceName().get();
        }
        if (metadata != null && !StringUtils.isNullOrEmpty(metadata.formattedImageName)) {
            return metadata.formattedImageName;
        }
        return Constants.UNKNOWN;
    }

    @Override
    public String getType() {
        if (config.getServiceType().isPresent()) {
            return config.getServiceType().get();
        }
        return ENVIRONMENT_TYPE;
    }

    @Override
    public String getLogGroupName() {
        // FireLens / fluent-bit does not need the log group to be included
        // since configuration of the LogGroup is handled by the
        // fluent bit config file
        if (this.fluentBitEndpoint != null) {
            return "";
        }
        return super.getLogGroupName();
    }

    @Override
    public void configureContext(MetricsContext context) {

        context.putProperty("containerId", getHostName());
        context.putProperty("createdAt", metadata.createdAt);
        context.putProperty("startedAt", metadata.startedAt);
        context.putProperty("image", metadata.image);
        context.putProperty("cluster", metadata.labels.get("com.amazonaws.ecs.cluster"));
        context.putProperty("taskArn", metadata.labels.get("com.amazonaws.ecs.task-arn"));
    }

    private String getHostName() {
        if (hostname != null) {
            return hostname;
        }
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException ex) {
            log.debug("Unable to get hostname: ", ex);
        }
        return hostname;
    }

    private String getEnv(String name) {
        return SystemWrapper.getenv(name);
    }

    private void checkAndSetFluentHost() {
        String fluentHost = getEnv(FLUENT_HOST);
        if (fluentHost != null && !config.getAgentEndpoint().isPresent()) {
            fluentBitEndpoint =
                    String.format("tcp://%s:%d", fluentHost, Constants.DEFAULT_AGENT_PORT);
            config.setAgentEndpoint(fluentBitEndpoint);
            log.info("Using FluentBit configuration. Endpoint: {}", fluentBitEndpoint);
        }
    }

    private void formatImageName() {
        if (metadata != null && metadata.image != null) {
            String imageName = metadata.image;
            String[] splitImageNames = imageName.split("\\/");
            metadata.formattedImageName = splitImageNames[splitImageNames.length - 1];
        }
    }

    @Data
    @JsonIgnoreProperties(ignoreUnknown = true)
    static class ECSMetadata {
        String name;
        String dockerId;
        String dockerName;
        String image;
        String formattedImageName;
        String imageID;
        Map<String, String> labels;
        String createdAt;
        String startedAt;
    }
}
