/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.timestream.jdbc;

import com.amazonaws.services.timestreamquery.model.ColumnInfo;
import com.amazonaws.services.timestreamquery.model.Datum;
import com.amazonaws.services.timestreamquery.model.Row;
import com.amazonaws.services.timestreamquery.model.TimeSeriesDataPoint;
import com.amazonaws.services.timestreamquery.model.Type;
import com.google.common.collect.ImmutableMap;
import java.sql.Array;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Struct;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.timestream.jdbc.Conversions;
import software.amazon.timestream.jdbc.Error;
import software.amazon.timestream.jdbc.JdbcType;
import software.amazon.timestream.jdbc.TimestreamArray;
import software.amazon.timestream.jdbc.TimestreamStruct;

enum TimestreamDataType {
    BIGINT(JdbcType.BIGINT, Long.class, 19),
    INTEGER(JdbcType.INTEGER, Integer.class, 11),
    BOOLEAN(JdbcType.BOOLEAN, Boolean.class, 5),
    DOUBLE(JdbcType.DOUBLE, Double.class, 53),
    VARCHAR(JdbcType.VARCHAR, String.class, Integer.MAX_VALUE),
    DATE(JdbcType.DATE, Date.class, 10),
    TIME(JdbcType.TIME, Time.class, 18),
    TIMESTAMP(JdbcType.TIMESTAMP, Timestamp.class, 29),
    INTERVAL_YEAR_TO_MONTH(JdbcType.VARCHAR, String.class, 0),
    INTERVAL_DAY_TO_SECOND(JdbcType.VARCHAR, String.class, 0),
    ARRAY(JdbcType.ARRAY, Array.class, 0),
    ROW(JdbcType.STRUCT, Struct.class, 0),
    TIMESERIES(JdbcType.JAVA_OBJECT, List.class, 0),
    UNKNOWN(JdbcType.VARCHAR, String.class, 0);

    private static final Logger LOGGER;
    private static final Map<String, JdbcType> CLASS_NAME_TO_JDBC_TYPE;
    private static final Map<String, TimestreamDataType> CLASS_NAME_TO_TIMESTREAM_TYPE;
    private final JdbcType jdbcType;
    private final Class<?> javaClass;
    private final int precision;

    private TimestreamDataType(JdbcType jdbcType, Class<?> javaClass, int precision) {
        this.jdbcType = jdbcType;
        this.javaClass = javaClass;
        this.precision = precision;
    }

    JdbcType getJdbcType() {
        return this.jdbcType;
    }

    Class<?> getJavaClass() {
        return this.javaClass;
    }

    String getClassName() {
        return this.javaClass.getName();
    }

    int getPrecision() {
        return this.precision;
    }

    static ColumnInfo createColumnInfo(TimestreamDataType type, String name) {
        return new ColumnInfo().withName(name).withType(new Type().withScalarType(type.toString()));
    }

    static Type createTypeWithMap(Type type, Type originalType, Map<String, Class<?>> conversionMap) throws SQLException {
        String originalScalarType = originalType.getScalarType();
        if (originalScalarType != null) {
            TimestreamDataType targetType = TimestreamDataType.retrieveTargetType(TimestreamDataType.valueOf(originalScalarType), conversionMap);
            return type.withScalarType(targetType.name());
        }
        List originalRow = originalType.getRowColumnInfo();
        if (originalRow != null) {
            TimestreamDataType targetType = TimestreamDataType.retrieveTargetType(ROW, conversionMap);
            switch (targetType) {
                case VARCHAR: {
                    return type.withScalarType(targetType.name());
                }
                case TIMESERIES: {
                    throw Error.createSQLException(LOGGER, Error.UNSUPPORTED_CONVERSION, new Object[]{ROW.name(), targetType});
                }
            }
            ArrayList<ColumnInfo> targetRow = new ArrayList<ColumnInfo>();
            for (ColumnInfo columnInfo : originalRow) {
                targetRow.add(new ColumnInfo().withType(TimestreamDataType.createTypeWithMap(new Type(), columnInfo.getType(), conversionMap)));
            }
            return type.withRowColumnInfo(targetRow);
        }
        ColumnInfo originalTimeSeries = originalType.getTimeSeriesMeasureValueColumnInfo();
        if (originalTimeSeries != null) {
            return TimestreamDataType.createType(type, originalTimeSeries, TIMESERIES, conversionMap);
        }
        ColumnInfo originalArray = originalType.getArrayColumnInfo();
        if (originalArray != null) {
            return TimestreamDataType.createType(type, originalArray, ARRAY, conversionMap);
        }
        throw new RuntimeException(Error.getErrorMessage(LOGGER, Error.INVALID_TYPE, type));
    }

    private static Type createType(Type type, ColumnInfo originalColumn, TimestreamDataType sourceType, Map<String, Class<?>> conversionMap) throws SQLException {
        TimestreamDataType targetType = TimestreamDataType.retrieveTargetType(sourceType, conversionMap);
        Type newType = TimestreamDataType.createTypeWithMap(new Type(), originalColumn.getType(), conversionMap);
        switch (targetType) {
            case ARRAY: {
                return type.withArrayColumnInfo(new ColumnInfo().withType(newType));
            }
            case TIMESERIES: {
                return type.withTimeSeriesMeasureValueColumnInfo(new ColumnInfo().withType(newType));
            }
        }
        return type.withScalarType(targetType.name());
    }

    static TimestreamDataType retrieveTargetType(TimestreamDataType sourceType, Map<String, Class<?>> conversionMap) throws SQLException {
        Class<?> targetClass = conversionMap.get(sourceType.name());
        TimestreamDataType targetType = targetClass != null ? TimestreamDataType.convertClassNameToTimestreamType(targetClass.getName()) : sourceType;
        Conversions.retrieveAndValidateConversion(sourceType, targetType.getJdbcType());
        return targetType;
    }

    static ColumnInfo createColumnInfo(ColumnInfo columnInfo, Type type, String name) {
        if (type.getScalarType() != null) {
            return columnInfo.withName(type.getScalarType()).withType(new Type().withScalarType(type.getScalarType()));
        }
        ColumnInfo arrayColumnInfo = type.getArrayColumnInfo();
        if (arrayColumnInfo != null) {
            return columnInfo.withName(name).withType(new Type().withArrayColumnInfo(TimestreamDataType.createColumnInfo(new ColumnInfo(), arrayColumnInfo.getType(), ARRAY.name())));
        }
        List rowColumnInfo = type.getRowColumnInfo();
        if (rowColumnInfo != null) {
            return columnInfo.withName(name).withType(new Type().withRowColumnInfo((Collection)rowColumnInfo.parallelStream().map(rowCol -> TimestreamDataType.createColumnInfo(new ColumnInfo(), rowCol.getType(), ROW.name())).collect(Collectors.toList())));
        }
        ColumnInfo timeSeriesColumnInfo = type.getTimeSeriesMeasureValueColumnInfo();
        if (timeSeriesColumnInfo != null) {
            return columnInfo.withName(name).withType(new Type().withTimeSeriesMeasureValueColumnInfo(TimestreamDataType.createColumnInfo(new ColumnInfo(), timeSeriesColumnInfo.getType(), TIMESERIES.name())));
        }
        throw new RuntimeException(Error.getErrorMessage(LOGGER, Error.INVALID_TYPE, type));
    }

    static Datum createDatum(Datum datum, Object value, Type valueType) throws SQLException {
        if (valueType.getScalarType() != null) {
            if (value instanceof Datum) {
                return (Datum)value;
            }
            return datum.withScalarValue(value.toString());
        }
        if (valueType.getArrayColumnInfo() != null) {
            value = ((TimestreamArray)((Object)value)).getArrayList();
            ArrayList<Datum> arrayValueList = new ArrayList<Datum>();
            for (Object val : value) {
                arrayValueList.add(TimestreamDataType.createDatum(new Datum(), val, valueType.getArrayColumnInfo().getType()));
            }
            return datum.withArrayValue(arrayValueList);
        }
        if (valueType.getRowColumnInfo() != null) {
            ArrayList<Datum> newRowData = new ArrayList<Datum>();
            List<Datum> rowData = ((TimestreamStruct)((Object)value)).getStruct();
            List columnInfoList = valueType.getRowColumnInfo();
            for (int i = 0; i < rowData.size(); ++i) {
                Datum val = rowData.get(i);
                newRowData.add(TimestreamDataType.createDatum(new Datum(), val, ((ColumnInfo)columnInfoList.get(i)).getType()));
            }
            return datum.withRowValue(new Row().withData(newRowData));
        }
        if (valueType.getTimeSeriesMeasureValueColumnInfo() != null) {
            if (value instanceof TimeSeriesDataPoint) {
                LOGGER.warn("The TimeSeries value {} is not wrapped in an array list.", value);
                return new Datum().withTimeSeriesValue(new TimeSeriesDataPoint[]{(TimeSeriesDataPoint)value});
            }
            return new Datum().withTimeSeriesValue((Collection)value);
        }
        throw new RuntimeException(Error.getErrorMessage(LOGGER, Error.INVALID_TYPE, valueType));
    }

    static TimestreamDataType fromType(Type type) {
        if (type.getArrayColumnInfo() != null) {
            return ARRAY;
        }
        if (type.getRowColumnInfo() != null) {
            return ROW;
        }
        if (type.getTimeSeriesMeasureValueColumnInfo() != null) {
            return TIMESERIES;
        }
        try {
            return TimestreamDataType.valueOf(type.getScalarType().toUpperCase());
        }
        catch (IllegalArgumentException e) {
            LOGGER.info("Unknown Timestream data type {}", (Object)type, (Object)e);
            return UNKNOWN;
        }
    }

    static JdbcType getJdbcTypeCode(Type type) {
        return TimestreamDataType.fromType((Type)type).jdbcType;
    }

    static String getJavaClassName(Type type) {
        return TimestreamDataType.fromType(type).getClassName();
    }

    static JdbcType convertClassNameToJdbcType(String className) throws SQLException {
        JdbcType type = CLASS_NAME_TO_JDBC_TYPE.get(className);
        if (type == null) {
            throw Error.createSQLException(LOGGER, Error.UNSUPPORTED_CLASS, className);
        }
        return type;
    }

    static TimestreamDataType convertClassNameToTimestreamType(String className) throws SQLException {
        TimestreamDataType type = CLASS_NAME_TO_TIMESTREAM_TYPE.get(className);
        if (type == null) {
            throw Error.createSQLException(LOGGER, Error.UNSUPPORTED_CLASS, className);
        }
        return type;
    }

    static {
        LOGGER = LoggerFactory.getLogger(TimestreamDataType.class);
        CLASS_NAME_TO_JDBC_TYPE = new ImmutableMap.Builder().put((Object)Long.class.getName(), (Object)JdbcType.BIGINT).put((Object)Integer.class.getName(), (Object)JdbcType.INTEGER).put((Object)Boolean.class.getName(), (Object)JdbcType.BOOLEAN).put((Object)Double.class.getName(), (Object)JdbcType.DOUBLE).put((Object)String.class.getName(), (Object)JdbcType.VARCHAR).put((Object)Date.class.getName(), (Object)JdbcType.DATE).put((Object)Time.class.getName(), (Object)JdbcType.TIME).put((Object)Timestamp.class.getName(), (Object)JdbcType.TIMESTAMP).put((Object)Array.class.getName(), (Object)JdbcType.ARRAY).put((Object)Struct.class.getName(), (Object)JdbcType.STRUCT).put((Object)List.class.getName(), (Object)JdbcType.JAVA_OBJECT).build();
        CLASS_NAME_TO_TIMESTREAM_TYPE = new ImmutableMap.Builder().put((Object)Long.class.getName(), (Object)BIGINT).put((Object)Integer.class.getName(), (Object)INTEGER).put((Object)Boolean.class.getName(), (Object)BOOLEAN).put((Object)Double.class.getName(), (Object)DOUBLE).put((Object)String.class.getName(), (Object)VARCHAR).put((Object)Date.class.getName(), (Object)DATE).put((Object)Time.class.getName(), (Object)TIME).put((Object)Timestamp.class.getName(), (Object)TIMESTAMP).put((Object)Array.class.getName(), (Object)ARRAY).put((Object)Struct.class.getName(), (Object)ROW).put((Object)List.class.getName(), (Object)TIMESERIES).build();
    }
}

