/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.timestream.jdbc;

import com.amazonaws.services.timestreamquery.AmazonTimestreamQuery;
import com.amazonaws.services.timestreamquery.model.QueryRequest;
import com.amazonaws.services.timestreamquery.model.QueryResult;
import com.google.common.annotations.VisibleForTesting;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.timestream.jdbc.Error;
import software.amazon.timestream.jdbc.TimestreamBaseResultSet;
import software.amazon.timestream.jdbc.TimestreamStatement;

public class TimestreamResultSet
extends TimestreamBaseResultSet {
    private static final Logger LOGGER = LoggerFactory.getLogger(TimestreamResultSet.class);
    private QueryResult result;
    private final long largeMaxRows;
    private int totalRows;
    private final TimestreamResultRetriever resultRetriever;
    private final ExecutorService executorService = Executors.newSingleThreadExecutor();
    @VisibleForTesting
    static final QueryResult TERMINATION_MARKER = new QueryResult();

    TimestreamResultSet(TimestreamStatement statement, String query, QueryResult result) throws SQLException {
        this(statement, query, result, new HashMap(), 0L, 0);
    }

    TimestreamResultSet(TimestreamStatement statement, String query, QueryResult result, Map<String, Class<?>> map, long largeMaxRows, int maxFieldSize) throws SQLException {
        this(statement, query, result, map, largeMaxRows, maxFieldSize, 0L, 0);
    }

    TimestreamResultSet(TimestreamStatement statement, String query, QueryResult result, Map<String, Class<?>> map, long largeMaxRows, int maxFieldSize, long executionTimeForFirstResultSet, int numPages) throws SQLException {
        super(statement, statement.getFetchSize(), map, maxFieldSize);
        this.result = result;
        List rows = result.getRows();
        this.rowItr = rows == null ? Collections.emptyIterator() : rows.iterator();
        this.rsMeta = this.createColumnMetadata(result.getColumnInfo());
        this.largeMaxRows = largeMaxRows;
        String token = result.getNextToken();
        if (token == null) {
            this.resultRetriever = new TimestreamNoOpResultRetriever();
        } else {
            this.resultRetriever = new TimestreamResultRetriever(this, this.getStatement().getClient(), this.getFetchSize(), query, token, executionTimeForFirstResultSet, numPages);
            this.executorService.execute(this.resultRetriever);
        }
    }

    @Override
    public boolean isAfterLast() throws SQLException {
        this.verifyOpen();
        return null == this.result;
    }

    @Override
    public boolean isLast() throws SQLException {
        this.verifyOpen();
        return null != this.result && null == this.result.getNextToken() && !this.rowItr.hasNext();
    }

    @Override
    protected void doClose() throws SQLException {
        try {
            this.resultRetriever.interrupt();
            this.executorService.shutdown();
            this.executorService.awaitTermination(10L, TimeUnit.SECONDS);
            this.resultRetriever.addTerminationMarker();
        }
        catch (InterruptedException e) {
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
            throw Error.createSQLException(LOGGER, e, Error.FAILED_TO_SHUTDOWN_RETRIEVAL_EXECUTOR_SERVICE, new Object[0]);
        }
        finally {
            this.getStatement().childClose();
        }
    }

    @Override
    protected boolean doNextPage() throws SQLException {
        if (this.largeMaxRows != 0L && (long)this.totalRows >= this.largeMaxRows || this.result == null || this.result.getNextToken() == null) {
            this.result = null;
            LOGGER.debug("Reached max rows limit or no more result sets.");
            return false;
        }
        TimestreamResultHolder resultHolder = this.resultRetriever.getResult();
        this.result = resultHolder.queryResult;
        if (this.result == TERMINATION_MARKER) {
            LOGGER.debug("Retrieved a termination marker.");
            return false;
        }
        List rows = this.result.getRows();
        int rowSize = rows.size();
        LOGGER.info("QueryID: {}\nNumber of rows: {}", (Object)this.result.getQueryId(), (Object)rowSize);
        LOGGER.debug("Execution time to retrieve the next page: {}ms", (Object)resultHolder.executionTime);
        if (this.largeMaxRows != 0L) {
            this.totalRows += rowSize;
            long overflow = this.largeMaxRows - (long)this.totalRows;
            if (overflow < 0L) {
                LOGGER.debug("Total number of rows retrieved has exceeded max rows limit of {}, truncating the extra {} rows.", (Object)this.largeMaxRows, (Object)Math.abs(overflow));
                rows = rows.subList(0, rowSize - (int)Math.abs(overflow));
            }
        }
        this.rowItr = rows.iterator();
        return true;
    }

    synchronized int getBufferSize() {
        return this.resultRetriever.getBufferSize();
    }

    boolean isTerminated() {
        return this.executorService.isTerminated();
    }

    private static class TimestreamNoOpResultRetriever
    extends TimestreamResultRetriever {
        TimestreamNoOpResultRetriever() {
            super(null, null, 0, null, null, 0L, 0);
        }

        @Override
        public void run() {
        }

        @Override
        synchronized int getBufferSize() {
            return 0;
        }

        @Override
        synchronized void addTerminationMarker() {
        }

        @Override
        TimestreamResultHolder getResult() {
            return new TimestreamResultHolder(TERMINATION_MARKER, -1L, null);
        }

        @Override
        void interrupt() {
        }
    }

    private static class TimestreamResultRetriever
    implements Runnable {
        private static final Logger LOGGER = LoggerFactory.getLogger(TimestreamResultRetriever.class);
        private final long executionTimeForFirstResultSet;
        private final AtomicInteger numRequests = new AtomicInteger();
        private final AtomicLong totalReadingTimeMilli = new AtomicLong();
        private final BlockingQueue<TimestreamResultHolder> resultSets = new LinkedBlockingDeque<TimestreamResultHolder>(2);
        private final TimestreamResultSet resultSet;
        private final AmazonTimestreamQuery client;
        private final int fetchSize;
        private final String query;
        private String nextToken;
        private volatile boolean isInterrupted;

        TimestreamResultRetriever(TimestreamResultSet resultSet, AmazonTimestreamQuery client, int fetchSize, String query, String nextToken, long executionTimeForFirstResultSet, int numPages) {
            this.executionTimeForFirstResultSet = executionTimeForFirstResultSet;
            this.resultSet = resultSet;
            this.client = client;
            this.fetchSize = fetchSize;
            this.query = query;
            this.nextToken = nextToken;
            this.numRequests.addAndGet(numPages);
        }

        @Override
        public void run() {
            QueryRequest request = new QueryRequest().withQueryString(this.query);
            if (this.fetchSize != 0) {
                request.withMaxRows(Integer.valueOf(this.fetchSize));
            }
            while (!this.isInterrupted && this.nextToken != null) {
                try {
                    long startExecutionTime = System.nanoTime();
                    QueryResult result = this.client.query(request.withNextToken(this.nextToken));
                    long executionTimeMilli = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startExecutionTime);
                    String queryId = result.getQueryId();
                    this.numRequests.incrementAndGet();
                    this.totalReadingTimeMilli.addAndGet(executionTimeMilli);
                    this.nextToken = result.getNextToken();
                    while (!this.resultSets.offer(new TimestreamResultHolder(result, executionTimeMilli, null), 50L, TimeUnit.MILLISECONDS)) {
                        if (!this.resultSet.isClosed()) continue;
                        LOGGER.info("Result set is closed while trying to add more result sets to the buffer.\nQuery ID: {}\nTime to read results: {}ms\nTotal execution time: {}ms\nTotal number of pages: {}", new Object[]{queryId, this.totalReadingTimeMilli.get(), this.totalReadingTimeMilli.get() + this.executionTimeForFirstResultSet, this.numRequests});
                        return;
                    }
                }
                catch (Exception e) {
                    this.resultSets.clear();
                    this.nextToken = null;
                    if (this.resultSets.offer(new TimestreamResultHolder(null, -1L, Error.createSQLException(LOGGER, e, Error.ASYNC_RETRIEVAL_ERROR, this.query)))) continue;
                    throw new RuntimeException(Error.getErrorMessage(LOGGER, Error.FAILED_TO_PROPAGATE_ERROR, new Object[0]));
                }
            }
            this.resultSet.getStatement().setResultNoMoreRows();
        }

        synchronized int getBufferSize() {
            return this.resultSets.size();
        }

        synchronized void addTerminationMarker() throws InterruptedException {
            this.resultSets.clear();
            LOGGER.info("Terminating background thread retrieving more result sets. \nTime to read results: {}ms\nTotal execution time: {}ms\nTotal number of pages: {}", new Object[]{this.totalReadingTimeMilli, this.totalReadingTimeMilli.get() + this.executionTimeForFirstResultSet, this.numRequests});
            if (!this.resultSets.offer(new TimestreamResultHolder(TERMINATION_MARKER, -1L, null), 50L, TimeUnit.MILLISECONDS)) {
                throw new RuntimeException(Error.getErrorMessage(LOGGER, Error.FAILED_TO_NOTIFY_CONSUMER_THREAD, new Object[0]));
            }
        }

        TimestreamResultHolder getResult() throws SQLException {
            try {
                TimestreamResultHolder result = this.resultSets.take();
                if (result.exception != null) {
                    throw result.exception;
                }
                return result;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw Error.createSQLException(LOGGER, e, Error.FAILED_TO_BUFFER_RESULT_SET, new Object[0]);
            }
        }

        void interrupt() {
            this.isInterrupted = true;
        }
    }

    private static class TimestreamResultHolder {
        final QueryResult queryResult;
        final long executionTime;
        final SQLException exception;

        TimestreamResultHolder(QueryResult queryResult, long executionTime, SQLException exception) {
            this.queryResult = queryResult;
            this.executionTime = executionTime;
            this.exception = exception;
        }
    }
}

