package com.etsy.sbt.checkstyle

import sbt.File

import java.io.{FileInputStream, InputStream}
import java.net.URLClassLoader

/**
 * Represents a Checkstyle XML configuration located locally, on the class path or remotely at a URL
 *
 * @author
 *   Joseph Earl
 */
sealed abstract class CheckstyleConfigLocation {
  def load(resources: Seq[File]): InputStream
}

object CheckstyleConfigLocation {
  final case class URL(url: String) extends CheckstyleConfigLocation {
    override def load(resources: Seq[sbt.File]): InputStream = {
      new java.net.URL(url).openStream()
    }
  }

  final case class File(path: String) extends CheckstyleConfigLocation {
    override def load(resources: Seq[sbt.File]): InputStream = {
      new FileInputStream(path)
    }
  }

  final case class Classpath(name: String) extends CheckstyleConfigLocation {
    override def load(resources: Seq[sbt.File]): InputStream = {
      val classpath = resources.map(f => f.toURI.toURL)
      val loader = new URLClassLoader(classpath.toArray, getClass.getClassLoader)
      loader.getResourceAsStream(name)
    }
  }
}
