/*
 * Decompiled with CFR 0.152.
 */
package com.actelion.research.chem;

import com.actelion.research.chem.DepictorTransformation;
import com.actelion.research.chem.StereoMolecule;
import com.actelion.research.gui.editor.AtomQueryFeatureDialogBuilder;
import com.actelion.research.gui.generic.GenericPoint;
import com.actelion.research.gui.generic.GenericPolygon;
import com.actelion.research.gui.generic.GenericRectangle;
import com.actelion.research.util.ColorHelper;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;

public abstract class AbstractDepictor<T> {
    private static final int[] ATOM_LABEL_COLOR = new int[]{0, 0xFFFFFF, 0xD9FFFF, 13402367, 12779264, 0xFFB5B5, 0x909090, 3166456, 0xFF0D0D, 9494608, 11789301, 11230450, 9109248, 12560038, 15780000, 0xFF8000, 0xFFFF30, 0x1FF01F, 8442339, 9388244, 4062976, 0xE6E6E6, 12567239, 0xA6A6AB, 9083335, 10255047, 14706227, 15765664, 0x50D050, 13140019, 8224944, 12750735, 0x668F8F, 12419299, 16752896, 10889513, 6076625, 7351984, 65280, 0x94FFFF, 9756896, 7586505, 0x54B5B5, 3907230, 2396047, 687500, 27013, 0xC0C0C0, 16767375, 10909043, 0x668080, 10380213, 13924864, 0x940094, 4366000, 5707663, 51456, 7394559, 0xFFFFC7, 14286791, 0xC7FFC7, 10747847, 9437127, 6422471, 4587463, 3211207, 2097095, 65436, 58997, 54354, 48952, 43812, 5096191, 5089023, 2200790, 2522539, 0x266696, 1528967, 0xD0D0E0, 16765219, 12105936, 10900557, 5724513, 10375093, 11230208, 7688005, 4358806, 4325478, 32000, 7384058, 47871, 41471, 36863, 33023, 27647, 5528818, 7888099, 9064419, 10565332, 11739092, 11739066, 11734438, 12389767, 13041766, 13369433, 13697103, 14221381, 14680120, 15073326, 15400998, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xC8C8C8, 1334015, 56540, 15075850, 0xE6E600, 56540, 15075850, 0xEBEBEB, 0x8282D2, 1016335, 1016335, 1334015, 0xE6E600, 0x3232AA, 14456450, 16422400, 16422400, 11819700, 0x3232AA, 1016335};
    private static final int BOND_FG_HILITE_COLOR = Short.MIN_VALUE;
    private static final int BOND_BG_HILITE_COLOR = -10706689;
    private static final int FG_EXCLUDE_GROUP_COLOR = -6291392;
    private static final int BG_EXCLUDE_GROUP_COLOR = -24321;
    private static final int COLOR_SELECTED = 128;
    private static final int COLOR_CIP_LETTER = 448;
    private static final int COLOR_CHIRALITY_TEXT = 448;
    private static final int COLOR_UNDEFINED = -1;
    private static final int COLOR_HILITE_BOND_BG = -2;
    private static final int COLOR_HILITE_BOND_FG = -3;
    private static final int COLOR_OVERRULED = -4;
    private static final int COLOR_RGB = -5;
    private static final int COLOR_CUSTOM_FOREGROUND = -6;
    private static final int COLOR_EXCLUDE_GROUP_BG = -7;
    private static final int COLOR_EXCLUDE_GROUP_FG = -8;
    private static final int COLOR_RESTORE_PREVIOUS = -9;
    private static final int COLOR_INITIALIZE = -10;
    public static final int COLOR_BLUE = -14655233;
    public static final int COLOR_RED = -65536;
    public static final int COLOR_GREEN = -16711936;
    public static final int COLOR_MAGENTA = -4194049;
    public static final int COLOR_ORANGE = -24576;
    public static final int COLOR_DARK_GREEN = -16744448;
    public static final int COLOR_DARK_RED = -6291456;
    public static final int cOptAvBondLen = 24;
    public static final int cColorGray = 1;
    protected static final int cModeMaxBondLength = 65535;
    protected static final int cModeInflateToAVBL = 196608;
    private static final int cModeChiralTextLocation = 786432;
    public static final int cModeInflateToMaxAVBL = 65536;
    public static final int cModeInflateToHighResAVBL = 131072;
    public static final int cModeChiralTextBelowMolecule = 0;
    public static final int cModeChiralTextAboveMolecule = 262144;
    public static final int cModeChiralTextOnFrameTop = 524288;
    public static final int cModeChiralTextOnFrameBottom = 786432;
    public static final int cDModeNoTabus = 1;
    public static final int cDModeAtomNo = 2;
    public static final int cDModeBondNo = 4;
    public static final int cDModeHiliteAllQueryFeatures = 8;
    public static final int cDModeShowMapping = 16;
    public static final int cDModeSuppressChiralText = 32;
    public static final int cDModeSuppressCIPParity = 64;
    public static final int cDModeSuppressESR = 128;
    private static final int cDModeShowSymmetryAny = 1792;
    public static final int cDModeShowSymmetrySimple = 256;
    public static final int cDModeShowSymmetryDiastereotopic = 512;
    public static final int cDModeShowSymmetryEnantiotopic = 1024;
    public static final int cDModeNoImplicitAtomLabelColors = 2048;
    public static final int cDModeNoStereoProblem = 4096;
    public static final int cDModeNoColorOnESRAndCIP = 8192;
    public static final int cDModeNoImplicitHydrogen = 16384;
    private static final double cFactorTextSize = 0.6;
    private static final double cFactorChiralTextSize = 0.5;
    private static final double cFactorBondSpacing = 0.15;
    private static final double cFactorBondHiliteRadius = 0.38;
    private static final double cFactorExcludeGroupRadius = 0.47;
    private static final double cFactorDotDiameter = 0.12;
    private static final double cFactorQFDiameter = 0.4;
    private static final double cFactorLineWidth = 0.06;
    private boolean[] mAtomIsConnected;
    private boolean[] mAtomLabelDisplayed;
    private double mpBondSpacing;
    private double mpDotDiameter;
    private double mpLineWidth;
    private double mpQFDiameter;
    private double mpBondHiliteRadius;
    private double mFactorTextSize;
    private double mpExcludeGroupRadius;
    private double mChiralTextSize;
    private int mpLabelSize;
    private int mStandardForegroundColor;
    private int mDisplayMode;
    private int mCurrentColor;
    private int mPreviousColor;
    private boolean mIsValidatingView;
    private ArrayList<GenericRectangle> mpTabuZone;
    private ArrayList<DepictorDot> mpDot;
    private StereoMolecule mMol;
    private GenericRectangle mBoundingRect = new GenericRectangle();
    private DepictorTransformation mTransformation;
    private GenericPoint mChiralTextLocation;
    private int[] mAtomColor;
    private int[] mAtomHiliteColor;
    private float[] mAtomHiliteRadius;
    private String[] mAtomText;
    private GenericPoint[] mAlternativeCoords;
    private int mOverruleForeground;
    private int mOverruleBackground;
    private int mBondBGHiliteColor;
    private int mBondFGHiliteColor;
    private int mExcludeGroupFGColor;
    private int mExcludeGroupBGColor;
    private int mCustomForeground;
    private int mCustomBackground;
    private int mRGBColor;
    protected T mContext;

    public AbstractDepictor(StereoMolecule mol) {
        this(mol, 0);
    }

    public AbstractDepictor(StereoMolecule mol, int displayMode) {
        this.mMol = mol;
        this.mDisplayMode = displayMode;
        this.init();
    }

    public void setDisplayMode(int displayMode) {
        this.mDisplayMode = displayMode;
    }

    public void setAtomText(String[] atomText) {
        this.mAtomText = atomText;
    }

    @Deprecated
    public void setForegroundColor(Color foreground, Color background) {
        this.setForegroundColor(foreground == null ? 0 : foreground.getRGB(), background == null ? 0 : background.getRGB());
    }

    public void setForegroundColor(int foreground, int background) {
        this.mStandardForegroundColor = -6;
        this.mCustomForeground = foreground;
        this.mCustomBackground = background;
        this.updateBondHiliteColor();
    }

    @Deprecated
    public void setOverruleColor(Color foreground, Color background) {
        this.setOverruleColor(foreground == null ? 0 : foreground.getRGB(), background == null ? 0 : background.getRGB());
    }

    public void setOverruleColor(int foreground, int background) {
        this.mOverruleForeground = foreground;
        this.mOverruleBackground = background;
        this.updateBondHiliteColor();
    }

    public void setAtomHighlightColors(int[] argb, float[] radius) {
        this.mAtomHiliteColor = argb;
        this.mAtomHiliteRadius = radius;
    }

    public void setTransformation(DepictorTransformation t) {
        this.mTransformation = t;
    }

    public void setFactorTextSize(double factor) {
        this.mFactorTextSize = factor;
    }

    public DepictorTransformation getTransformation() {
        return this.mTransformation;
    }

    public void applyTransformation(DepictorTransformation t) {
        t.applyTo(this.mTransformation);
        t.applyTo(this.mBoundingRect);
        t.applyTo(this.mChiralTextLocation);
    }

    public DepictorTransformation updateCoords(T context, GenericRectangle viewRect, int mode) {
        this.validateView(context, viewRect, mode);
        if (this.mTransformation.isVoidTransformation()) {
            return null;
        }
        DepictorTransformation t = this.mTransformation;
        this.mTransformation.applyTo(this.mMol);
        this.mTransformation = new DepictorTransformation();
        return t;
    }

    public DepictorTransformation simpleUpdateCoords(GenericRectangle viewRect, int mode) {
        this.simpleValidateView(viewRect, mode);
        if (this.mTransformation.isVoidTransformation()) {
            return null;
        }
        DepictorTransformation t = this.mTransformation;
        this.mTransformation.applyTo(this.mMol);
        this.mTransformation = new DepictorTransformation();
        return t;
    }

    public StereoMolecule getMolecule() {
        return this.mMol;
    }

    public DepictorTransformation validateView(T context, GenericRectangle viewRect, int mode) {
        if (this.mMol.getAllAtoms() == 0) {
            return null;
        }
        DepictorTransformation t1 = this.simpleValidateView(viewRect, mode);
        this.mMol.ensureHelperArrays(this.requiredHelperArrays());
        this.markIsolatedAtoms();
        this.mpDot.clear();
        this.mpTabuZone.clear();
        this.mContext = context;
        this.calculateParameters();
        this.mpSetNormalLabelSize();
        this.mIsValidatingView = true;
        for (int i = 0; i < this.mMol.getAllAtoms(); ++i) {
            this.mpDrawAtom(i, null);
        }
        this.mIsValidatingView = false;
        double avbl = this.mTransformation.getScaling() * this.mMol.getAverageBondLength();
        this.expandBoundsByTabuZones(avbl);
        this.setChiralTextLocation(viewRect, avbl, mode);
        if (viewRect == null || viewRect.contains(this.mBoundingRect)) {
            return t1;
        }
        DepictorTransformation t2 = new DepictorTransformation(this.mBoundingRect, viewRect, avbl, mode);
        t2.applyTo(this.mTransformation);
        t2.applyTo(this.mBoundingRect);
        t2.applyTo(this.mChiralTextLocation);
        if (t1 == null) {
            return t2;
        }
        t2.applyTo(t1);
        return t1;
    }

    public DepictorTransformation simpleValidateView(GenericRectangle viewRect, int mode) {
        if (this.mMol.getAllAtoms() == 0) {
            return null;
        }
        this.mBoundingRect = this.simpleCalculateBounds();
        double avbl = this.mTransformation.getScaling() * this.mMol.getAverageBondLength();
        DepictorTransformation t = new DepictorTransformation(this.mBoundingRect, viewRect, avbl, mode);
        if (t.isVoidTransformation()) {
            t = null;
        } else {
            t.applyTo(this.mTransformation);
            t.applyTo(this.mBoundingRect);
        }
        this.setChiralTextLocation(viewRect, avbl, mode);
        return t;
    }

    protected void onDrawBond(int bond, double x1, double y1, double x2, double y2) {
    }

    protected void onDrawAtom(int atom, String symbol, double x, double y) {
    }

    public GenericRectangle simpleCalculateBounds() {
        double minx = this.getAtomX(0);
        double maxx = this.getAtomX(0);
        double miny = this.getAtomY(0);
        double maxy = this.getAtomY(0);
        for (int i = 0; i < this.mMol.getAllAtoms(); ++i) {
            if (minx > this.getAtomX(i)) {
                minx = this.getAtomX(i);
            }
            if (maxx < this.getAtomX(i)) {
                maxx = this.getAtomX(i);
            }
            if (miny > this.getAtomY(i)) {
                miny = this.getAtomY(i);
            }
            if (!(maxy < this.getAtomY(i))) continue;
            maxy = this.getAtomY(i);
        }
        return new GenericRectangle(minx, miny, maxx - minx, maxy - miny);
    }

    private void expandBoundsByTabuZones(double avbl) {
        for (int i = 0; i < this.mpTabuZone.size(); ++i) {
            this.mBoundingRect = this.mBoundingRect.union(this.mpTabuZone.get(i));
        }
        this.expandByHiliteBackgrounds(avbl);
        double border = 0.1 * avbl;
        this.mBoundingRect.x -= border;
        this.mBoundingRect.y -= border;
        this.mBoundingRect.width += 2.0 * border;
        this.mBoundingRect.height += 2.0 * border;
    }

    private void expandByHiliteBackgrounds(double avbl) {
        boolean[] isAtomHilited = new boolean[this.mMol.getAllAtoms()];
        for (int i = 0; i < this.mMol.getAllBonds(); ++i) {
            if (!this.mMol.isBondBackgroundHilited(i)) continue;
            isAtomHilited[this.mMol.getBondAtom((int)0, (int)i)] = true;
            isAtomHilited[this.mMol.getBondAtom((int)1, (int)i)] = true;
        }
        GenericRectangle rect = new GenericRectangle();
        for (int i = 0; i < this.mMol.getAllAtoms(); ++i) {
            double radius;
            double d = (this.mMol.getAtomQueryFeatures(i) & 0x20000000L) != 0L ? avbl * 0.47 : (radius = isAtomHilited[i] ? avbl * 0.38 : 0.0);
            if (radius == 0.0) continue;
            double x = this.mTransformation.transformX(this.mMol.getAtomX(i));
            double y = this.mTransformation.transformY(this.mMol.getAtomY(i));
            rect.set(x - radius, y - radius, radius * 2.0, radius * 2.0);
            this.mBoundingRect = this.mBoundingRect.union(rect);
        }
    }

    private void setChiralTextLocation(GenericRectangle viewRect, double avbl, int mode) {
        double spacing = avbl / 2.0;
        switch (mode & 0xC0000) {
            case 786432: {
                if (viewRect != null) {
                    this.mChiralTextLocation.x = viewRect.x + viewRect.width / 2.0;
                    this.mChiralTextLocation.y = viewRect.y + viewRect.height - spacing;
                    break;
                }
            }
            case 0: {
                this.mChiralTextLocation.x = this.mBoundingRect.x + this.mBoundingRect.width / 2.0;
                this.mChiralTextLocation.y = this.mBoundingRect.y + this.mBoundingRect.height + spacing;
                if (viewRect == null || !(this.mChiralTextLocation.y > viewRect.y + viewRect.height - spacing)) break;
                this.mChiralTextLocation.y = viewRect.y + viewRect.height - spacing;
                break;
            }
            case 524288: {
                if (viewRect != null) {
                    this.mChiralTextLocation.x = viewRect.x + viewRect.width / 2.0;
                    this.mChiralTextLocation.y = viewRect.y + spacing;
                    break;
                }
            }
            case 262144: {
                this.mChiralTextLocation.x = this.mBoundingRect.x + this.mBoundingRect.width / 2.0;
                this.mChiralTextLocation.y = this.mBoundingRect.y - spacing;
                if (viewRect == null || !(this.mChiralTextLocation.y < viewRect.y + spacing)) break;
                this.mChiralTextLocation.y = viewRect.y + spacing;
            }
        }
    }

    public double getAtomX(int atom) {
        return this.mTransformation.transformX(this.mMol.getAtomX(atom));
    }

    public double getAtomY(int atom) {
        return this.mTransformation.transformY(this.mMol.getAtomY(atom));
    }

    public final GenericRectangle getBoundingRect() {
        return this.mBoundingRect;
    }

    protected void init() {
        this.mFactorTextSize = 1.0;
        this.mTransformation = new DepictorTransformation();
        this.mpTabuZone = new ArrayList();
        this.mpDot = new ArrayList();
        this.mAtomLabelDisplayed = new boolean[this.mMol.getAllAtoms()];
        this.mChiralTextLocation = new GenericPoint();
        this.mStandardForegroundColor = 0;
        this.mCurrentColor = -1;
        this.updateBondHiliteColor();
    }

    private void updateBondHiliteColor() {
        int background = this.mOverruleBackground != 0 ? this.mOverruleBackground : (this.mCustomBackground != 0 ? this.mCustomBackground : -1);
        this.mBondBGHiliteColor = ColorHelper.intermediateColor(background, -10706689, 0.3f);
        this.mBondFGHiliteColor = ColorHelper.getContrastColor(Short.MIN_VALUE, background);
        this.mExcludeGroupBGColor = -24321;
        this.mExcludeGroupFGColor = -6291392;
    }

    private void calculateParameters() {
        double averageBondLength = this.mTransformation.getScaling() * this.mMol.getAverageBondLength();
        this.mpLineWidth = averageBondLength * 0.06;
        this.mpBondSpacing = averageBondLength * 0.15;
        this.mpBondHiliteRadius = averageBondLength * 0.38;
        this.mpExcludeGroupRadius = averageBondLength * 0.47;
        this.mpLabelSize = (int)(averageBondLength * this.mFactorTextSize * 0.6 + 0.5);
        this.mpDotDiameter = averageBondLength * 0.12;
        this.mpQFDiameter = averageBondLength * 0.4;
        this.mChiralTextSize = averageBondLength * 0.5 + 0.5;
    }

    public synchronized void paint(T context) {
        if (this.mMol.getAllAtoms() == 0) {
            return;
        }
        this.mMol.ensureHelperArrays(this.requiredHelperArrays());
        this.mContext = context;
        this.calculateParameters();
        int[][] esrGroupMemberCount = this.mMol.getESRGroupMemberCounts();
        boolean explicitAtomColors = false;
        this.mAtomColor = new int[this.mMol.getAllAtoms()];
        for (int atom = 0; atom < this.mMol.getAllAtoms(); ++atom) {
            this.mAtomColor[atom] = this.mMol.getAtomColor(atom);
            if (this.mAtomColor[atom] != 0) {
                explicitAtomColors = true;
            }
            if (this.mMol.isSelectedAtom(atom)) {
                this.mAtomColor[atom] = 128;
            }
            if (!this.mMol.getStereoProblem(atom) || (this.mDisplayMode & 0x1000) != 0) continue;
            this.mAtomColor[atom] = 256;
        }
        this.setColor_(-10);
        if (this.mAtomHiliteColor != null && this.mAtomHiliteColor.length >= this.mMol.getAtoms()) {
            this.hiliteAtomBackgrounds(this.mAtomHiliteColor, this.mAtomHiliteRadius);
        }
        this.hiliteExcludeGroups();
        this.hiliteBondBackgrounds();
        this.indicateQueryFeatures();
        this.addChiralInfo();
        this.mpSetNormalLabelSize();
        this.setLineWidth(this.mpLineWidth);
        this.setColor_(this.mStandardForegroundColor);
        this.markIsolatedAtoms();
        this.mpDot.clear();
        this.mpTabuZone.clear();
        for (int i = 0; i < this.mMol.getAllAtoms(); ++i) {
            if (this.isHighlightedAtom(i)) {
                this.setColor_(-3);
                this.mpDrawAtom(i, esrGroupMemberCount);
                this.setColor_(this.mStandardForegroundColor);
                continue;
            }
            if (this.mAtomColor[i] != 0) {
                this.setColor_(this.mAtomColor[i]);
                this.mpDrawAtom(i, esrGroupMemberCount);
                this.setColor_(this.mStandardForegroundColor);
                continue;
            }
            if (!explicitAtomColors && this.mMol.getMoleculeColor() != 1 && this.mMol.getAtomicNo(i) != 1 && this.mMol.getAtomicNo(i) != 6 && (this.mDisplayMode & 0x800) == 0 && this.mMol.getAtomList(i) == null && this.mMol.getAtomicNo(i) < ATOM_LABEL_COLOR.length) {
                this.setRGBColor(this.getContrastColor(ATOM_LABEL_COLOR[this.mMol.getAtomicNo(i)], i));
                this.mpDrawAtom(i, esrGroupMemberCount);
                this.setColor_(this.mStandardForegroundColor);
                continue;
            }
            this.mpDrawAtom(i, esrGroupMemberCount);
        }
        this.mpDrawAllDots();
        this.mpDrawBondQueryFeatures();
        this.mpDrawAllBonds(esrGroupMemberCount);
    }

    public Color getBackgroundColor() {
        return new Color(this.getBackgroundRGB());
    }

    public int getBackgroundRGB() {
        return this.mOverruleBackground != 0 ? this.mOverruleBackground : (this.mCustomBackground != 0 ? this.mCustomBackground : -1);
    }

    private int getContrastColor(int rgb, int atom) {
        int bg = this.mOverruleBackground != 0 ? this.mOverruleBackground : (this.mAtomHiliteColor != null && atom < this.mAtomHiliteColor.length && (this.mAtomHiliteColor[atom] & 0xFF000000) != 0 ? this.mAtomHiliteColor[atom] : (this.mCustomBackground != 0 ? this.mCustomBackground : -1));
        return ColorHelper.getContrastColor(rgb, bg);
    }

    private boolean isHighlightedAtom(int atom) {
        if (this.mMol.getAllConnAtoms(atom) == 0) {
            return false;
        }
        for (int i = 0; i < this.mMol.getAllConnAtoms(atom); ++i) {
            if (this.mMol.isBondForegroundHilited(this.mMol.getConnBond(atom, i))) continue;
            return false;
        }
        return true;
    }

    private int requiredHelperArrays() {
        return (this.mDisplayMode & 0x100) != 0 ? 63 : ((this.mDisplayMode & 0x200) != 0 ? 95 : ((this.mDisplayMode & 0x400) != 0 ? 159 : 31));
    }

    private void markIsolatedAtoms() {
        this.mAtomIsConnected = new boolean[this.mMol.getAllAtoms()];
        for (int bnd = 0; bnd < this.mMol.getAllBonds(); ++bnd) {
            this.mAtomIsConnected[this.mMol.getBondAtom((int)0, (int)bnd)] = true;
            this.mAtomIsConnected[this.mMol.getBondAtom((int)1, (int)bnd)] = true;
        }
    }

    private void addChiralInfo() {
        if ((this.mDisplayMode & 0x20) != 0) {
            return;
        }
        String chiralText = this.mMol.getChiralText();
        if (chiralText != null) {
            if (this.mChiralTextLocation.x == 0.0 && this.mChiralTextLocation.y == 0.0) {
                double avbl = this.mTransformation.getScaling() * this.mMol.getAverageBondLength();
                this.mBoundingRect = this.simpleCalculateBounds();
                this.expandBoundsByTabuZones(avbl);
                this.setChiralTextLocation(null, avbl, 0);
            }
            this.setTextSize((int)this.mChiralTextSize);
            if (this.mMol.getMoleculeColor() != 1) {
                this.setColor_(448);
            }
            this.drawString(chiralText, this.mChiralTextLocation.x, this.mChiralTextLocation.y + (double)0.3f * this.mChiralTextSize);
        }
    }

    public void hiliteAtomBackgrounds(int[] argb, float[] radius) {
        int background = this.mOverruleBackground != 0 ? this.mOverruleBackground : (this.mCustomBackground != 0 ? this.mCustomBackground : -1);
        double avbl = this.mTransformation.getScaling() * this.mMol.getAverageBondLength();
        for (int atom = 0; atom < this.mMol.getAtoms(); ++atom) {
            int alpha = (argb[atom] & 0xFF000000) >>> 24;
            if (alpha == 0) continue;
            int rgb = argb[atom];
            if (alpha != 255) {
                rgb = ColorHelper.intermediateColor(background, argb[atom], (float)alpha / 255.0f);
            }
            double r = radius == null ? 0.5 * avbl : 0.6 * (double)radius[atom] * avbl;
            this.setRGB(rgb);
            this.fillCircle(this.getAtomX(atom) - r, this.getAtomY(atom) - r, 2.0 * r);
        }
    }

    private void hiliteBondBackgrounds() {
        this.setLineWidth(2.0 * this.mpBondHiliteRadius);
        DepictorLine line = new DepictorLine();
        for (int bond = 0; bond < this.mMol.getAllBonds(); ++bond) {
            int atom1 = this.mMol.getBondAtom(0, bond);
            int atom2 = this.mMol.getBondAtom(1, bond);
            if (!this.mMol.isBondBackgroundHilited(bond)) continue;
            line.x1 = this.getAtomX(atom1);
            line.y1 = this.getAtomY(atom1);
            line.x2 = this.getAtomX(atom2);
            line.y2 = this.getAtomY(atom2);
            this.setColor_(-2);
            this.drawBlackLine(line);
        }
    }

    private void hiliteExcludeGroups() {
        if (this.mMol.isFragment()) {
            double radius = this.mpExcludeGroupRadius;
            this.setColor_(-7);
            for (int atom = 0; atom < this.mMol.getAtoms(); ++atom) {
                if ((this.mMol.getAtomQueryFeatures(atom) & 0x20000000L) == 0L) continue;
                this.fillCircle(this.getAtomX(atom) - radius, this.getAtomY(atom) - radius, 2.0 * radius);
            }
            this.setLineWidth(2.0 * this.mpExcludeGroupRadius);
            DepictorLine line = new DepictorLine();
            for (int bond = 0; bond < this.mMol.getAllBonds(); ++bond) {
                int atom1 = this.mMol.getBondAtom(0, bond);
                int atom2 = this.mMol.getBondAtom(1, bond);
                if ((this.mMol.getAtomQueryFeatures(atom1) & this.mMol.getAtomQueryFeatures(atom2) & 0x20000000L) == 0L) continue;
                line.x1 = this.getAtomX(atom1);
                line.y1 = this.getAtomY(atom1);
                line.x2 = this.getAtomX(atom2);
                line.y2 = this.getAtomY(atom2);
                this.drawBlackLine(line);
            }
        }
    }

    private void indicateQueryFeatures() {
        if (this.mMol.isFragment()) {
            this.setColor_(320);
            if ((this.mDisplayMode & 8) != 0) {
                for (int atom = 0; atom < this.mMol.getAtoms(); ++atom) {
                    if ((this.mMol.getAtomQueryFeatures(atom) & 0xFFFFFFFFDFFFFFFFL) == 0L) continue;
                    this.fillCircle(this.getAtomX(atom) - this.mpQFDiameter / 2.0, this.getAtomY(atom) - this.mpQFDiameter / 2.0, this.mpQFDiameter);
                }
            }
            for (int bond = 0; bond < this.mMol.getBonds(); ++bond) {
                if (this.mMol.getBondQueryFeatures(bond) == 0) continue;
                int atom1 = this.mMol.getBondAtom(0, bond);
                int atom2 = this.mMol.getBondAtom(1, bond);
                this.fillCircle((this.getAtomX(atom1) + this.getAtomX(atom2) - this.mpQFDiameter) / 2.0, (this.getAtomY(atom1) + this.getAtomY(atom2) - this.mpQFDiameter) / 2.0, this.mpQFDiameter);
            }
        }
    }

    private void mpDrawAllBonds(int[][] esrGroupMemberCount) {
        double y;
        double x;
        int i;
        this.mAlternativeCoords = new GenericPoint[this.mMol.getAllAtoms()];
        for (i = 0; i < this.mMol.getAllBonds(); ++i) {
            if (this.mMol.getBondType(i) != 2 && this.mMol.getBondType(i) != 26 && this.mMol.getBondType(i) != 64) continue;
            this.mpDrawBond(i);
        }
        for (i = 0; i < this.mMol.getAllBonds(); ++i) {
            if (this.mMol.getBondType(i) == 2 || this.mMol.getBondType(i) == 26 || this.mMol.getBondType(i) == 64) continue;
            this.mpDrawBond(i);
        }
        if ((this.mDisplayMode & 0x40) == 0) {
            for (i = 0; i < this.mMol.getAllBonds(); ++i) {
                if (this.mMol.getBondCIPParity(i) == 0) continue;
                String cipStr = null;
                if (this.mMol.getBondCIPParity(i) == 1 || this.mMol.getBondCIPParity(i) == 2) {
                    if (this.mMol.getBondOrder(i) == 2 || this.mMol.getBondESRType(i) == 0 || esrGroupMemberCount == null || esrGroupMemberCount[this.mMol.getBondESRType(i)][this.mMol.getBondESRGroup(i)] > 1) {
                        cipStr = this.mMol.getBondCIPParity(i) == 1 ? (this.mMol.getBondOrder(i) == 2 ? "E" : (this.mMol.isBondParityPseudo(i) ? "p" : "P")) : (this.mMol.getBondOrder(i) == 2 ? "Z" : (this.mMol.isBondParityPseudo(i) ? "m" : "M"));
                    }
                } else {
                    cipStr = "?";
                }
                if (cipStr == null) continue;
                this.mpSetSmallLabelSize();
                this.setColor_(this.mMol.isBondForegroundHilited(i) ? -3 : (this.mMol.getMoleculeColor() == 1 || (this.mDisplayMode & 0x2000) != 0 ? this.mStandardForegroundColor : 448));
                int atom1 = this.mMol.getBondAtom(0, i);
                int atom2 = this.mMol.getBondAtom(1, i);
                x = (this.getAtomX(atom1) + this.getAtomX(atom2)) / 2.0;
                y = (this.getAtomY(atom1) + this.getAtomY(atom2)) / 2.0;
                double dx = (this.getAtomX(atom1) - this.getAtomX(atom2)) / 3.0;
                double dy = (this.getAtomY(atom1) - this.getAtomY(atom2)) / 3.0;
                this.mpDrawString(x + dy, y - dx, cipStr, true);
                this.setColor_(this.mStandardForegroundColor);
                this.mpSetNormalLabelSize();
            }
        }
        if ((this.mDisplayMode & 4) != 0) {
            this.mpSetSmallLabelSize();
            this.setColor_(384);
            for (i = 0; i < this.mMol.getAllBonds(); ++i) {
                int atom1 = this.mMol.getBondAtom(0, i);
                int atom2 = this.mMol.getBondAtom(1, i);
                String type = this.mMol.isDelocalizedBond(i) ? "d" : (this.mMol.isAromaticBond(i) ? "a" : "");
                x = (this.getAtomX(atom1) + this.getAtomX(atom2)) / 2.0;
                y = (this.getAtomY(atom1) + this.getAtomY(atom2)) / 2.0;
                this.mpDrawString(x, y, type + String.valueOf(i), true);
            }
            this.setColor_(this.mStandardForegroundColor);
            this.mpSetNormalLabelSize();
        }
    }

    private void mpDrawBond(int bnd) {
        DepictorLine theLine = new DepictorLine();
        DepictorLine aLine = new DepictorLine();
        DepictorLine bLine = new DepictorLine();
        GenericPoint piBondOffset = new GenericPoint();
        GenericPoint nextBondOffset = new GenericPoint();
        int atom1 = this.mMol.getBondAtom(0, bnd);
        int atom2 = this.mMol.getBondAtom(1, bnd);
        this.onDrawBond(bnd, this.getAtomX(atom1), this.getAtomY(atom1), this.getAtomX(atom2), this.getAtomY(atom2));
        if (!this.mMol.isSelectedAtom(atom1) && !this.mMol.isSelectedAtom(atom2) && ((this.mMol.getAtomQueryFeatures(atom1) | this.mMol.getAtomQueryFeatures(atom2)) & 0x20000000L) != 0L) {
            this.setColor_(-8);
        }
        if (this.mAlternativeCoords[atom1] == null) {
            theLine.x1 = this.getAtomX(atom1);
            theLine.y1 = this.getAtomY(atom1);
        } else {
            theLine.x1 = this.mAlternativeCoords[atom1].x;
            theLine.y1 = this.mAlternativeCoords[atom1].y;
        }
        if (this.mAlternativeCoords[atom2] == null) {
            theLine.x2 = this.getAtomX(atom2);
            theLine.y2 = this.getAtomY(atom2);
        } else {
            theLine.x2 = this.mAlternativeCoords[atom2].x;
            theLine.y2 = this.mAlternativeCoords[atom2].y;
        }
        if ((this.mMol.getBondQueryFeatures(bnd) & 0x7F80) != 0) {
            this.mpHandleDottedLine(theLine, atom1, atom2);
            this.setColor_(-9);
            return;
        }
        int bondOrder = this.mMol.getBondType(bnd) == 64 ? 0 : (this.mMol.getBondType(bnd) == 32 ? 1 : this.mMol.getBondOrder(bnd));
        block0 : switch (bondOrder) {
            case 1: {
                int stereoCenter;
                int esrType;
                int bondType = this.mMol.getBondType(bnd);
                if ((this.mDisplayMode & 0x80) != 0 && (bondType == 17 || bondType == 9) && (esrType = this.mMol.getAtomESRType(stereoCenter = this.mMol.getBondAtom(0, bnd))) != 0) {
                    int esrGroup = this.mMol.getAtomESRGroup(stereoCenter);
                    int count = 0;
                    for (int atom = 0; atom < this.mMol.getAtoms(); ++atom) {
                        if (this.mMol.getAtomESRType(atom) != esrType || this.mMol.getAtomESRGroup(atom) != esrGroup) continue;
                        ++count;
                    }
                    if (count == 1) {
                        bondType = 1;
                    }
                }
                switch (bondType) {
                    case 1: {
                        this.mpHandleLine(theLine, atom1, atom2);
                        break;
                    }
                    case 17: {
                        this.mpHandleWedge(theLine, atom1, atom2);
                        break;
                    }
                    case 9: {
                        int color2;
                        int color1;
                        double xdiff = theLine.x2 - theLine.x1;
                        double ydiff = theLine.y2 - theLine.y1;
                        if (this.mMol.isBondForegroundHilited(this.mMol.getBond(atom1, atom2))) {
                            color1 = -3;
                            color2 = -3;
                        } else {
                            color1 = this.mAtomColor[atom1];
                            color2 = this.getESRColor(atom1);
                            if (color1 == this.mMol.getAtomColor(atom1)) {
                                color1 = color2;
                            }
                        }
                        for (int i = 2; i < 17; i += 2) {
                            aLine.x1 = theLine.x1 + (double)i * xdiff / 17.0 - (double)i * ydiff / 128.0;
                            aLine.y1 = theLine.y1 + (double)i * ydiff / 17.0 + (double)i * xdiff / 128.0;
                            aLine.x2 = theLine.x1 + (double)i * xdiff / 17.0 + (double)i * ydiff / 128.0;
                            aLine.y2 = theLine.y1 + (double)i * ydiff / 17.0 - (double)i * xdiff / 128.0;
                            if (!this.mpProperLine(aLine)) continue;
                            this.setColor_(i < 9 ? color1 : color2);
                            this.drawBlackLine(aLine);
                            this.setColor_(this.mStandardForegroundColor);
                        }
                        break block0;
                    }
                    case 32: {
                        this.mpHandleShortDashedLine(theLine, atom1, atom2);
                    }
                }
                break;
            }
            case 0: 
            case 2: {
                if (!(!this.mAtomLabelDisplayed[atom1] && this.mMol.getAtomPi(atom1) != 2 || !this.mAtomLabelDisplayed[atom2] && this.mMol.getAtomPi(atom2) != 2 || this.mMol.isRingBond(bnd) || bondOrder != 2)) {
                    if (!this.mpProperLine(theLine)) break;
                    this.mpCalcPiBondOffset(theLine.x2 - theLine.x1, theLine.y2 - theLine.y1, piBondOffset);
                    double xdiff = piBondOffset.x / 2.0;
                    double ydiff = piBondOffset.y / 2.0;
                    aLine.x1 = theLine.x1 + xdiff;
                    aLine.y1 = theLine.y1 + ydiff;
                    aLine.x2 = theLine.x2 + xdiff;
                    aLine.y2 = theLine.y2 + ydiff;
                    bLine.x1 = theLine.x1 - xdiff;
                    bLine.y1 = theLine.y1 - ydiff;
                    bLine.x2 = theLine.x2 - xdiff;
                    bLine.y2 = theLine.y2 - ydiff;
                    if (this.mMol.getBondType(bnd) == 26) {
                        this.mpMakeCrossBond(aLine, bLine);
                    }
                    this.drawLine(aLine, atom1, atom2);
                    if (bondOrder == 2) {
                        this.drawLine(bLine, atom1, atom2);
                        break;
                    }
                    this.drawDashedLine(bLine, atom1, atom2);
                    break;
                }
                if ((this.mAtomLabelDisplayed[atom2] || this.mMol.getAtomPi(atom2) == 2) && bondOrder == 2) {
                    this.mpDBFromNonLabelToLabel(theLine, bnd, false);
                    break;
                }
                if ((this.mAtomLabelDisplayed[atom1] || this.mMol.getAtomPi(atom1) == 2) && bondOrder == 2) {
                    this.mpDBFromNonLabelToLabel(theLine, bnd, true);
                    break;
                }
                int side = this.mpPreferredSide(bnd);
                if (side == 0) {
                    side = 1;
                }
                aLine.x1 = theLine.x1;
                aLine.y1 = theLine.y1;
                aLine.x2 = theLine.x2;
                aLine.y2 = theLine.y2;
                this.mpCalcPiBondOffset(theLine.x2 - theLine.x1, theLine.y2 - theLine.y1, piBondOffset);
                if (side > 0) {
                    bLine.x1 = theLine.x1 + piBondOffset.x;
                    bLine.y1 = theLine.y1 + piBondOffset.y;
                    bLine.x2 = theLine.x2 + piBondOffset.x;
                    bLine.y2 = theLine.y2 + piBondOffset.y;
                    if (this.mpCalcNextBondOffset(atom1, atom2, 1, nextBondOffset) || this.mMol.getConnAtoms(atom1) > 1) {
                        bLine.x1 += nextBondOffset.x + piBondOffset.y;
                        bLine.y1 += nextBondOffset.y - piBondOffset.x;
                    }
                    if (this.mpCalcNextBondOffset(atom2, atom1, -1, nextBondOffset) || this.mMol.getConnAtoms(atom2) > 1) {
                        bLine.x2 += nextBondOffset.x - piBondOffset.y;
                        bLine.y2 += nextBondOffset.y + piBondOffset.x;
                    }
                } else {
                    bLine.x1 = theLine.x1 - piBondOffset.x;
                    bLine.y1 = theLine.y1 - piBondOffset.y;
                    bLine.x2 = theLine.x2 - piBondOffset.x;
                    bLine.y2 = theLine.y2 - piBondOffset.y;
                    if (this.mpCalcNextBondOffset(atom1, atom2, -1, nextBondOffset) || this.mMol.getConnAtoms(atom1) > 1) {
                        bLine.x1 += nextBondOffset.x + piBondOffset.y;
                        bLine.y1 += nextBondOffset.y - piBondOffset.x;
                    }
                    if (this.mpCalcNextBondOffset(atom2, atom1, 1, nextBondOffset) || this.mMol.getConnAtoms(atom2) > 1) {
                        bLine.x2 += nextBondOffset.x - piBondOffset.y;
                        bLine.y2 += nextBondOffset.y + piBondOffset.x;
                    }
                }
                if (this.mMol.getBondType(bnd) == 26) {
                    this.mpMakeCrossBond(aLine, bLine);
                }
                this.mpHandleLine(aLine, atom1, atom2);
                if (bondOrder == 2) {
                    this.mpHandleLine(bLine, atom1, atom2);
                    break;
                }
                this.mpHandleDashedLine(bLine, atom1, atom2);
                break;
            }
            case 3: {
                if (!this.mpProperLine(theLine)) break;
                this.drawLine(theLine, atom1, atom2);
                this.mpCalcPiBondOffset(theLine.x2 - theLine.x1, theLine.y2 - theLine.y1, piBondOffset);
                aLine.x1 = theLine.x1 + piBondOffset.x;
                aLine.y1 = theLine.y1 + piBondOffset.y;
                aLine.x2 = theLine.x2 + piBondOffset.x;
                aLine.y2 = theLine.y2 + piBondOffset.y;
                this.drawLine(aLine, atom1, atom2);
                aLine.x1 = theLine.x1 - piBondOffset.x;
                aLine.y1 = theLine.y1 - piBondOffset.y;
                aLine.x2 = theLine.x2 - piBondOffset.x;
                aLine.y2 = theLine.y2 - piBondOffset.y;
                this.drawLine(aLine, atom1, atom2);
            }
        }
        if (this.mCurrentColor == -8) {
            this.setColor_(-9);
        }
    }

    private void mpDBFromNonLabelToLabel(DepictorLine theLine, int bnd, boolean inverted) {
        DepictorLine aLine = new DepictorLine();
        DepictorLine bLine = new DepictorLine();
        GenericPoint piBondOffset = new GenericPoint();
        GenericPoint nextBondOffset = new GenericPoint();
        int atm1 = this.mMol.getBondAtom(0, bnd);
        int atm2 = this.mMol.getBondAtom(1, bnd);
        if (inverted) {
            double td = theLine.x1;
            theLine.x1 = theLine.x2;
            theLine.x2 = td;
            td = theLine.y1;
            theLine.y1 = theLine.y2;
            theLine.y2 = td;
            int ti = atm1;
            atm1 = atm2;
            atm2 = ti;
        }
        if (!this.mpProperLine(theLine)) {
            return;
        }
        if (this.mMol.isRingBond(bnd)) {
            int side;
            aLine.x1 = theLine.x1;
            aLine.y1 = theLine.y1;
            aLine.x2 = theLine.x2;
            aLine.y2 = theLine.y2;
            int n = side = inverted ? -this.mpPreferredSide(bnd) : this.mpPreferredSide(bnd);
            if (side == 0) {
                side = 1;
            }
            this.mpCalcPiBondOffset(theLine.x2 - theLine.x1, theLine.y2 - theLine.y1, piBondOffset);
            if (side > 0) {
                bLine.x1 = theLine.x1 + piBondOffset.x;
                bLine.y1 = theLine.y1 + piBondOffset.y;
                bLine.x2 = theLine.x2 + piBondOffset.x;
                bLine.y2 = theLine.y2 + piBondOffset.y;
                if (this.mpCalcNextBondOffset(atm1, atm2, 1, nextBondOffset) || this.mMol.getConnAtoms(atm1) > 1) {
                    bLine.x1 += nextBondOffset.x + piBondOffset.y;
                    bLine.y1 += nextBondOffset.y - piBondOffset.x;
                }
            } else {
                bLine.x1 = theLine.x1 - piBondOffset.x;
                bLine.y1 = theLine.y1 - piBondOffset.y;
                bLine.x2 = theLine.x2 - piBondOffset.x;
                bLine.y2 = theLine.y2 - piBondOffset.y;
                if (this.mpCalcNextBondOffset(atm1, atm2, -1, nextBondOffset) || this.mMol.getConnAtoms(atm1) > 1) {
                    bLine.x1 += nextBondOffset.x + piBondOffset.y;
                    bLine.y1 += nextBondOffset.y - piBondOffset.x;
                }
            }
            if (this.mMol.getBondType(bnd) == 26) {
                this.mpMakeCrossBond(aLine, bLine);
            }
            this.mpHandleLine(aLine, atm1, atm2);
            if (this.mMol.getBondType(bnd) == 64) {
                this.mpHandleDashedLine(bLine, atm1, atm2);
            } else {
                this.mpHandleLine(bLine, atm1, atm2);
            }
        } else {
            this.mpCalcPiBondOffset(theLine.x2 - theLine.x1, theLine.y2 - theLine.y1, piBondOffset);
            double xdiff = piBondOffset.x / 2.0;
            double ydiff = piBondOffset.y / 2.0;
            boolean aLineIsInnerLine = false;
            aLine.x1 = theLine.x1 + xdiff;
            aLine.y1 = theLine.y1 + ydiff;
            aLine.x2 = theLine.x2 + xdiff;
            aLine.y2 = theLine.y2 + ydiff;
            if (this.mMol.getConnAtoms(atm1) > 1) {
                if (!this.mpCalcNextBondOffset(atm1, atm2, 1, nextBondOffset)) {
                    this.mAlternativeCoords[atm1] = new GenericPoint(aLine.x1, aLine.y1);
                } else {
                    aLine.x1 += nextBondOffset.x;
                    aLine.y1 += nextBondOffset.y;
                    if (this.mMol.getConnAtoms(atm1) == 2 && (nextBondOffset.x != 0.0 || nextBondOffset.y != 0.0)) {
                        aLine.x1 += piBondOffset.y;
                        aLine.y1 -= piBondOffset.x;
                    }
                }
            }
            bLine.x1 = theLine.x1 - xdiff;
            bLine.y1 = theLine.y1 - ydiff;
            bLine.x2 = theLine.x2 - xdiff;
            bLine.y2 = theLine.y2 - ydiff;
            if (this.mMol.getConnAtoms(atm1) > 1) {
                if (!this.mpCalcNextBondOffset(atm1, atm2, 0, nextBondOffset)) {
                    this.mAlternativeCoords[atm1] = new GenericPoint(bLine.x1, bLine.y1);
                    aLineIsInnerLine = true;
                } else {
                    bLine.x1 += nextBondOffset.x;
                    bLine.y1 += nextBondOffset.y;
                    if (this.mMol.getConnAtoms(atm1) == 2 && (nextBondOffset.x != 0.0 || nextBondOffset.y != 0.0)) {
                        bLine.x1 += piBondOffset.y;
                        bLine.y1 -= piBondOffset.x;
                    }
                }
            }
            if (this.mMol.getBondType(bnd) == 26) {
                this.mpMakeCrossBond(aLine, bLine);
            }
            if (this.mMol.getBondType(bnd) == 64) {
                if (aLineIsInnerLine) {
                    this.drawDashedLine(aLine, atm1, atm2);
                    this.drawLine(bLine, atm1, atm2);
                } else {
                    this.drawLine(aLine, atm1, atm2);
                    this.drawDashedLine(bLine, atm1, atm2);
                }
            } else {
                this.drawLine(aLine, atm1, atm2);
                this.drawLine(bLine, atm1, atm2);
            }
        }
    }

    private void mpMakeCrossBond(DepictorLine aLine, DepictorLine bLine) {
        double temp = aLine.x2;
        aLine.x2 = bLine.x2;
        bLine.x2 = temp;
        temp = aLine.y2;
        aLine.y2 = bLine.y2;
        bLine.y2 = temp;
    }

    private void mpCalcPiBondOffset(double dx, double dy, GenericPoint piBondOffset) {
        if (dx == 0.0) {
            piBondOffset.x = dy < 0.0 ? this.mpBondSpacing : -this.mpBondSpacing;
            piBondOffset.y = 0.0;
            return;
        }
        double alpha = Math.atan(dy / dx);
        if (dx < 0.0) {
            alpha += Math.PI;
        }
        piBondOffset.x = -(this.mpBondSpacing * Math.sin(alpha));
        piBondOffset.y = this.mpBondSpacing * Math.cos(alpha);
    }

    private boolean mpProperLine(DepictorLine theLine) {
        if (theLine.x1 == theLine.x2 && theLine.y1 == theLine.y2) {
            for (GenericRectangle tabuZone : this.mpTabuZone) {
                if (!tabuZone.contains(theLine.x1, theLine.y1)) continue;
                return false;
            }
            return true;
        }
        GenericRectangle theFrame = this.mpGetFrame(theLine);
        boolean endsExchanged = false;
        if (theLine.x1 > theLine.x2) {
            this.mpExchangeLineEnds(theLine);
            endsExchanged = true;
        }
        for (int i = 0; i < this.mpTabuZone.size(); ++i) {
            GenericRectangle tabuZone = this.mpTabuZone.get(i);
            if (tabuZone.x > theFrame.x + theFrame.width || tabuZone.y > theFrame.y + theFrame.height || theFrame.x > tabuZone.x + tabuZone.width || theFrame.y > tabuZone.y + tabuZone.height) continue;
            if (this.mpInTabuZone(theLine.x1, theLine.y1, i)) {
                if (this.mpInTabuZone(theLine.x2, theLine.y2, i)) {
                    if (endsExchanged) {
                        this.mpExchangeLineEnds(theLine);
                    }
                    return false;
                }
                this.mpShortenLine(theLine, 0, i);
                boolean retval = this.mpProperLine(theLine);
                if (endsExchanged) {
                    this.mpExchangeLineEnds(theLine);
                }
                return retval;
            }
            if (!this.mpInTabuZone(theLine.x2, theLine.y2, i)) continue;
            this.mpShortenLine(theLine, 1, i);
            boolean retval = this.mpProperLine(theLine);
            if (endsExchanged) {
                this.mpExchangeLineEnds(theLine);
            }
            return retval;
        }
        if (endsExchanged) {
            this.mpExchangeLineEnds(theLine);
        }
        return true;
    }

    private boolean mpCalcNextBondOffset(int atm1, int atm2, int side, GenericPoint nextBondOffset) {
        double RO_LIMIT = 2.617993878;
        double LO_LIMIT = 3.665191429;
        double RI_LIMIT = 0.523598776;
        double LI_LIMIT = 5.759586531;
        boolean retval = false;
        nextBondOffset.x = 0.0;
        nextBondOffset.y = 0.0;
        double angleDiff = side > 0 ? 2.617993878 : 3.665191429;
        double theBondAngle = this.mMol.getBondAngle(atm1, atm2);
        for (int i = 0; i < this.mMol.getConnAtoms(atm1); ++i) {
            double distance;
            int bnd = this.mMol.getConnBond(atm1, i);
            double bondAngle = theBondAngle;
            int remoteAtm = this.mMol.getBondAtom(0, bnd) == atm1 ? this.mMol.getBondAtom(1, bnd) : this.mMol.getBondAtom(0, bnd);
            if (remoteAtm == atm2) continue;
            double testAngle = this.mMol.getBondAngle(atm1, remoteAtm);
            if (bondAngle < testAngle) {
                bondAngle += Math.PI * 2;
            }
            double currentAngleDiff = bondAngle - testAngle;
            if (side > 0) {
                if (currentAngleDiff < Math.PI) {
                    retval = true;
                }
                if (currentAngleDiff > 2.617993878) {
                    currentAngleDiff = 2.617993878;
                }
                if (currentAngleDiff < 0.523598776) {
                    currentAngleDiff = 0.523598776;
                }
                if (!(currentAngleDiff <= angleDiff)) continue;
                angleDiff = currentAngleDiff;
                distance = this.mpBondSpacing * Math.tan(angleDiff - 1.5707963267948966) / 2.0;
                nextBondOffset.x = -(distance * Math.sin(bondAngle));
                nextBondOffset.y = -(distance * Math.cos(bondAngle));
                continue;
            }
            if (currentAngleDiff >= Math.PI) {
                retval = true;
            }
            if (currentAngleDiff < 3.665191429) {
                currentAngleDiff = 3.665191429;
            }
            if (currentAngleDiff > 5.759586531) {
                currentAngleDiff = 5.759586531;
            }
            if (!(currentAngleDiff >= angleDiff)) continue;
            angleDiff = currentAngleDiff;
            distance = this.mpBondSpacing * Math.tan(4.712388981 - angleDiff) / 2.0;
            nextBondOffset.x = -(distance * Math.sin(bondAngle));
            nextBondOffset.y = -(distance * Math.cos(bondAngle));
        }
        return retval;
    }

    private void mpExchangeLineEnds(DepictorLine theLine) {
        double temp = theLine.x1;
        theLine.x1 = theLine.x2;
        theLine.x2 = temp;
        temp = theLine.y1;
        theLine.y1 = theLine.y2;
        theLine.y2 = temp;
    }

    private void mpHandleLine(DepictorLine theLine, int atm1, int atm2) {
        if (this.mpProperLine(theLine)) {
            this.drawLine(theLine, atm1, atm2);
        }
    }

    private void mpHandleDashedLine(DepictorLine theLine, int atm1, int atm2) {
        if (this.mpProperLine(theLine)) {
            this.drawDashedLine(theLine, atm1, atm2);
        }
    }

    private void mpHandleShortDashedLine(DepictorLine theLine, int atm1, int atm2) {
        if (this.mpProperLine(theLine)) {
            this.drawShortDashedLine(theLine, atm1, atm2);
        }
    }

    private void mpHandleDottedLine(DepictorLine theLine, int atm1, int atm2) {
        if (this.mpProperLine(theLine)) {
            this.drawDottedLine(theLine);
        }
    }

    private void mpHandleWedge(DepictorLine origWedge, int atm1, int atm2) {
        DepictorLine theWedge = new DepictorLine();
        if (origWedge.x1 == origWedge.x2 && origWedge.y1 == origWedge.y2) {
            return;
        }
        theWedge.x1 = origWedge.x1;
        theWedge.y1 = origWedge.y1;
        theWedge.x2 = origWedge.x2;
        theWedge.y2 = origWedge.y2;
        GenericRectangle theFrame = this.mpGetFrame(theWedge);
        for (int i = 0; i < this.mpTabuZone.size(); ++i) {
            GenericRectangle tabuZone = this.mpTabuZone.get(i);
            if (tabuZone.x > theFrame.x + theFrame.width || tabuZone.y > theFrame.y + theFrame.height || theFrame.x > tabuZone.x + tabuZone.width || theFrame.y > tabuZone.y + tabuZone.height) continue;
            if (this.mpInTabuZone(theWedge.x1, theWedge.y1, i)) {
                if (this.mpInTabuZone(theWedge.x2, theWedge.y2, i)) {
                    return;
                }
                this.mpShortenLine(theWedge, 0, i);
                this.mpHandleWedge(theWedge, atm1, atm2);
                return;
            }
            if (!this.mpInTabuZone(theWedge.x2, theWedge.y2, i)) continue;
            this.mpShortenLine(theWedge, 1, i);
            this.mpHandleWedge(theWedge, atm1, atm2);
            return;
        }
        this.drawWedge(theWedge, atm1, atm2);
    }

    private GenericRectangle mpGetFrame(DepictorLine theLine) {
        GenericRectangle theFrame = new GenericRectangle();
        if (theLine.x1 <= theLine.x2) {
            theFrame.x = theLine.x1;
            theFrame.width = theLine.x2 - theLine.x1;
        } else {
            theFrame.x = theLine.x2;
            theFrame.width = theLine.x1 - theLine.x2;
        }
        if (theLine.y1 <= theLine.y2) {
            theFrame.y = theLine.y1;
            theFrame.height = theLine.y2 - theLine.y1;
        } else {
            theFrame.y = theLine.y2;
            theFrame.height = theLine.y1 - theLine.y2;
        }
        return theFrame;
    }

    private boolean mpInTabuZone(double x, double y, int tabuZoneNo) {
        if ((this.mDisplayMode & 1) != 0) {
            return false;
        }
        GenericRectangle tabuZone = this.mpTabuZone.get(tabuZoneNo);
        return x > tabuZone.x && x < tabuZone.x + tabuZone.width && y > tabuZone.y && y < tabuZone.y + tabuZone.height;
    }

    private void mpShortenLine(DepictorLine theLine, int pointNo, int tabuZoneNo) {
        double sy;
        double sx;
        double y2;
        double x2;
        double y1;
        double x1;
        if (pointNo == 0) {
            x1 = theLine.x1;
            y1 = theLine.y1;
            x2 = theLine.x2;
            y2 = theLine.y2;
        } else {
            x1 = theLine.x2;
            y1 = theLine.y2;
            x2 = theLine.x1;
            y2 = theLine.y1;
        }
        GenericRectangle tabuZone = this.mpTabuZone.get(tabuZoneNo);
        double tabuX = x2 > x1 ? tabuZone.x + tabuZone.width : tabuZone.x;
        double tabuY = y2 > y1 ? tabuZone.y + tabuZone.height : tabuZone.y;
        double dx = x2 - x1;
        double dy = y2 - y1;
        if (Math.abs(dx) > Math.abs(dy)) {
            if (y1 == y2) {
                sx = tabuX;
                sy = y1;
            } else if (x2 > x1 == tabuX > (sx = x1 + dx * (tabuY - y1) / dy)) {
                sy = tabuY;
            } else {
                sx = tabuX;
                sy = y1 + dy * (tabuX - x1) / dx;
            }
        } else if (x1 == x2) {
            sx = x1;
            sy = tabuY;
        } else if (y2 > y1 == tabuY > (sy = y1 + dy * (tabuX - x1) / dx)) {
            sx = tabuX;
        } else {
            sx = x1 + dx * (tabuY - y1) / dy;
            sy = tabuY;
        }
        if (pointNo == 0) {
            theLine.x1 = sx;
            theLine.y1 = sy;
        } else {
            theLine.x2 = sx;
            theLine.y2 = sy;
        }
    }

    private int mpPreferredSide(int bnd) {
        boolean changed;
        boolean[] isAromatic = new boolean[16];
        boolean[] isInRing = new boolean[16];
        double[] angle = new double[16];
        double[] bondAngle = new double[2];
        int angles = 0;
        for (int i = 0; i < 2; ++i) {
            int atm = this.mMol.getBondAtom(i, bnd);
            for (int j = 0; j < this.mMol.getConnAtoms(atm); ++j) {
                int connBond = this.mMol.getConnBond(atm, j);
                if (connBond == bnd) continue;
                if (angles == 4) {
                    return 0;
                }
                isAromatic[angles] = this.mMol.isAromaticBond(connBond);
                isInRing[angles] = this.mMol.isRingBond(connBond);
                angle[angles++] = this.mMol.getBondAngle(atm, this.mMol.getConnAtom(atm, j));
            }
        }
        bondAngle[0] = this.mMol.getBondAngle(this.mMol.getBondAtom(0, bnd), this.mMol.getBondAtom(1, bnd));
        if (bondAngle[0] < 0.0) {
            bondAngle[1] = bondAngle[0] + Math.PI;
            changed = false;
        } else {
            bondAngle[1] = bondAngle[0];
            bondAngle[0] = bondAngle[1] - Math.PI;
            changed = true;
        }
        int side = 0;
        for (int i = 0; i < angles; ++i) {
            int value = isAromatic[i] ? 20 : (isInRing[i] ? 17 : 16);
            if (angle[i] > bondAngle[0] && angle[i] < bondAngle[1]) {
                side -= value;
                continue;
            }
            side += value;
        }
        return changed ? -side : side;
    }

    private void mpDrawAtom(int atom, int[][] esrGroupMemberCount) {
        double chax;
        double chay;
        int theColor;
        double y;
        double x;
        int esrInfo;
        if (!this.mIsValidatingView) {
            this.onDrawAtom(atom, this.mMol.getAtomLabel(atom), this.getAtomX(atom), this.getAtomY(atom));
        }
        String propStr = null;
        if (this.mMol.getAtomCharge(atom) != 0) {
            String valStr = Math.abs(this.mMol.getAtomCharge(atom)) == 1 ? "" : String.valueOf(Math.abs(this.mMol.getAtomCharge(atom)));
            String string = propStr = this.mMol.getAtomCharge(atom) < 0 ? valStr + "-" : valStr + "+";
        }
        if (this.mAtomText != null && atom < this.mAtomText.length && this.mAtomText[atom] != null && this.mAtomText[atom].length() > 0) {
            propStr = this.append(propStr, this.mAtomText[atom]);
        }
        String isoStr = null;
        long queryFeatures = this.mMol.getAtomQueryFeatures(atom);
        if (queryFeatures != 0L) {
            if ((queryFeatures & 2L) != 0L) {
                isoStr = this.append(isoStr, "a");
            }
            if ((queryFeatures & 4L) != 0L) {
                isoStr = this.append(isoStr, "!a");
            }
            if ((queryFeatures & 0x1000L) != 0L) {
                isoStr = this.append(isoStr, "s");
            }
            if ((queryFeatures & 0x780L) != 0L) {
                long hydrogens = queryFeatures & 0x780L;
                if (hydrogens == 1792L) {
                    isoStr = this.append(isoStr, "h0");
                } else if (hydrogens == 1664L) {
                    isoStr = this.append(isoStr, "h1");
                } else if (hydrogens == 1408L) {
                    isoStr = this.append(isoStr, "h2");
                } else if (hydrogens == 128L) {
                    isoStr = this.append(isoStr, "h>0");
                } else if (hydrogens == 384L) {
                    isoStr = this.append(isoStr, "h>1");
                } else if (hydrogens == 896L) {
                    isoStr = this.append(isoStr, "h>2");
                } else if (hydrogens == 1024L) {
                    isoStr = this.append(isoStr, "h<3");
                } else if (hydrogens == 1536L) {
                    isoStr = this.append(isoStr, "h<2");
                }
            }
            if ((queryFeatures & 0xE000000L) != 0L) {
                long charge = queryFeatures & 0xE000000L;
                if (charge == 0xA000000L) {
                    isoStr = this.append(isoStr, "c0");
                } else if (charge == 0x6000000L) {
                    isoStr = this.append(isoStr, "c+");
                } else if (charge == 0xC000000L) {
                    isoStr = this.append(isoStr, "c-");
                }
            }
            if ((queryFeatures & 0x1C000L) != 0L) {
                long piElectrons = queryFeatures & 0x1C000L;
                if (piElectrons == 98304L) {
                    isoStr = this.append(isoStr, "pi0");
                } else if (piElectrons == 81920L) {
                    isoStr = this.append(isoStr, "pi1");
                } else if (piElectrons == 49152L) {
                    isoStr = this.append(isoStr, "pi2");
                } else if (piElectrons == 16384L) {
                    isoStr = this.append(isoStr, "pi>0");
                }
            }
            if ((queryFeatures & 0x3E0000L) != 0L) {
                long neighbours = queryFeatures & 0x3E0000L;
                if (neighbours == 0x3A0000L) {
                    isoStr = this.append(isoStr, "n1");
                } else if (neighbours == 0x360000L) {
                    isoStr = this.append(isoStr, "n2");
                } else if (neighbours == 0x2E0000L) {
                    isoStr = this.append(isoStr, "n3");
                } else if (neighbours == 0x300000L) {
                    isoStr = this.append(isoStr, "n<3");
                } else if (neighbours == 0x200000L) {
                    isoStr = this.append(isoStr, "n<4");
                } else if (neighbours == 393216L) {
                    isoStr = this.append(isoStr, "n>1");
                } else if (neighbours == 917504L) {
                    isoStr = this.append(isoStr, "n>2");
                } else if (neighbours == 0x1E0000L) {
                    isoStr = this.append(isoStr, "n>3");
                }
            }
            if ((queryFeatures & 0x78L) != 0L) {
                long ringBonds = queryFeatures & 0x78L;
                if (ringBonds == 112L) {
                    isoStr = this.append(isoStr, "!r");
                } else if (ringBonds == 8L) {
                    isoStr = this.append(isoStr, "r");
                } else if (ringBonds == 104L) {
                    isoStr = this.append(isoStr, "rb2");
                } else if (ringBonds == 88L) {
                    isoStr = this.append(isoStr, "rb3");
                } else if (ringBonds == 56L) {
                    isoStr = this.append(isoStr, "rb4");
                }
            }
            if ((queryFeatures & 0x1C00000L) != 0L) {
                isoStr = this.append(isoStr, "r" + ((queryFeatures & 0x1C00000L) >> 22));
            }
            if ((queryFeatures & 0x7F00000000L) != 0L) {
                isoStr = this.append(isoStr, this.createRingSizeText(queryFeatures));
            }
            if ((queryFeatures & 0x10000000L) != 0L) {
                isoStr = this.append(isoStr, "f");
            }
        }
        if (this.mMol.getAtomMass(atom) != 0) {
            isoStr = this.append(isoStr, String.valueOf(this.mMol.getAtomMass(atom)));
        }
        int unpairedElectrons = 0;
        if (this.mMol.getAtomRadical(atom) != 0) {
            switch (this.mMol.getAtomRadical(atom)) {
                case 16: {
                    propStr = this.append(propStr, "|");
                    break;
                }
                case 32: {
                    unpairedElectrons = 1;
                    break;
                }
                case 48: {
                    unpairedElectrons = 2;
                }
            }
        }
        String cipStr = null;
        if ((this.mDisplayMode & 0x40) == 0) {
            if (this.mMol.isAtomConfigurationUnknown(atom)) {
                cipStr = "?";
            } else if (this.mMol.getAtomCIPParity(atom) != 0 && (this.mMol.getAtomESRType(atom) == 0 || esrGroupMemberCount == null || esrGroupMemberCount[this.mMol.getAtomESRType(atom)][this.mMol.getAtomESRGroup(atom)] > 1)) {
                if (this.mMol.getConnAtoms(atom) == 2) {
                    switch (this.mMol.getAtomCIPParity(atom)) {
                        case 2: {
                            cipStr = this.mMol.isAtomParityPseudo(atom) ? "p" : "P";
                            break;
                        }
                        case 1: {
                            cipStr = this.mMol.isAtomParityPseudo(atom) ? "m" : "M";
                            break;
                        }
                        default: {
                            cipStr = "*";
                            break;
                        }
                    }
                } else {
                    switch (this.mMol.getAtomCIPParity(atom)) {
                        case 1: {
                            cipStr = this.mMol.isAtomParityPseudo(atom) ? "r" : "R";
                            break;
                        }
                        case 2: {
                            cipStr = this.mMol.isAtomParityPseudo(atom) ? "s" : "S";
                            break;
                        }
                        default: {
                            cipStr = "*";
                        }
                    }
                }
            }
        }
        if ((this.mDisplayMode & 0x700) != 0) {
            cipStr = this.append(cipStr, String.valueOf(this.mMol.getSymmetryRank(atom)));
        }
        String mapStr = null;
        if ((this.mDisplayMode & 0x10) != 0 && this.mMol.getAtomMapNo(atom) != 0) {
            mapStr = "" + this.mMol.getAtomMapNo(atom);
        }
        String esrStr = null;
        if (this.mMol.getStereoBond(atom) != -1 && (esrInfo = this.getESRTypeToDisplayAt(atom)) != -1) {
            esrStr = esrInfo == 0 ? "abs" : ((esrInfo & 0xFF) == 1 ? "&" : "or") + (1 + (esrInfo >> 8));
        }
        int hydrogensToAdd = 0;
        if ((this.mDisplayMode & 0x4000) == 0) {
            if (this.mMol.isFragment()) {
                if ((this.mMol.getAtomQueryFeatures(atom) & 0x800L) != 0L) {
                    hydrogensToAdd = this.mMol.getImplicitHydrogens(atom);
                }
            } else if (this.mMol.getAtomicNo(atom) != 6 || this.mMol.getAtomMass(atom) != 0 || !this.mAtomIsConnected[atom] || this.mMol.getAtomRadical(atom) != 0) {
                hydrogensToAdd = this.mMol.getImplicitHydrogens(atom);
            }
        }
        boolean largeIsoString = false;
        String atomStr = this.mMol.getAtomCustomLabel(atom);
        if (atomStr != null && atomStr.startsWith("]")) {
            isoStr = this.append(atomStr.substring(1), isoStr);
            atomStr = null;
            largeIsoString = true;
        }
        if (atomStr != null) {
            hydrogensToAdd = 0;
        } else if (this.mMol.getAtomList(atom) != null) {
            String atmStart = (this.mMol.getAtomQueryFeatures(atom) & 1L) != 0L ? "[!" : "[";
            atomStr = atmStart + this.mMol.getAtomListString(atom) + "]";
            if (atomStr.length() > 5) {
                atomStr = atmStart + this.mMol.getAtomList(atom).length + "]";
            }
            if ((this.mMol.getAtomQueryFeatures(atom) & 0x800L) != 0L) {
                hydrogensToAdd = -1;
            }
        } else if ((this.mMol.getAtomQueryFeatures(atom) & 1L) != 0L) {
            atomStr = "?";
            if ((this.mMol.getAtomQueryFeatures(atom) & 0x800L) != 0L) {
                hydrogensToAdd = -1;
            }
        } else if (this.mMol.getAtomicNo(atom) != 6 || propStr != null || isoStr != null || hydrogensToAdd > 0 || !this.mAtomIsConnected[atom]) {
            atomStr = this.mMol.getAtomLabel(atom);
        }
        double labelWidth = 0.0;
        if (!this.mMol.isSelectedAtom(atom) & (this.mMol.getAtomQueryFeatures(atom) & 0x20000000L) != 0L) {
            this.setColor_(-8);
        }
        if (atomStr != null) {
            labelWidth = this.getStringWidth(atomStr);
            this.mpDrawString(this.getAtomX(atom), this.getAtomY(atom), atomStr, true);
            this.mAtomLabelDisplayed[atom] = true;
        } else if (this.mpAlleneCenter(atom)) {
            this.mpDrawDot(this.getAtomX(atom), this.getAtomY(atom), atom);
        }
        if (propStr != null) {
            this.mpSetSmallLabelSize();
            x = this.getAtomX(atom) + ((labelWidth + this.getStringWidth(propStr)) / 2.0 + 1.0);
            y = this.getAtomY(atom) - (double)((this.getTextSize() * 4 - 4) / 8);
            this.mpDrawString(x, y, propStr, true);
            this.mpSetNormalLabelSize();
        }
        if ((this.mDisplayMode & 2) != 0) {
            isoStr = String.valueOf(atom);
        }
        if (isoStr != null) {
            if (largeIsoString) {
                this.mpSetReducedLabelSize();
            } else {
                this.mpSetSmallLabelSize();
            }
            x = this.getAtomX(atom) - (labelWidth + this.getStringWidth(isoStr)) / 2.0;
            y = this.getAtomY(atom) - (double)((this.getTextSize() * 4 - 4) / 8);
            this.mpDrawString(x, y, isoStr, true);
            this.mpSetNormalLabelSize();
        }
        if (cipStr != null) {
            this.mpSetSmallLabelSize();
            x = this.getAtomX(atom) - (labelWidth + this.getStringWidth(cipStr)) / 2.0;
            y = this.getAtomY(atom) + (double)((this.getTextSize() * 4 + 4) / 8);
            theColor = this.mCurrentColor;
            if (this.mMol.getMoleculeColor() != 1 && (this.mDisplayMode & 0x2000) == 0) {
                this.setColor_(448);
            }
            this.mpDrawString(x, y, cipStr, false);
            this.setColor_(theColor);
            this.mpSetNormalLabelSize();
        }
        if (mapStr != null) {
            this.mpSetSmallLabelSize();
            x = this.getAtomX(atom) + ((labelWidth + this.getStringWidth(mapStr)) / 2.0 + 1.0);
            y = this.getAtomY(atom) + (double)((this.getTextSize() * 4 + 4) / 8);
            theColor = this.mCurrentColor;
            this.setColor_(this.mMol.isAutoMappedAtom(atom) ? 384 : 448);
            this.mpDrawString(x, y, mapStr, true);
            this.setColor_(theColor);
            this.mpSetNormalLabelSize();
        }
        if (esrStr != null) {
            double angle = this.mpGetFreeSpaceAngle(atom);
            this.mpSetSmallLabelSize();
            x = this.getAtomX(atom) + 0.7 * (double)this.getTextSize() * Math.sin(angle);
            y = this.getAtomY(atom) + 0.7 * (double)this.getTextSize() * Math.cos(angle);
            int theColor2 = this.mCurrentColor;
            if (!this.mIsValidatingView && this.mMol.getMoleculeColor() != 1) {
                this.setColor_(this.getESRColor(atom));
            }
            this.mpDrawString(x, y, esrStr, false);
            this.setColor_(theColor2);
            this.mpSetNormalLabelSize();
        }
        if (hydrogensToAdd == 0 && unpairedElectrons == 0) {
            if (this.mCurrentColor == -8) {
                this.setColor_(-9);
            }
            return;
        }
        double[] hindrance = new double[4];
        for (int i = 0; i < this.mMol.getAllConnAtomsPlusMetalBonds(atom); ++i) {
            int bnd = this.mMol.getConnBond(atom, i);
            for (int j = 0; j < 2; ++j) {
                if (this.mMol.getBondAtom(j, bnd) != atom) continue;
                double theAngle = this.mMol.getBondAngle(this.mMol.getBondAtom(j, bnd), this.mMol.getBondAtom(1 - j, bnd));
                if (theAngle < -1.5707963267948966) {
                    hindrance[0] = hindrance[0] - (theAngle + 1.5707963267948966);
                    hindrance[3] = hindrance[3] + (theAngle + Math.PI);
                    continue;
                }
                if (theAngle < 0.0) {
                    hindrance[2] = hindrance[2] + (theAngle + 1.5707963267948966);
                    hindrance[3] = hindrance[3] - theAngle;
                    continue;
                }
                if (theAngle < 1.5707963267948966) {
                    hindrance[1] = hindrance[1] + theAngle;
                    hindrance[2] = hindrance[2] + (1.5707963267948966 - theAngle);
                    continue;
                }
                hindrance[0] = hindrance[0] + (theAngle - 1.5707963267948966);
                hindrance[1] = hindrance[1] + (Math.PI - theAngle);
            }
        }
        if (this.mMol.getConnAtoms(atom) == 0) {
            if (this.mMol.isElectronegative(atom)) {
                hindrance[3] = hindrance[3] - 0.2;
            } else {
                hindrance[1] = hindrance[1] - 0.2;
            }
        } else {
            hindrance[1] = hindrance[1] - 0.1;
        }
        if (propStr != null || mapStr != null) {
            hindrance[1] = hindrance[1] + 10.0;
        }
        if (isoStr != null || cipStr != null) {
            hindrance[3] = hindrance[3] + 10.0;
        }
        String hNoStr = "";
        if (hydrogensToAdd != 0) {
            double hydrogenWidth = this.getStringWidth("H");
            double hNoWidth = 0.0;
            double hHeight = this.getTextSize();
            if (hydrogensToAdd == -1) {
                hNoStr = "n";
                this.mpSetSmallLabelSize();
                hNoWidth = this.getStringWidth(hNoStr);
            } else if (hydrogensToAdd > 1) {
                hNoStr = String.valueOf(hydrogensToAdd);
                this.mpSetSmallLabelSize();
                hNoWidth = this.getStringWidth(hNoStr);
            }
            if (hindrance[1] < 0.6 || hindrance[3] < 0.6) {
                chay = this.getAtomY(atom);
                if (hindrance[1] <= hindrance[3]) {
                    hindrance[1] = hindrance[1] + 10.0;
                    chax = this.getAtomX(atom) + (labelWidth + hydrogenWidth) / 2.0;
                } else {
                    hindrance[3] = hindrance[3] + 10.0;
                    chax = this.getAtomX(atom) - (labelWidth + hydrogenWidth) / 2.0 - hNoWidth;
                }
            } else {
                chax = this.getAtomX(atom);
                if (hindrance[0] < hindrance[2]) {
                    hindrance[0] = hindrance[0] + 10.0;
                    chay = this.getAtomY(atom) - hHeight;
                } else {
                    hindrance[2] = hindrance[2] + 10.0;
                    chay = this.getAtomY(atom) + hHeight;
                }
            }
            if (hNoWidth > 0.0) {
                x = chax + (hydrogenWidth + hNoWidth) / 2.0;
                y = chay + (double)((this.getTextSize() * 4 + 4) / 8);
                this.mpDrawString(x, y, hNoStr, true);
                this.mpSetNormalLabelSize();
            }
            this.mpDrawString(chax, chay, "H", true);
        }
        int bestSide = 0;
        if (unpairedElectrons != 0) {
            double minHindrance = 50.0;
            double counterHindrance = 0.0;
            for (int i = 0; i < 4; ++i) {
                int counterSide;
                int n = counterSide = i > 1 ? i - 2 : i + 2;
                if (hindrance[i] < minHindrance) {
                    bestSide = i;
                    minHindrance = hindrance[i];
                    counterHindrance = hindrance[counterSide];
                    continue;
                }
                if (hindrance[i] != minHindrance || !(hindrance[counterSide] > counterHindrance)) continue;
                bestSide = i;
                counterHindrance = hindrance[counterSide];
            }
            switch (bestSide) {
                case 0: {
                    chax = this.getAtomX(atom);
                    chay = this.getAtomY(atom) - this.mpDotDiameter - labelWidth / 2.0;
                    break;
                }
                case 1: {
                    chax = this.getAtomX(atom) + this.mpDotDiameter + labelWidth / 2.0;
                    chay = this.getAtomY(atom);
                    break;
                }
                case 2: {
                    chax = this.getAtomX(atom);
                    chay = this.getAtomY(atom) + this.mpDotDiameter + labelWidth / 2.0;
                    break;
                }
                default: {
                    chax = this.getAtomX(atom) - this.mpDotDiameter - labelWidth / 2.0;
                    chay = this.getAtomY(atom);
                }
            }
            if (unpairedElectrons == 1) {
                this.mpDrawDot(chax, chay, atom);
            } else {
                double ydiff;
                double xdiff;
                switch (bestSide) {
                    case 0: {
                        xdiff = 2.0 * this.mpDotDiameter;
                        ydiff = 0.0;
                        chax -= this.mpDotDiameter;
                        break;
                    }
                    case 1: {
                        xdiff = 0.0;
                        ydiff = 2.0 * this.mpDotDiameter;
                        chay -= this.mpDotDiameter;
                        break;
                    }
                    case 2: {
                        xdiff = 2.0 * this.mpDotDiameter;
                        ydiff = 0.0;
                        chax -= this.mpDotDiameter;
                        break;
                    }
                    default: {
                        xdiff = 0.0;
                        ydiff = 2.0 * this.mpDotDiameter;
                        chay -= this.mpDotDiameter;
                    }
                }
                this.mpDrawDot(chax, chay, atom);
                this.mpDrawDot(chax + xdiff, chay + ydiff, atom);
            }
        }
        if (this.mCurrentColor == -8) {
            this.setColor_(-9);
        }
    }

    private String createRingSizeText(long queryFeatures) {
        queryFeatures &= 0x7F00000000L;
        for (int i = 0; i < AtomQueryFeatureDialogBuilder.RING_SIZE_VALUES.length; ++i) {
            if (queryFeatures != AtomQueryFeatureDialogBuilder.RING_SIZE_VALUES[i]) continue;
            return AtomQueryFeatureDialogBuilder.RING_SIZE_SHORT_TEXT[i];
        }
        StringBuilder customOption = new StringBuilder("R");
        if ((queryFeatures & 0x100000000L) != 0L) {
            customOption.append("0");
        }
        if ((queryFeatures & 0x200000000L) != 0L) {
            customOption.append("3");
        }
        if ((queryFeatures & 0x400000000L) != 0L) {
            customOption.append("4");
        }
        if ((queryFeatures & 0x800000000L) != 0L) {
            customOption.append("5");
        }
        if ((queryFeatures & 0x1000000000L) != 0L) {
            customOption.append("6");
        }
        if ((queryFeatures & 0x2000000000L) != 0L) {
            customOption.append("7");
        }
        if ((queryFeatures & 0x4000000000L) != 0L) {
            customOption.append("8");
        }
        return customOption.toString();
    }

    private void mpSetNormalLabelSize() {
        this.setTextSize(this.mpLabelSize);
    }

    private void mpSetReducedLabelSize() {
        this.setTextSize((this.mpLabelSize * 5 + 1) / 6);
    }

    private void mpSetSmallLabelSize() {
        this.setTextSize((this.mpLabelSize * 2 + 1) / 3);
    }

    private double mpGetFreeSpaceAngle(int atom) {
        double[] angle = new double[this.mMol.getAllConnAtoms(atom)];
        for (int i = 0; i < this.mMol.getAllConnAtoms(atom); ++i) {
            angle[i] = this.mMol.getBondAngle(atom, this.mMol.getConnAtom(atom, i));
        }
        Arrays.sort(angle);
        double maxMean = this.mpGetMeanAngle(angle, 0);
        double maxVal = this.mpGetAngleESRScore(angle, 0, maxMean);
        for (int i = 1; i < angle.length; ++i) {
            double mean = this.mpGetMeanAngle(angle, i);
            double val = this.mpGetAngleESRScore(angle, i, mean);
            if (!(maxVal < val)) continue;
            maxVal = val;
            maxMean = mean;
        }
        return maxMean;
    }

    private double mpGetAngleESRScore(double[] angleList, int index, double meanAngle) {
        double score;
        double d = score = index == 0 ? Math.PI * 2 + angleList[0] - angleList[angleList.length - 1] : angleList[index] - angleList[index - 1];
        score = meanAngle > -2.0943951023931953 && meanAngle < 1.0471975511965976 ? (score -= 2.0 * Math.cos(meanAngle + 0.5235987755982988)) : (score -= 0.5 * Math.cos(meanAngle + 0.5235987755982988));
        return score;
    }

    private double mpGetMeanAngle(double[] angle, int index) {
        if (index > 0) {
            return (angle[index] + angle[index - 1]) / 2.0;
        }
        double mean = Math.PI + (angle[0] + angle[angle.length - 1]) / 2.0;
        return mean > Math.PI ? mean - Math.PI * 2 : mean;
    }

    private String append(String a, String b) {
        return a == null ? b : (b == null ? a : a + "," + b);
    }

    private void mpDrawString(double x, double y, String str, boolean withTabu) {
        if (withTabu) {
            double strWidth = this.getStringWidth(str);
            double xdiff = strWidth / 2.0 + (double)(this.getTextSize() / 8);
            double ydiff = this.getTextSize() / 2;
            if (str == "+" || str == "-") {
                ydiff = ydiff * 2.0 / 3.0;
            }
            this.mpTabuZone.add(new GenericRectangle(x - xdiff, y - ydiff, 2.0 * xdiff, 2.0 * ydiff));
        }
        if (!this.mIsValidatingView) {
            this.drawString(str, x, y);
        }
    }

    private void mpDrawDot(double x, double y, int atm) {
        this.mpTabuZone.add(new GenericRectangle(x - this.mpDotDiameter, y - this.mpDotDiameter, 2.0 * this.mpDotDiameter, 2.0 * this.mpDotDiameter));
        if (!this.mIsValidatingView) {
            this.mpDot.add(new DepictorDot(x, y, this.isHighlightedAtom(atm) ? -3 : this.mAtomColor[atm]));
        }
    }

    private void mpDrawAllDots() {
        for (DepictorDot dot : this.mpDot) {
            this.setColor_(dot.color);
            this.drawDot(dot.x, dot.y);
        }
        this.setColor_(this.mStandardForegroundColor);
    }

    private boolean mpAlleneCenter(int atm) {
        if (this.mMol.getConnAtoms(atm) != 2) {
            return false;
        }
        for (int i = 0; i < 2; ++i) {
            if (this.mMol.getConnBondOrder(atm, i) == 2) continue;
            return false;
        }
        return true;
    }

    private void mpDrawBondQueryFeatures() {
        boolean textSizeChanged = false;
        for (int bond = 0; bond < this.mMol.getBonds(); ++bond) {
            String label = null;
            if (this.mMol.isBondBridge(bond)) {
                int maxAtoms;
                int minAtoms = this.mMol.getBondBridgeMinSize(bond);
                label = minAtoms == (maxAtoms = this.mMol.getBondBridgeMaxSize(bond)) ? "[" + minAtoms + "]" : "[" + minAtoms + ":" + maxAtoms + "]";
            } else if ((this.mMol.getBondQueryFeatures(bond) & 0x180000) != 0) {
                label = (this.mMol.getBondQueryFeatures(bond) & 0x180000) == 524288 ? "a" : ((this.mMol.getBondQueryFeatures(bond) & 0x60) == 64 ? "r!a" : "!a");
            } else if ((this.mMol.getBondQueryFeatures(bond) & 0x60) != 0) {
                label = (this.mMol.getBondQueryFeatures(bond) & 0x60) == 64 ? "r" : "!r";
            }
            int ringSize = (this.mMol.getBondQueryFeatures(bond) & 0x38000) >> 15;
            if (ringSize != 0) {
                label = (label == null ? "" : label) + ringSize;
            }
            if (label == null) continue;
            int atom1 = this.mMol.getBondAtom(0, bond);
            int atom2 = this.mMol.getBondAtom(1, bond);
            if (!textSizeChanged) {
                this.mpSetSmallLabelSize();
                textSizeChanged = true;
            }
            double x = (this.getAtomX(atom1) + this.getAtomX(atom2)) / 2.0;
            double y = (this.getAtomY(atom1) + this.getAtomY(atom2)) / 2.0;
            double dx = this.getAtomX(atom2) - this.getAtomX(atom1);
            double dy = this.getAtomY(atom2) - this.getAtomY(atom1);
            double d = Math.sqrt(dx * dx + dy * dy);
            double hw = 0.6 * this.getStringWidth(label);
            double hh = 0.55 * (double)this.getTextSize();
            if (d == 0.0) continue;
            if (dx > 0.0) {
                this.mpDrawString(x + hw * dy / d, y - hh * dx / d, label, true);
                continue;
            }
            this.mpDrawString(x - hw * dy / d, y + hh * dx / d, label, true);
        }
        if (textSizeChanged) {
            this.mpSetNormalLabelSize();
        }
    }

    private void drawLine(DepictorLine theLine, int atom1, int atom2) {
        if (this.mMol.isBondForegroundHilited(this.mMol.getBond(atom1, atom2))) {
            this.setColor_(-3);
            this.drawBlackLine(theLine);
            this.setColor_(this.mStandardForegroundColor);
        } else if (this.mAtomColor[atom1] != this.mAtomColor[atom2]) {
            this.drawColorLine(theLine, atom1, atom2);
        } else if (this.mAtomColor[atom1] != 0) {
            this.setColor_(this.mAtomColor[atom1]);
            this.drawBlackLine(theLine);
            this.setColor_(this.mStandardForegroundColor);
        } else {
            this.drawBlackLine(theLine);
        }
    }

    private void drawColorLine(DepictorLine theLine, int atm1, int atm2) {
        DepictorLine line1 = new DepictorLine();
        DepictorLine line2 = new DepictorLine();
        line1.x1 = theLine.x1;
        line1.y1 = theLine.y1;
        line1.x2 = (theLine.x1 + theLine.x2) / 2.0;
        line1.y2 = (theLine.y1 + theLine.y2) / 2.0;
        line2.x1 = line1.x2;
        line2.y1 = line1.y2;
        line2.x2 = theLine.x2;
        line2.y2 = theLine.y2;
        if (this.mpProperLine(line1)) {
            this.setColor_(this.mAtomColor[atm1]);
            this.drawBlackLine(line1);
        }
        if (this.mpProperLine(line2)) {
            this.setColor_(this.mAtomColor[atm2]);
            this.drawBlackLine(line2);
        }
        this.setColor_(this.mStandardForegroundColor);
    }

    private void drawDashedLine(DepictorLine theLine, int atom1, int atom2) {
        int color2;
        int color1;
        double xinc = (theLine.x2 - theLine.x1) / 10.0;
        double yinc = (theLine.y2 - theLine.y1) / 10.0;
        DepictorLine aLine = new DepictorLine();
        if (this.mMol.isBondForegroundHilited(this.mMol.getBond(atom1, atom2))) {
            color1 = -3;
            color2 = -3;
        } else {
            color1 = this.mAtomColor[atom1];
            color2 = this.mAtomColor[atom2];
        }
        this.setColor_(color1);
        aLine.x1 = theLine.x1;
        aLine.y1 = theLine.y1;
        aLine.x2 = theLine.x1 + xinc * 2.0;
        aLine.y2 = theLine.y1 + yinc * 2.0;
        this.drawBlackLine(aLine);
        aLine.x1 = theLine.x1 + xinc * 4.0;
        aLine.y1 = theLine.y1 + yinc * 4.0;
        aLine.x2 = theLine.x1 + xinc * 5.0;
        aLine.y2 = theLine.y1 + yinc * 5.0;
        this.drawBlackLine(aLine);
        this.setColor_(color2);
        aLine.x1 = theLine.x1 + xinc * 5.0;
        aLine.y1 = theLine.y1 + yinc * 5.0;
        aLine.x2 = theLine.x1 + xinc * 6.0;
        aLine.y2 = theLine.y1 + yinc * 6.0;
        this.drawBlackLine(aLine);
        aLine.x1 = theLine.x1 + xinc * 8.0;
        aLine.y1 = theLine.y1 + yinc * 8.0;
        aLine.x2 = theLine.x2;
        aLine.y2 = theLine.y2;
        this.drawBlackLine(aLine);
        this.setColor_(this.mStandardForegroundColor);
    }

    private void drawShortDashedLine(DepictorLine theLine, int atom1, int atom2) {
        int i;
        int color2;
        int color1;
        double xdif = theLine.x2 - theLine.x1;
        double ydif = theLine.y2 - theLine.y1;
        double length = Math.sqrt(xdif * xdif + ydif * ydif);
        int points = 2 * (int)Math.round(length / (4.0 * this.mpLineWidth));
        double xinc = xdif / (double)(points - 1);
        double yinc = ydif / (double)(points - 1);
        if (this.mMol.isBondForegroundHilited(this.mMol.getBond(atom1, atom2))) {
            color1 = -3;
            color2 = -3;
        } else {
            color1 = this.mAtomColor[atom1];
            color2 = this.mAtomColor[atom2];
        }
        double x = theLine.x1 - this.mpLineWidth / 2.0;
        double y = theLine.y1 - this.mpLineWidth / 2.0;
        this.setColor_(color1);
        for (i = 0; i < points / 2; ++i) {
            this.fillCircle(x, y, this.mpLineWidth);
            x += xinc;
            y += yinc;
        }
        this.setColor_(color2);
        for (i = 0; i < points / 2; ++i) {
            this.fillCircle(x, y, this.mpLineWidth);
            x += xinc;
            y += yinc;
        }
        this.setColor_(this.mStandardForegroundColor);
    }

    private void drawWedge(DepictorLine theWedge, int atom1, int atom2) {
        int color2;
        int color1;
        double xdiff = (theWedge.y1 - theWedge.y2) / 9.0;
        double ydiff = (theWedge.x2 - theWedge.x1) / 9.0;
        double xe1 = theWedge.x2 + xdiff;
        double ye1 = theWedge.y2 + ydiff;
        double xe2 = theWedge.x2 - xdiff;
        double ye2 = theWedge.y2 - ydiff;
        double xm1 = (theWedge.x1 + xe1) / 2.0;
        double ym1 = (theWedge.y1 + ye1) / 2.0;
        double xm2 = (theWedge.x1 + xe2) / 2.0;
        double ym2 = (theWedge.y1 + ye2) / 2.0;
        GenericPolygon p1 = new GenericPolygon(3);
        GenericPolygon p2 = new GenericPolygon(4);
        p1.addPoint(theWedge.x1, theWedge.y1);
        p1.addPoint(xm1, ym1);
        p1.addPoint(xm2, ym2);
        p2.addPoint(xm2, ym2);
        p2.addPoint(xm1, ym1);
        p2.addPoint(xe1, ye1);
        p2.addPoint(xe2, ye2);
        if (this.mMol.isBondForegroundHilited(this.mMol.getBond(atom1, atom2))) {
            color1 = -3;
            color2 = -3;
        } else {
            color1 = this.mAtomColor[atom1];
            color2 = this.mAtomColor[atom2];
            if (this.mMol.getMoleculeColor() != 1) {
                color2 = this.getESRColor(atom1);
                if (color1 == this.mMol.getAtomColor(atom1)) {
                    color1 = color2;
                }
            }
        }
        this.setColor_(color1);
        this.drawPolygon(p1);
        this.setColor_(color2);
        this.drawPolygon(p2);
        this.setColor_(this.mStandardForegroundColor);
    }

    protected void drawDot(double x, double y) {
        this.fillCircle(x - this.mpDotDiameter / 2.0, y - this.mpDotDiameter / 2.0, this.mpDotDiameter);
    }

    private void setRGBColor(int rgb) {
        if (this.mOverruleForeground != 0) {
            if (this.mCurrentColor != -4) {
                this.mCurrentColor = -4;
                this.setRGB(this.mOverruleForeground);
            }
            return;
        }
        this.mCurrentColor = -5;
        this.mRGBColor = rgb;
        this.setRGB(rgb);
    }

    public void setColor_(int theColor) {
        if (this.mIsValidatingView) {
            return;
        }
        if (theColor == -10) {
            this.mCurrentColor = -999;
            theColor = this.mStandardForegroundColor;
        }
        if (theColor != -2 && theColor != -7 && this.mOverruleForeground != 0) {
            theColor = -4;
        }
        if (theColor == this.mCurrentColor) {
            return;
        }
        if (this.mCurrentColor == -8 && theColor != -9) {
            return;
        }
        if (theColor == -8) {
            this.mPreviousColor = this.mCurrentColor;
        }
        if (theColor == -9) {
            theColor = this.mPreviousColor;
        }
        this.mCurrentColor = theColor;
        switch (theColor) {
            case 0: {
                this.setRGB(this.mCustomForeground == 0 ? -16777216 : this.mCustomForeground);
                break;
            }
            case -6: {
                this.setRGB(this.mCustomForeground);
                break;
            }
            case -4: {
                this.setRGB(this.mOverruleForeground);
                break;
            }
            case -2: {
                this.setRGB(this.mBondBGHiliteColor);
                break;
            }
            case -3: {
                this.setRGB(this.mBondFGHiliteColor);
                break;
            }
            case -7: {
                this.setRGB(this.mExcludeGroupBGColor);
                break;
            }
            case -8: {
                this.setRGB(this.mExcludeGroupFGColor);
                break;
            }
            case -5: {
                this.setRGB(this.mRGBColor);
                break;
            }
            case 64: {
                this.setRGB(-14655233);
                break;
            }
            case 128: {
                this.setRGB(-65536);
                break;
            }
            case 256: {
                this.setRGB(-4194049);
                break;
            }
            case 192: {
                this.setRGB(-16711936);
                break;
            }
            case 320: {
                this.setRGB(-24576);
                break;
            }
            case 384: {
                this.setRGB(-16744448);
                break;
            }
            case 448: {
                this.setRGB(-6291456);
                break;
            }
            case 1: {
                this.setRGB(-8355712);
                break;
            }
            default: {
                this.setRGB(-16777216);
            }
        }
    }

    private int getESRTypeToDisplayAt(int atom) {
        int bond;
        int type = -1;
        int group = -1;
        if ((this.mDisplayMode & 0x80) != 0) {
            return type;
        }
        if (this.mMol.isAtomStereoCenter(atom)) {
            type = this.mMol.getAtomESRType(atom);
            group = this.mMol.getAtomESRGroup(atom);
        }
        if ((bond = this.mMol.findBINAPChiralityBond(atom)) != -1) {
            type = this.mMol.getBondESRType(bond);
            group = this.mMol.getBondESRGroup(bond);
        }
        if (type != -1 && type != 0) {
            type |= group << 8;
        }
        return type;
    }

    private int getESRColor(int atom) {
        int alleneCenter;
        if ((this.mDisplayMode & 0x2080) != 0) {
            return this.mAtomColor[atom];
        }
        int esrInfo = this.getESRTypeToDisplayAt(atom);
        if (esrInfo == -1 && (alleneCenter = this.mMol.findAlleneCenterAtom(atom)) != -1) {
            atom = alleneCenter;
            esrInfo = this.getESRTypeToDisplayAt(atom);
        }
        if (esrInfo == -1) {
            return this.mAtomColor[atom];
        }
        switch (esrInfo & 0xFF) {
            case 1: {
                return 384;
            }
            case 2: {
                return 64;
            }
        }
        return 448;
    }

    protected abstract void drawBlackLine(DepictorLine var1);

    protected abstract void drawDottedLine(DepictorLine var1);

    protected abstract void drawPolygon(GenericPolygon var1);

    protected abstract void drawString(String var1, double var2, double var4);

    protected abstract void fillCircle(double var1, double var3, double var5);

    protected abstract double getLineWidth();

    protected abstract double getStringWidth(String var1);

    protected abstract int getTextSize();

    protected abstract void setTextSize(int var1);

    protected abstract void setLineWidth(double var1);

    protected abstract void setRGB(int var1);

    public static class DepictorLine {
        public double x1;
        public double y1;
        public double x2;
        public double y2;

        public DepictorLine(double x1, double y1, double x2, double y2) {
            this.x1 = x1;
            this.y1 = y1;
            this.x2 = x2;
            this.y2 = y2;
        }

        public DepictorLine() {
        }
    }

    public static class DepictorDot {
        public double x;
        public double y;
        public int color;

        DepictorDot(double x, double y, int color) {
            this.x = x;
            this.y = y;
            this.color = color;
        }
    }
}

