package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.AssertJBooleanRules}.
 */
@SuppressWarnings("all")
public class AssertJBooleanRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public AssertJBooleanRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "`AssertJBooleanRules` Refaster recipes";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.AssertJBooleanRules`. [Source](https://error-prone.picnic.tech/refasterrules/AssertJBooleanRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new AbstractBooleanAssertIsEqualToRecipe(),
                new AbstractBooleanAssertIsNotEqualToRecipe(),
                new AbstractBooleanAssertIsTrueRecipe(),
                new AssertThatBooleanIsTrueRecipe(),
                new AbstractBooleanAssertIsFalseRecipe(),
                new AssertThatBooleanIsFalseRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJBooleanRules.AbstractBooleanAssertIsEqualTo}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AbstractBooleanAssertIsEqualToRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AbstractBooleanAssertIsEqualToRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJBooleanRules.AbstractBooleanAssertIsEqualTo`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AbstractBooleanAssertIsEqualTo {\n    \n    @BeforeTemplate\n    AbstractBooleanAssert<?> before(AbstractBooleanAssert<?> boolAssert, boolean other) {\n        return boolAssert.isNotEqualTo(!other);\n    }\n    \n    @AfterTemplate\n    AbstractBooleanAssert<?> after(AbstractBooleanAssert<?> boolAssert, boolean other) {\n        return boolAssert.isEqualTo(other);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isNotEqualTo(!#{other:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isEqualTo(#{other:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractBooleanAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isNotEqualTo(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJBooleanRules.AbstractBooleanAssertIsNotEqualTo}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AbstractBooleanAssertIsNotEqualToRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AbstractBooleanAssertIsNotEqualToRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJBooleanRules.AbstractBooleanAssertIsNotEqualTo`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AbstractBooleanAssertIsNotEqualTo {\n    \n    @BeforeTemplate\n    AbstractBooleanAssert<?> before(AbstractBooleanAssert<?> boolAssert, boolean other) {\n        return boolAssert.isEqualTo(!other);\n    }\n    \n    @AfterTemplate\n    AbstractBooleanAssert<?> after(AbstractBooleanAssert<?> boolAssert, boolean other) {\n        return boolAssert.isNotEqualTo(other);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isEqualTo(!#{other:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isNotEqualTo(#{other:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractBooleanAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isEqualTo(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJBooleanRules.AbstractBooleanAssertIsTrue}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AbstractBooleanAssertIsTrueRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AbstractBooleanAssertIsTrueRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJBooleanRules.AbstractBooleanAssertIsTrue`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AbstractBooleanAssertIsTrue {\n    \n    @BeforeTemplate\n    AbstractBooleanAssert<?> before(AbstractBooleanAssert<?> boolAssert) {\n        return Refaster.anyOf(boolAssert.isEqualTo(true), boolAssert.isEqualTo(Boolean.TRUE), boolAssert.isNotEqualTo(false), boolAssert.isNotEqualTo(Boolean.FALSE));\n    }\n    \n    @AfterTemplate\n    AbstractBooleanAssert<?> after(AbstractBooleanAssert<?> boolAssert) {\n        return boolAssert.isTrue();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isEqualTo(true)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isEqualTo(Boolean.TRUE)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isNotEqualTo(false)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isNotEqualTo(Boolean.FALSE)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractBooleanAssert", true),
                        Preconditions.or(
                            new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isEqualTo(..)"),
                            new UsesMethod<>("org.assertj.core.api.AbstractAssert isEqualTo(..)"),
                            new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isNotEqualTo(..)"),
                            new UsesMethod<>("org.assertj.core.api.AbstractAssert isNotEqualTo(..)")
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJBooleanRules.AssertThatBooleanIsTrue}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatBooleanIsTrueRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatBooleanIsTrueRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJBooleanRules.AssertThatBooleanIsTrue`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatBooleanIsTrue {\n    \n    @BeforeTemplate\n    AbstractBooleanAssert<?> before(boolean b) {\n        return assertThat(!b).isFalse();\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractBooleanAssert<?> after(boolean b) {\n        return assertThat(b).isTrue();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(!#{b:any(boolean)}).isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{b:any(boolean)}).isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractBooleanAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isFalse(..)"),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJBooleanRules.AbstractBooleanAssertIsFalse}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AbstractBooleanAssertIsFalseRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AbstractBooleanAssertIsFalseRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJBooleanRules.AbstractBooleanAssertIsFalse`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AbstractBooleanAssertIsFalse {\n    \n    @BeforeTemplate\n    AbstractBooleanAssert<?> before(AbstractBooleanAssert<?> boolAssert) {\n        return Refaster.anyOf(boolAssert.isEqualTo(false), boolAssert.isEqualTo(Boolean.FALSE), boolAssert.isNotEqualTo(true), boolAssert.isNotEqualTo(Boolean.TRUE));\n    }\n    \n    @AfterTemplate\n    AbstractBooleanAssert<?> after(AbstractBooleanAssert<?> boolAssert) {\n        return boolAssert.isFalse();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isEqualTo(false)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isEqualTo(Boolean.FALSE)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isNotEqualTo(true)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isNotEqualTo(Boolean.TRUE)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{boolAssert:any(org.assertj.core.api.AbstractBooleanAssert<?>)}.isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractBooleanAssert", true),
                        Preconditions.or(
                            new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isEqualTo(..)"),
                            new UsesMethod<>("org.assertj.core.api.AbstractAssert isEqualTo(..)"),
                            new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isNotEqualTo(..)"),
                            new UsesMethod<>("org.assertj.core.api.AbstractAssert isNotEqualTo(..)")
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJBooleanRules.AssertThatBooleanIsFalse}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatBooleanIsFalseRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatBooleanIsFalseRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJBooleanRules.AssertThatBooleanIsFalse`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatBooleanIsFalse {\n    \n    @BeforeTemplate\n    AbstractBooleanAssert<?> before(boolean b) {\n        return assertThat(!b).isTrue();\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractBooleanAssert<?> after(boolean b) {\n        return assertThat(b).isFalse();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(!#{b:any(boolean)}).isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{b:any(boolean)}).isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractBooleanAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isTrue(..)"),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
