package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.AssertJStringRules}.
 */
@SuppressWarnings("all")
public class AssertJStringRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public AssertJStringRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "`AssertJStringRules` Refaster recipes";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.AssertJStringRules`. [Source](https://error-prone.picnic.tech/refasterrules/AssertJStringRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new AbstractStringAssertStringIsEmptyRecipe(),
                new AssertThatStringIsEmptyRecipe(),
                new AbstractStringAssertStringIsNotEmptyRecipe(),
                new AssertThatStringIsNotEmptyRecipe(),
                new AssertThatMatchesRecipe(),
                new AssertThatDoesNotMatchRecipe(),
                new AssertThatPathContentRecipe(),
                new AssertThatPathContentUtf8Recipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AbstractStringAssertStringIsEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AbstractStringAssertStringIsEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AbstractStringAssertStringIsEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AbstractStringAssertStringIsEmpty`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AbstractStringAssertStringIsEmpty {\n    \n    @BeforeTemplate\n    void before(AbstractStringAssert<?> stringAssert) {\n        stringAssert.isEqualTo(\"\");\n    }\n    \n    @AfterTemplate\n    void after(AbstractStringAssert<?> stringAssert) {\n        stringAssert.isEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stringAssert:any(org.assertj.core.api.AbstractStringAssert<?>)}.isEqualTo(\"\")")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stringAssert:any(org.assertj.core.api.AbstractStringAssert<?>)}.isEmpty()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractStringAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractStringAssert isEqualTo(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AssertThatStringIsEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatStringIsEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatStringIsEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AssertThatStringIsEmpty`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatStringIsEmpty {\n    \n    @BeforeTemplate\n    void before(String string) {\n        assertThat(string.isEmpty()).isTrue();\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(String string) {\n        assertThat(string).isEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}.isEmpty()).isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}).isEmpty()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isTrue(..)"),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)"),
                        new UsesMethod<>("java.lang.String isEmpty(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AbstractStringAssertStringIsNotEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AbstractStringAssertStringIsNotEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AbstractStringAssertStringIsNotEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AbstractStringAssertStringIsNotEmpty`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AbstractStringAssertStringIsNotEmpty {\n    \n    @BeforeTemplate\n    AbstractStringAssert<?> before(AbstractStringAssert<?> stringAssert) {\n        return stringAssert.isNotEqualTo(\"\");\n    }\n    \n    @AfterTemplate\n    AbstractStringAssert<?> after(AbstractStringAssert<?> stringAssert) {\n        return stringAssert.isNotEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stringAssert:any(org.assertj.core.api.AbstractStringAssert<?>)}.isNotEqualTo(\"\")")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stringAssert:any(org.assertj.core.api.AbstractStringAssert<?>)}.isNotEmpty()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractStringAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractAssert isNotEqualTo(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AssertThatStringIsNotEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatStringIsNotEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatStringIsNotEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AssertThatStringIsNotEmpty`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatStringIsNotEmpty {\n    \n    @BeforeTemplate\n    AbstractAssert<?, ?> before(String string) {\n        return assertThat(string.isEmpty()).isFalse();\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractAssert<?, ?> after(String string) {\n        return assertThat(string).isNotEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}.isEmpty()).isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}).isNotEmpty()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isFalse(..)"),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)"),
                        new UsesMethod<>("java.lang.String isEmpty(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AssertThatMatches}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatMatchesRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatMatchesRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AssertThatMatches`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatMatches {\n    \n    @BeforeTemplate\n    AbstractAssert<?, ?> before(String string, String regex) {\n        return assertThat(string.matches(regex)).isTrue();\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractAssert<?, ?> after(String string, String regex) {\n        return assertThat(string).matches(regex);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}.matches(#{regex:any(java.lang.String)})).isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}).matches(#{regex:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isTrue(..)"),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)"),
                        new UsesMethod<>("java.lang.String matches(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AssertThatDoesNotMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatDoesNotMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatDoesNotMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AssertThatDoesNotMatch`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatDoesNotMatch {\n    \n    @BeforeTemplate\n    AbstractAssert<?, ?> before(String string, String regex) {\n        return assertThat(string.matches(regex)).isFalse();\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractAssert<?, ?> after(String string, String regex) {\n        return assertThat(string).doesNotMatch(regex);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}.matches(#{regex:any(java.lang.String)})).isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{string:any(java.lang.String)}).doesNotMatch(#{regex:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.assertj.core.api.AbstractAssert", true),
                        new UsesMethod<>("org.assertj.core.api.AbstractBooleanAssert isFalse(..)"),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)"),
                        new UsesMethod<>("java.lang.String matches(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AssertThatPathContent}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatPathContentRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatPathContentRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AssertThatPathContent`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatPathContent {\n    \n    @BeforeTemplate\n    AbstractStringAssert<?> before(Path path, Charset charset) throws IOException {\n        return assertThat(Files.readString(path, charset));\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractStringAssert<?> after(Path path, Charset charset) {\n        return assertThat(path).content(charset);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(java.nio.file.Files.readString(#{path:any(java.nio.file.Path)}, #{charset:any(java.nio.charset.Charset)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{path:any(java.nio.file.Path)}).content(#{charset:any(java.nio.charset.Charset)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.nio.file.Files");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.nio.file.Files", true),
                        new UsesType<>("org.assertj.core.api.AbstractStringAssert", true),
                        new UsesType<>("java.nio.file.Path", true),
                        new UsesType<>("java.nio.charset.Charset", true),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)"),
                        new UsesMethod<>("java.nio.file.Files readString(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code AssertJStringRules.AssertThatPathContentUtf8}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThatPathContentUtf8Recipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThatPathContentUtf8Recipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `AssertJStringRules.AssertThatPathContentUtf8`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThatPathContentUtf8 {\n    \n    @BeforeTemplate\n    AbstractStringAssert<?> before(Path path) throws IOException {\n        return assertThat(Files.readString(path));\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    AbstractStringAssert<?> after(Path path) {\n        return assertThat(path).content(UTF_8);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(java.nio.file.Files.readString(#{path:any(java.nio.file.Path)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{path:any(java.nio.file.Path)}).content(java.nio.charset.StandardCharsets.UTF_8)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.nio.file.Files");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.nio.file.Files", true),
                        new UsesType<>("org.assertj.core.api.AbstractStringAssert", true),
                        new UsesType<>("java.nio.file.Path", true),
                        new UsesMethod<>("org.assertj.core.api.Assertions assertThat(..)"),
                        new UsesMethod<>("java.nio.file.Files readString(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
