package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.DoubleStreamRules}.
 */
@SuppressWarnings("all")
public class DoubleStreamRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public DoubleStreamRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with `DoubleStream`s";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.DoubleStreamRules`. [Source](https://error-prone.picnic.tech/refasterrules/DoubleStreamRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new ConcatOneDoubleStreamRecipe(),
                new ConcatTwoDoubleStreamsRecipe(),
                new DoubleStreamFilterSortedRecipe(),
                new DoubleStreamIsEmptyRecipe(),
                new DoubleStreamIsNotEmptyRecipe(),
                new DoubleStreamMinRecipe(),
                new DoubleStreamNoneMatchRecipe(),
                new DoubleStreamAnyMatchRecipe(),
                new DoubleStreamAllMatchRecipe(),
                new DoubleStreamTakeWhileRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.ConcatOneDoubleStream}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class ConcatOneDoubleStreamRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public ConcatOneDoubleStreamRecipe() {}

        @Override
        public String getDisplayName() {
            return "Don't unnecessarily call `Streams#concat(DoubleStream...)`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class ConcatOneDoubleStream {\n    \n    @BeforeTemplate\n    DoubleStream before(DoubleStream stream) {\n        return Streams.concat(stream);\n    }\n    \n    @AfterTemplate\n    @CanIgnoreReturnValue\n    DoubleStream after(DoubleStream stream) {\n        return stream;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.collect.Streams.concat(#{stream:any(java.util.stream.DoubleStream)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.collect.Streams");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.collect.Streams", true),
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesMethod<>("com.google.common.collect.Streams concat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.ConcatTwoDoubleStreams}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class ConcatTwoDoubleStreamsRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public ConcatTwoDoubleStreamsRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `DoubleStream#concat(DoubleStream, DoubleStream)` over the Guava alternative";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class ConcatTwoDoubleStreams {\n    \n    @BeforeTemplate\n    DoubleStream before(DoubleStream s1, DoubleStream s2) {\n        return Streams.concat(s1, s2);\n    }\n    \n    @AfterTemplate\n    DoubleStream after(DoubleStream s1, DoubleStream s2) {\n        return DoubleStream.concat(s1, s2);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.collect.Streams.concat(#{s1:any(java.util.stream.DoubleStream)}, #{s2:any(java.util.stream.DoubleStream)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.util.stream.DoubleStream.concat(#{s1:any(java.util.stream.DoubleStream)}, #{s2:any(java.util.stream.DoubleStream)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.collect.Streams");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.collect.Streams", true),
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesMethod<>("com.google.common.collect.Streams concat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamFilterSorted}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamFilterSortedRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamFilterSortedRecipe() {}

        @Override
        public String getDisplayName() {
            return "Apply `DoubleStream#filter(DoublePredicate)` before `DoubleStream#sorted()` to reduce the number of elements to sort";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamFilterSorted {\n    \n    @BeforeTemplate\n    DoubleStream before(DoubleStream stream, DoublePredicate predicate) {\n        return stream.sorted().filter(predicate);\n    }\n    \n    @AfterTemplate\n    DoubleStream after(DoubleStream stream, DoublePredicate predicate) {\n        return stream.filter(predicate).sorted();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.sorted().filter(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.filter(#{predicate:any(java.util.function.DoublePredicate)}).sorted()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesType<>("java.util.function.DoublePredicate", true),
                        new UsesMethod<>("java.util.stream.DoubleStream filter(..)"),
                        new UsesMethod<>("java.util.stream.DoubleStream sorted(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamIsEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamIsEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamIsEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "In order to test whether a stream has any element, simply try to find one";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamIsEmpty {\n    \n    @BeforeTemplate\n    boolean before(DoubleStream stream) {\n        return Refaster.anyOf(stream.count() == 0, stream.count() <= 0, stream.count() < 1, stream.findFirst().isEmpty());\n    }\n    \n    @AfterTemplate\n    boolean after(DoubleStream stream) {\n        return stream.findAny().isEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.count() == 0")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.count() <= 0")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.count() < 1")
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.findFirst().isEmpty()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.findAny().isEmpty()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.DoubleStream count(..)"),
                            new UsesMethod<>("java.util.stream.DoubleStream count(..)"),
                            new UsesMethod<>("java.util.stream.DoubleStream count(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalDouble isEmpty(..)"),
                                new UsesMethod<>("java.util.stream.DoubleStream findFirst(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamIsNotEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamIsNotEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamIsNotEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "In order to test whether a stream has any element, simply try to find one";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamIsNotEmpty {\n    \n    @BeforeTemplate\n    boolean before(DoubleStream stream) {\n        return Refaster.anyOf(stream.count() != 0, stream.count() > 0, stream.count() >= 1, stream.findFirst().isPresent());\n    }\n    \n    @AfterTemplate\n    boolean after(DoubleStream stream) {\n        return stream.findAny().isPresent();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.count() != 0")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.count() > 0")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.count() >= 1")
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.findFirst().isPresent()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.findAny().isPresent()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.DoubleStream count(..)"),
                            new UsesMethod<>("java.util.stream.DoubleStream count(..)"),
                            new UsesMethod<>("java.util.stream.DoubleStream count(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalDouble isPresent(..)"),
                                new UsesMethod<>("java.util.stream.DoubleStream findFirst(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamMin}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamMinRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamMinRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `DoubleStreamRules.DoubleStreamMin`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamMin {\n    \n    @BeforeTemplate\n    OptionalDouble before(DoubleStream stream) {\n        return stream.sorted().findFirst();\n    }\n    \n    @AfterTemplate\n    OptionalDouble after(DoubleStream stream) {\n        return stream.min();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.sorted().findFirst()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.min()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.OptionalDouble", true),
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesMethod<>("java.util.stream.DoubleStream findFirst(..)"),
                        new UsesMethod<>("java.util.stream.DoubleStream sorted(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamNoneMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamNoneMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamNoneMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `DoubleStream#noneMatch(DoublePredicate)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamNoneMatch {\n    \n    @BeforeTemplate\n    boolean before(DoubleStream stream, DoublePredicate predicate) {\n        return Refaster.anyOf(!stream.anyMatch(predicate), stream.allMatch(predicate.negate()), stream.filter(predicate).findAny().isEmpty());\n    }\n    \n    @AfterTemplate\n    boolean after(DoubleStream stream, DoublePredicate predicate) {\n        return stream.noneMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!#{stream:any(java.util.stream.DoubleStream)}.anyMatch(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.allMatch(#{predicate:any(java.util.function.DoublePredicate)}.negate())")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.filter(#{predicate:any(java.util.function.DoublePredicate)}).findAny().isEmpty()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.noneMatch(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesType<>("java.util.function.DoublePredicate", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.DoubleStream anyMatch(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.stream.DoubleStream allMatch(..)"),
                                new UsesMethod<>("java.util.function.DoublePredicate negate(..)")
                            ),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalDouble isEmpty(..)"),
                                new UsesMethod<>("java.util.stream.DoubleStream findAny(..)"),
                                new UsesMethod<>("java.util.stream.DoubleStream filter(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamAnyMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamAnyMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamAnyMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `DoubleStream#anyMatch(DoublePredicate)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamAnyMatch {\n    \n    @BeforeTemplate\n    @SuppressWarnings(value = \"java:S4034\")\n    boolean before(DoubleStream stream, DoublePredicate predicate) {\n        return Refaster.anyOf(!stream.noneMatch(predicate), stream.filter(predicate).findAny().isPresent());\n    }\n    \n    @AfterTemplate\n    boolean after(DoubleStream stream, DoublePredicate predicate) {\n        return stream.anyMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!#{stream:any(java.util.stream.DoubleStream)}.noneMatch(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.filter(#{predicate:any(java.util.function.DoublePredicate)}).findAny().isPresent()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.anyMatch(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesType<>("java.util.function.DoublePredicate", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.DoubleStream noneMatch(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalDouble isPresent(..)"),
                                new UsesMethod<>("java.util.stream.DoubleStream findAny(..)"),
                                new UsesMethod<>("java.util.stream.DoubleStream filter(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamAllMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamAllMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamAllMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `DoubleStreamRules.DoubleStreamAllMatch`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamAllMatch {\n    \n    @BeforeTemplate\n    boolean before(DoubleStream stream, DoublePredicate predicate) {\n        return stream.noneMatch(predicate.negate());\n    }\n    \n    @AfterTemplate\n    boolean after(DoubleStream stream, DoublePredicate predicate) {\n        return stream.allMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.noneMatch(#{predicate:any(java.util.function.DoublePredicate)}.negate())")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.allMatch(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesType<>("java.util.function.DoublePredicate", true),
                        new UsesMethod<>("java.util.stream.DoubleStream noneMatch(..)"),
                        new UsesMethod<>("java.util.function.DoublePredicate negate(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code DoubleStreamRules.DoubleStreamTakeWhile}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleStreamTakeWhileRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleStreamTakeWhileRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `DoubleStreamRules.DoubleStreamTakeWhile`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleStreamTakeWhile {\n    \n    @BeforeTemplate\n    DoubleStream before(DoubleStream stream, DoublePredicate predicate) {\n        return stream.takeWhile(predicate).filter(predicate);\n    }\n    \n    @AfterTemplate\n    DoubleStream after(DoubleStream stream, DoublePredicate predicate) {\n        return stream.takeWhile(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.takeWhile(#{predicate:any(java.util.function.DoublePredicate)}).filter(#{predicate})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.DoubleStream)}.takeWhile(#{predicate:any(java.util.function.DoublePredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.DoubleStream", true),
                        new UsesType<>("java.util.function.DoublePredicate", true),
                        new UsesMethod<>("java.util.stream.DoubleStream filter(..)"),
                        new UsesMethod<>("java.util.stream.DoubleStream takeWhile(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
