package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.EqualityRules}.
 */
@SuppressWarnings("all")
public class EqualityRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public EqualityRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with (in)equalities";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.EqualityRules`. [Source](https://error-prone.picnic.tech/refasterrules/EqualityRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new DoubleNegationRecipe(),
                new NegationRecipe(),
                new IndirectDoubleNegationRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code EqualityRules.DoubleNegation}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class DoubleNegationRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public DoubleNegationRecipe() {}

        @Override
        public String getDisplayName() {
            return "Avoid double negations; this is not Javascript";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class DoubleNegation {\n    \n    @BeforeTemplate\n    @SuppressWarnings(value = \"java:S2761\")\n    boolean before(boolean b) {\n        return !!b;\n    }\n    \n    @AfterTemplate\n    @CanIgnoreReturnValue\n    boolean after(boolean b) {\n        return b;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("!!#{b:any(boolean)}")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{b:any(boolean)}")
                        .build();

                @Override
                public J visitUnary(J.Unary elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitUnary(elem, ctx);
                }

            };
            return javaVisitor;
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code EqualityRules.Negation}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class NegationRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public NegationRecipe() {}

        @Override
        public String getDisplayName() {
            return "Don't negate an equality test or use the ternary operator to compare two booleans; directly test for inequality instead";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\n@SuppressWarnings(value = \"java:S1940\")\nstatic final class Negation {\n    \n    @BeforeTemplate\n    boolean before(boolean a, boolean b) {\n        return Refaster.anyOf(!(a == b), a ? !b : b);\n    }\n    \n    @BeforeTemplate\n    @SuppressWarnings(value = \"java:S1244\")\n    boolean before(double a, double b) {\n        return !(a == b);\n    }\n    \n    @BeforeTemplate\n    boolean before(Object a, Object b) {\n        return !(a == b);\n    }\n    \n    @AfterTemplate\n    boolean after(boolean a, boolean b) {\n        return a != b;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!(#{a:any(boolean)} == #{b:any(boolean)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{a:any(boolean)} ? !#{b:any(boolean)} : #{b}")
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("!(#{a:any(double)} == #{b:any(double)})")
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("!(#{a:any(java.lang.Object)} == #{b:any(java.lang.Object)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{a:any(boolean)} != #{b:any(boolean)}")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return javaVisitor;
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code EqualityRules.IndirectDoubleNegation}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IndirectDoubleNegationRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IndirectDoubleNegationRecipe() {}

        @Override
        public String getDisplayName() {
            return "Don't negate an inequality test or use the ternary operator to compare two booleans; directly test for equality instead";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\n@SuppressWarnings(value = \"java:S1940\")\nstatic final class IndirectDoubleNegation {\n    \n    @BeforeTemplate\n    boolean before(boolean a, boolean b) {\n        return Refaster.anyOf(!(a != b), a ? b : !b);\n    }\n    \n    @BeforeTemplate\n    @SuppressWarnings(value = \"java:S1244\")\n    boolean before(double a, double b) {\n        return !(a != b);\n    }\n    \n    @BeforeTemplate\n    boolean before(Object a, Object b) {\n        return !(a != b);\n    }\n    \n    @AfterTemplate\n    boolean after(boolean a, boolean b) {\n        return a == b;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!(#{a:any(boolean)} != #{b:any(boolean)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{a:any(boolean)} ? #{b:any(boolean)} : !#{b}")
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("!(#{a:any(double)} != #{b:any(double)})")
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("!(#{a:any(java.lang.Object)} != #{b:any(java.lang.Object)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{a:any(boolean)} == #{b:any(boolean)}")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return javaVisitor;
        }
    }

}
