package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.FileRules}.
 */
@SuppressWarnings("all")
public class FileRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public FileRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with files";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.FileRules`. [Source](https://error-prone.picnic.tech/refasterrules/FileRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new FilesReadStringWithCharsetRecipe(),
                new FilesReadStringRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code FileRules.FilesReadStringWithCharset}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class FilesReadStringWithCharsetRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public FilesReadStringWithCharsetRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `Files#readString(Path, Charset)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class FilesReadStringWithCharset {\n    \n    @BeforeTemplate\n    String before(Path path, Charset charset) throws IOException {\n        return new String(Files.readAllBytes(path), charset);\n    }\n    \n    @AfterTemplate\n    String after(Path path, Charset charset) throws IOException {\n        return Files.readString(path, charset);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("new String(java.nio.file.Files.readAllBytes(#{path:any(java.nio.file.Path)}), #{charset:any(java.nio.charset.Charset)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.nio.file.Files.readString(#{path:any(java.nio.file.Path)}, #{charset:any(java.nio.charset.Charset)})")
                        .build();

                @Override
                public J visitNewClass(J.NewClass elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitNewClass(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.nio.file.Files", true),
                        new UsesType<>("java.nio.file.Path", true),
                        new UsesType<>("java.nio.charset.Charset", true),
                        new UsesMethod<>("java.lang.String <constructor>(..)"),
                        new UsesMethod<>("java.nio.file.Files readAllBytes(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code FileRules.FilesReadString}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class FilesReadStringRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public FilesReadStringRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `Files#readString(Path)` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class FilesReadString {\n    \n    @BeforeTemplate\n    String before(Path path) throws IOException {\n        return Files.readString(path, UTF_8);\n    }\n    \n    @AfterTemplate\n    String after(Path path) throws IOException {\n        return Files.readString(path);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("java.nio.file.Files.readString(#{path:any(java.nio.file.Path)}, java.nio.charset.StandardCharsets.UTF_8)")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.nio.file.Files.readString(#{path:any(java.nio.file.Path)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.nio.charset.StandardCharsets.UTF_8");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.nio.file.Files", true),
                        new UsesType<>("java.nio.file.Path", true),
                        new UsesMethod<>("java.nio.file.Files readString(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
