package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.IntStreamRules}.
 */
@SuppressWarnings("all")
public class IntStreamRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public IntStreamRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with `IntStream`s";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.IntStreamRules`. [Source](https://error-prone.picnic.tech/refasterrules/IntStreamRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new IntStreamClosedOpenRangeRecipe(),
                new ConcatOneIntStreamRecipe(),
                new ConcatTwoIntStreamsRecipe(),
                new IntStreamFilterSortedRecipe(),
                new IntStreamIsEmptyRecipe(),
                new IntStreamIsNotEmptyRecipe(),
                new IntStreamMinRecipe(),
                new IntStreamNoneMatchRecipe(),
                new IntStreamAnyMatchRecipe(),
                new IntStreamAllMatchRecipe(),
                new IntStreamTakeWhileRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamClosedOpenRange}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamClosedOpenRangeRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamClosedOpenRangeRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `IntStream#range(int, int)` over the more contrived alternative";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamClosedOpenRange {\n    \n    @BeforeTemplate\n    IntStream before(int from, int to) {\n        return IntStream.rangeClosed(from, to - 1);\n    }\n    \n    @AfterTemplate\n    IntStream after(int from, int to) {\n        return IntStream.range(from, to);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("java.util.stream.IntStream.rangeClosed(#{from:any(int)}, #{to:any(int)} - 1)")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.util.stream.IntStream.range(#{from:any(int)}, #{to:any(int)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesMethod<>("java.util.stream.IntStream rangeClosed(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.ConcatOneIntStream}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class ConcatOneIntStreamRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public ConcatOneIntStreamRecipe() {}

        @Override
        public String getDisplayName() {
            return "Don't unnecessarily call `Streams#concat(IntStream...)`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class ConcatOneIntStream {\n    \n    @BeforeTemplate\n    IntStream before(IntStream stream) {\n        return Streams.concat(stream);\n    }\n    \n    @AfterTemplate\n    @CanIgnoreReturnValue\n    IntStream after(IntStream stream) {\n        return stream;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.collect.Streams.concat(#{stream:any(java.util.stream.IntStream)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.collect.Streams");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.collect.Streams", true),
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesMethod<>("com.google.common.collect.Streams concat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.ConcatTwoIntStreams}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class ConcatTwoIntStreamsRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public ConcatTwoIntStreamsRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `IntStream#concat(IntStream, IntStream)` over the Guava alternative";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class ConcatTwoIntStreams {\n    \n    @BeforeTemplate\n    IntStream before(IntStream s1, IntStream s2) {\n        return Streams.concat(s1, s2);\n    }\n    \n    @AfterTemplate\n    IntStream after(IntStream s1, IntStream s2) {\n        return IntStream.concat(s1, s2);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.collect.Streams.concat(#{s1:any(java.util.stream.IntStream)}, #{s2:any(java.util.stream.IntStream)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.util.stream.IntStream.concat(#{s1:any(java.util.stream.IntStream)}, #{s2:any(java.util.stream.IntStream)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.collect.Streams");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.collect.Streams", true),
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesMethod<>("com.google.common.collect.Streams concat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamFilterSorted}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamFilterSortedRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamFilterSortedRecipe() {}

        @Override
        public String getDisplayName() {
            return "Apply `IntStream#filter(IntPredicate)` before `IntStream#sorted()` to reduce the number of elements to sort";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamFilterSorted {\n    \n    @BeforeTemplate\n    IntStream before(IntStream stream, IntPredicate predicate) {\n        return stream.sorted().filter(predicate);\n    }\n    \n    @AfterTemplate\n    IntStream after(IntStream stream, IntPredicate predicate) {\n        return stream.filter(predicate).sorted();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.sorted().filter(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.filter(#{predicate:any(java.util.function.IntPredicate)}).sorted()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesType<>("java.util.function.IntPredicate", true),
                        new UsesMethod<>("java.util.stream.IntStream filter(..)"),
                        new UsesMethod<>("java.util.stream.IntStream sorted(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamIsEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamIsEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamIsEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "In order to test whether a stream has any element, simply try to find one";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamIsEmpty {\n    \n    @BeforeTemplate\n    boolean before(IntStream stream) {\n        return Refaster.anyOf(stream.count() == 0, stream.count() <= 0, stream.count() < 1, stream.findFirst().isEmpty());\n    }\n    \n    @AfterTemplate\n    boolean after(IntStream stream) {\n        return stream.findAny().isEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.count() == 0")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.count() <= 0")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.count() < 1")
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.findFirst().isEmpty()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.findAny().isEmpty()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.IntStream count(..)"),
                            new UsesMethod<>("java.util.stream.IntStream count(..)"),
                            new UsesMethod<>("java.util.stream.IntStream count(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalInt isEmpty(..)"),
                                new UsesMethod<>("java.util.stream.IntStream findFirst(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamIsNotEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamIsNotEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamIsNotEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "In order to test whether a stream has any element, simply try to find one";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamIsNotEmpty {\n    \n    @BeforeTemplate\n    boolean before(IntStream stream) {\n        return Refaster.anyOf(stream.count() != 0, stream.count() > 0, stream.count() >= 1, stream.findFirst().isPresent());\n    }\n    \n    @AfterTemplate\n    boolean after(IntStream stream) {\n        return stream.findAny().isPresent();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.count() != 0")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.count() > 0")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.count() >= 1")
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.findFirst().isPresent()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.findAny().isPresent()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.IntStream count(..)"),
                            new UsesMethod<>("java.util.stream.IntStream count(..)"),
                            new UsesMethod<>("java.util.stream.IntStream count(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalInt isPresent(..)"),
                                new UsesMethod<>("java.util.stream.IntStream findFirst(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamMin}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamMinRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamMinRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `IntStreamRules.IntStreamMin`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamMin {\n    \n    @BeforeTemplate\n    OptionalInt before(IntStream stream) {\n        return stream.sorted().findFirst();\n    }\n    \n    @AfterTemplate\n    OptionalInt after(IntStream stream) {\n        return stream.min();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.sorted().findFirst()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.min()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.OptionalInt", true),
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesMethod<>("java.util.stream.IntStream findFirst(..)"),
                        new UsesMethod<>("java.util.stream.IntStream sorted(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamNoneMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamNoneMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamNoneMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `IntStream#noneMatch(IntPredicate)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamNoneMatch {\n    \n    @BeforeTemplate\n    boolean before(IntStream stream, IntPredicate predicate) {\n        return Refaster.anyOf(!stream.anyMatch(predicate), stream.allMatch(predicate.negate()), stream.filter(predicate).findAny().isEmpty());\n    }\n    \n    @AfterTemplate\n    boolean after(IntStream stream, IntPredicate predicate) {\n        return stream.noneMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!#{stream:any(java.util.stream.IntStream)}.anyMatch(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.allMatch(#{predicate:any(java.util.function.IntPredicate)}.negate())")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.filter(#{predicate:any(java.util.function.IntPredicate)}).findAny().isEmpty()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.noneMatch(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesType<>("java.util.function.IntPredicate", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.IntStream anyMatch(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.stream.IntStream allMatch(..)"),
                                new UsesMethod<>("java.util.function.IntPredicate negate(..)")
                            ),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalInt isEmpty(..)"),
                                new UsesMethod<>("java.util.stream.IntStream findAny(..)"),
                                new UsesMethod<>("java.util.stream.IntStream filter(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamAnyMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamAnyMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamAnyMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `IntStream#anyMatch(IntPredicate)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamAnyMatch {\n    \n    @BeforeTemplate\n    @SuppressWarnings(value = \"java:S4034\")\n    boolean before(IntStream stream, IntPredicate predicate) {\n        return Refaster.anyOf(!stream.noneMatch(predicate), stream.filter(predicate).findAny().isPresent());\n    }\n    \n    @AfterTemplate\n    boolean after(IntStream stream, IntPredicate predicate) {\n        return stream.anyMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!#{stream:any(java.util.stream.IntStream)}.noneMatch(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.filter(#{predicate:any(java.util.function.IntPredicate)}).findAny().isPresent()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.anyMatch(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesType<>("java.util.function.IntPredicate", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.IntStream noneMatch(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalInt isPresent(..)"),
                                new UsesMethod<>("java.util.stream.IntStream findAny(..)"),
                                new UsesMethod<>("java.util.stream.IntStream filter(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamAllMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamAllMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamAllMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `IntStreamRules.IntStreamAllMatch`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamAllMatch {\n    \n    @BeforeTemplate\n    boolean before(IntStream stream, IntPredicate predicate) {\n        return stream.noneMatch(predicate.negate());\n    }\n    \n    @AfterTemplate\n    boolean after(IntStream stream, IntPredicate predicate) {\n        return stream.allMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.noneMatch(#{predicate:any(java.util.function.IntPredicate)}.negate())")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.allMatch(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesType<>("java.util.function.IntPredicate", true),
                        new UsesMethod<>("java.util.stream.IntStream noneMatch(..)"),
                        new UsesMethod<>("java.util.function.IntPredicate negate(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code IntStreamRules.IntStreamTakeWhile}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IntStreamTakeWhileRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IntStreamTakeWhileRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `IntStreamRules.IntStreamTakeWhile`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IntStreamTakeWhile {\n    \n    @BeforeTemplate\n    IntStream before(IntStream stream, IntPredicate predicate) {\n        return stream.takeWhile(predicate).filter(predicate);\n    }\n    \n    @AfterTemplate\n    IntStream after(IntStream stream, IntPredicate predicate) {\n        return stream.takeWhile(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.takeWhile(#{predicate:any(java.util.function.IntPredicate)}).filter(#{predicate})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.IntStream)}.takeWhile(#{predicate:any(java.util.function.IntPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.IntStream", true),
                        new UsesType<>("java.util.function.IntPredicate", true),
                        new UsesMethod<>("java.util.stream.IntStream filter(..)"),
                        new UsesMethod<>("java.util.stream.IntStream takeWhile(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
