package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.LongStreamRules}.
 */
@SuppressWarnings("all")
public class LongStreamRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public LongStreamRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with `LongStream`s";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.LongStreamRules`. [Source](https://error-prone.picnic.tech/refasterrules/LongStreamRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new LongStreamClosedOpenRangeRecipe(),
                new ConcatOneLongStreamRecipe(),
                new ConcatTwoLongStreamsRecipe(),
                new LongStreamFilterSortedRecipe(),
                new LongStreamIsEmptyRecipe(),
                new LongStreamIsNotEmptyRecipe(),
                new LongStreamMinRecipe(),
                new LongStreamNoneMatchRecipe(),
                new LongStreamAnyMatchRecipe(),
                new LongStreamAllMatchRecipe(),
                new LongStreamTakeWhileRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamClosedOpenRange}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamClosedOpenRangeRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamClosedOpenRangeRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `LongStream#range(long, long)` over the more contrived alternative";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamClosedOpenRange {\n    \n    @BeforeTemplate\n    LongStream before(long from, long to) {\n        return LongStream.rangeClosed(from, to - 1);\n    }\n    \n    @AfterTemplate\n    LongStream after(long from, long to) {\n        return LongStream.range(from, to);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("java.util.stream.LongStream.rangeClosed(#{from:any(long)}, #{to:any(long)} - 1)")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.util.stream.LongStream.range(#{from:any(long)}, #{to:any(long)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesMethod<>("java.util.stream.LongStream rangeClosed(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.ConcatOneLongStream}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class ConcatOneLongStreamRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public ConcatOneLongStreamRecipe() {}

        @Override
        public String getDisplayName() {
            return "Don't unnecessarily call `Streams#concat(LongStream...)`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class ConcatOneLongStream {\n    \n    @BeforeTemplate\n    LongStream before(LongStream stream) {\n        return Streams.concat(stream);\n    }\n    \n    @AfterTemplate\n    @CanIgnoreReturnValue\n    LongStream after(LongStream stream) {\n        return stream;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.collect.Streams.concat(#{stream:any(java.util.stream.LongStream)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.collect.Streams");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.collect.Streams", true),
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesMethod<>("com.google.common.collect.Streams concat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.ConcatTwoLongStreams}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class ConcatTwoLongStreamsRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public ConcatTwoLongStreamsRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `LongStream#concat(LongStream, LongStream)` over the Guava alternative";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class ConcatTwoLongStreams {\n    \n    @BeforeTemplate\n    LongStream before(LongStream s1, LongStream s2) {\n        return Streams.concat(s1, s2);\n    }\n    \n    @AfterTemplate\n    LongStream after(LongStream s1, LongStream s2) {\n        return LongStream.concat(s1, s2);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.collect.Streams.concat(#{s1:any(java.util.stream.LongStream)}, #{s2:any(java.util.stream.LongStream)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.util.stream.LongStream.concat(#{s1:any(java.util.stream.LongStream)}, #{s2:any(java.util.stream.LongStream)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.collect.Streams");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.collect.Streams", true),
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesMethod<>("com.google.common.collect.Streams concat(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamFilterSorted}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamFilterSortedRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamFilterSortedRecipe() {}

        @Override
        public String getDisplayName() {
            return "Apply `LongStream#filter(LongPredicate)` before `LongStream#sorted()` to reduce the number of elements to sort";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamFilterSorted {\n    \n    @BeforeTemplate\n    LongStream before(LongStream stream, LongPredicate predicate) {\n        return stream.sorted().filter(predicate);\n    }\n    \n    @AfterTemplate\n    LongStream after(LongStream stream, LongPredicate predicate) {\n        return stream.filter(predicate).sorted();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.sorted().filter(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.filter(#{predicate:any(java.util.function.LongPredicate)}).sorted()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesType<>("java.util.function.LongPredicate", true),
                        new UsesMethod<>("java.util.stream.LongStream filter(..)"),
                        new UsesMethod<>("java.util.stream.LongStream sorted(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamIsEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamIsEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamIsEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "In order to test whether a stream has any element, simply try to find one";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamIsEmpty {\n    \n    @BeforeTemplate\n    boolean before(LongStream stream) {\n        return Refaster.anyOf(stream.count() == 0, stream.count() <= 0, stream.count() < 1, stream.findFirst().isEmpty());\n    }\n    \n    @AfterTemplate\n    boolean after(LongStream stream) {\n        return stream.findAny().isEmpty();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.count() == 0")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.count() <= 0")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.count() < 1")
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.findFirst().isEmpty()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.findAny().isEmpty()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.LongStream count(..)"),
                            new UsesMethod<>("java.util.stream.LongStream count(..)"),
                            new UsesMethod<>("java.util.stream.LongStream count(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalLong isEmpty(..)"),
                                new UsesMethod<>("java.util.stream.LongStream findFirst(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamIsNotEmpty}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamIsNotEmptyRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamIsNotEmptyRecipe() {}

        @Override
        public String getDisplayName() {
            return "In order to test whether a stream has any element, simply try to find one";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamIsNotEmpty {\n    \n    @BeforeTemplate\n    boolean before(LongStream stream) {\n        return Refaster.anyOf(stream.count() != 0, stream.count() > 0, stream.count() >= 1, stream.findFirst().isPresent());\n    }\n    \n    @AfterTemplate\n    boolean after(LongStream stream) {\n        return stream.findAny().isPresent();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.count() != 0")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.count() > 0")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.count() >= 1")
                        .build();
                final JavaTemplate before$3 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.findFirst().isPresent()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.findAny().isPresent()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$3.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.LongStream count(..)"),
                            new UsesMethod<>("java.util.stream.LongStream count(..)"),
                            new UsesMethod<>("java.util.stream.LongStream count(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalLong isPresent(..)"),
                                new UsesMethod<>("java.util.stream.LongStream findFirst(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamMin}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamMinRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamMinRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `LongStreamRules.LongStreamMin`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamMin {\n    \n    @BeforeTemplate\n    OptionalLong before(LongStream stream) {\n        return stream.sorted().findFirst();\n    }\n    \n    @AfterTemplate\n    OptionalLong after(LongStream stream) {\n        return stream.min();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.sorted().findFirst()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.min()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.OptionalLong", true),
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesMethod<>("java.util.stream.LongStream findFirst(..)"),
                        new UsesMethod<>("java.util.stream.LongStream sorted(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamNoneMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamNoneMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamNoneMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `LongStream#noneMatch(LongPredicate)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamNoneMatch {\n    \n    @BeforeTemplate\n    boolean before(LongStream stream, LongPredicate predicate) {\n        return Refaster.anyOf(!stream.anyMatch(predicate), stream.allMatch(predicate.negate()), stream.filter(predicate).findAny().isEmpty());\n    }\n    \n    @AfterTemplate\n    boolean after(LongStream stream, LongPredicate predicate) {\n        return stream.noneMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!#{stream:any(java.util.stream.LongStream)}.anyMatch(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.allMatch(#{predicate:any(java.util.function.LongPredicate)}.negate())")
                        .build();
                final JavaTemplate before$2 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.filter(#{predicate:any(java.util.function.LongPredicate)}).findAny().isEmpty()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.noneMatch(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$2.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesType<>("java.util.function.LongPredicate", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.LongStream anyMatch(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.stream.LongStream allMatch(..)"),
                                new UsesMethod<>("java.util.function.LongPredicate negate(..)")
                            ),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalLong isEmpty(..)"),
                                new UsesMethod<>("java.util.stream.LongStream findAny(..)"),
                                new UsesMethod<>("java.util.stream.LongStream filter(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamAnyMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamAnyMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamAnyMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `LongStream#anyMatch(LongPredicate)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamAnyMatch {\n    \n    @BeforeTemplate\n    @SuppressWarnings(value = \"java:S4034\")\n    boolean before(LongStream stream, LongPredicate predicate) {\n        return Refaster.anyOf(!stream.noneMatch(predicate), stream.filter(predicate).findAny().isPresent());\n    }\n    \n    @AfterTemplate\n    boolean after(LongStream stream, LongPredicate predicate) {\n        return stream.anyMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("!#{stream:any(java.util.stream.LongStream)}.noneMatch(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.filter(#{predicate:any(java.util.function.LongPredicate)}).findAny().isPresent()")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.anyMatch(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesType<>("java.util.function.LongPredicate", true),
                        Preconditions.or(
                            new UsesMethod<>("java.util.stream.LongStream noneMatch(..)"),
                            Preconditions.and(
                                new UsesMethod<>("java.util.OptionalLong isPresent(..)"),
                                new UsesMethod<>("java.util.stream.LongStream findAny(..)"),
                                new UsesMethod<>("java.util.stream.LongStream filter(..)")
                            )
                        )
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamAllMatch}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamAllMatchRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamAllMatchRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `LongStreamRules.LongStreamAllMatch`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamAllMatch {\n    \n    @BeforeTemplate\n    boolean before(LongStream stream, LongPredicate predicate) {\n        return stream.noneMatch(predicate.negate());\n    }\n    \n    @AfterTemplate\n    boolean after(LongStream stream, LongPredicate predicate) {\n        return stream.allMatch(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.noneMatch(#{predicate:any(java.util.function.LongPredicate)}.negate())")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.allMatch(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesType<>("java.util.function.LongPredicate", true),
                        new UsesMethod<>("java.util.stream.LongStream noneMatch(..)"),
                        new UsesMethod<>("java.util.function.LongPredicate negate(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code LongStreamRules.LongStreamTakeWhile}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class LongStreamTakeWhileRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public LongStreamTakeWhileRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `LongStreamRules.LongStreamTakeWhile`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class LongStreamTakeWhile {\n    \n    @BeforeTemplate\n    LongStream before(LongStream stream, LongPredicate predicate) {\n        return stream.takeWhile(predicate).filter(predicate);\n    }\n    \n    @AfterTemplate\n    LongStream after(LongStream stream, LongPredicate predicate) {\n        return stream.takeWhile(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.takeWhile(#{predicate:any(java.util.function.LongPredicate)}).filter(#{predicate})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{stream:any(java.util.stream.LongStream)}.takeWhile(#{predicate:any(java.util.function.LongPredicate)})")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.stream.LongStream", true),
                        new UsesType<>("java.util.function.LongPredicate", true),
                        new UsesMethod<>("java.util.stream.LongStream filter(..)"),
                        new UsesMethod<>("java.util.stream.LongStream takeWhile(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
