package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.NullRules}.
 */
@SuppressWarnings("all")
public class NullRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public NullRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with (possibly) null values";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.NullRules`. [Source](https://error-prone.picnic.tech/refasterrules/NullRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new IsNullRecipe(),
                new IsNotNullRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code NullRules.IsNull}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IsNullRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IsNullRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer the `==` operator (with `null` as the second operand) over `Objects#isNull(Object)`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IsNull {\n    \n    @BeforeTemplate\n    boolean before(@Nullable\n    Object object) {\n        return Refaster.anyOf(null == object, Objects.isNull(object));\n    }\n    \n    @AfterTemplate\n    boolean after(@Nullable\n    Object object) {\n        return object == null;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("null == #{object:any(java.lang.@org.jspecify.annotations.Nullable Object)}")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("java.util.Objects.isNull(#{object:any(java.lang.@org.jspecify.annotations.Nullable Object)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{object:any(java.lang.@org.jspecify.annotations.Nullable Object)} == null")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Objects");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.Objects", true),
                        new UsesMethod<>("java.util.Objects isNull(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code NullRules.IsNotNull}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class IsNotNullRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public IsNotNullRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer the `!=` operator (with `null` as the second operand) over `Objects#nonNull(Object)`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class IsNotNull {\n    \n    @BeforeTemplate\n    boolean before(@Nullable\n    Object object) {\n        return Refaster.anyOf(null != object, Objects.nonNull(object));\n    }\n    \n    @AfterTemplate\n    boolean after(@Nullable\n    Object object) {\n        return object != null;\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before$0 = JavaTemplate
                        .builder("null != #{object:any(java.lang.@org.jspecify.annotations.Nullable Object)}")
                        .build();
                final JavaTemplate before$1 = JavaTemplate
                        .builder("java.util.Objects.nonNull(#{object:any(java.lang.@org.jspecify.annotations.Nullable Object)})")
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{object:any(java.lang.@org.jspecify.annotations.Nullable Object)} != null")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before$0.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    if ((matcher = before$1.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Objects");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES, SIMPLIFY_BOOLEANS
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.Objects", true),
                        new UsesMethod<>("java.util.Objects nonNull(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
