package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.PatternRules}.
 */
@SuppressWarnings("all")
public class PatternRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public PatternRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to code dealing with regular expressions";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.PatternRules`. [Source](https://error-prone.picnic.tech/refasterrules/PatternRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new PatternAsPredicateRecipe(),
                new PatternCompileAsPredicateRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code PatternRules.PatternAsPredicate}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class PatternAsPredicateRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public PatternAsPredicateRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `Pattern#asPredicate()` over non-JDK alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class PatternAsPredicate {\n    \n    @BeforeTemplate\n    Predicate<CharSequence> before(Pattern pattern) {\n        return Predicates.contains(pattern);\n    }\n    \n    @AfterTemplate\n    Predicate<String> after(Pattern pattern) {\n        return pattern.asPredicate();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.base.Predicates.contains(#{pattern:any(java.util.regex.Pattern)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{pattern:any(java.util.regex.Pattern)}.asPredicate()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.base.Predicates");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.common.base.Predicates", true),
                        new UsesType<>("java.util.function.Predicate", true),
                        new UsesType<>("java.util.regex.Pattern", true),
                        new UsesMethod<>("com.google.common.base.Predicates contains(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code PatternRules.PatternCompileAsPredicate}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class PatternCompileAsPredicateRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public PatternCompileAsPredicateRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `Pattern#asPredicate()` over non-JDK alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class PatternCompileAsPredicate {\n    \n    @BeforeTemplate\n    Predicate<CharSequence> before(String pattern) {\n        return containsPattern(pattern);\n    }\n    \n    @AfterTemplate\n    Predicate<String> after(String pattern) {\n        return Pattern.compile(pattern).asPredicate();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.common.base.Predicates.containsPattern(#{pattern:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("guava"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("java.util.regex.Pattern.compile(#{pattern:any(java.lang.String)}).asPredicate()")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("com.google.common.base.Predicates.containsPattern");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.function.Predicate", true),
                        new UsesMethod<>("com.google.common.base.Predicates containsPattern(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
