package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.ReactorRules}.
 */
@SuppressWarnings("all")
public class ReactorRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public ReactorRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to Reactor expressions and statements";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.ReactorRules`. [Source](https://error-prone.picnic.tech/refasterrules/ReactorRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new FluxJustRecipe(),
                new StepVerifierLastStepVerifyCompleteRecipe(),
                new StepVerifierLastStepVerifyErrorRecipe(),
                new StepVerifierLastStepVerifyErrorMatchesRecipe(),
                new StepVerifierLastStepVerifyErrorSatisfiesRecipe(),
                new StepVerifierLastStepVerifyErrorMessageRecipe(),
                new StepVerifierLastStepVerifyTimeoutRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.FluxJust}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class FluxJustRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public FluxJustRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `Flux#just(Object)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class FluxJust {\n    \n    @BeforeTemplate\n    Flux<Integer> before(int start) {\n        return Flux.range(start, 1);\n    }\n    \n    @AfterTemplate\n    Flux<Integer> after(int start) {\n        return Flux.just(start);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("reactor.core.publisher.Flux.range(#{start:any(int)}, 1)")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-core"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("reactor.core.publisher.Flux.just(#{start:any(int)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("reactor.core.publisher.Flux", true),
                        new UsesMethod<>("reactor.core.publisher.Flux range(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.StepVerifierLastStepVerifyComplete}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class StepVerifierLastStepVerifyCompleteRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public StepVerifierLastStepVerifyCompleteRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `StepVerifier.LastStep#verifyComplete()` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class StepVerifierLastStepVerifyComplete {\n    \n    @BeforeTemplate\n    Duration before(StepVerifier.LastStep step) {\n        return step.expectComplete().verify();\n    }\n    \n    @AfterTemplate\n    Duration after(StepVerifier.LastStep step) {\n        return step.verifyComplete();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.expectComplete().verify()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.verifyComplete()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.time.Duration", true),
                        new UsesMethod<>("reactor.test.StepVerifier verify(..)"),
                        new UsesMethod<>("reactor.test.StepVerifier.LastStep expectComplete(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.StepVerifierLastStepVerifyError}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class StepVerifierLastStepVerifyErrorRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public StepVerifierLastStepVerifyErrorRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `StepVerifier.LastStep#verifyError()` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class StepVerifierLastStepVerifyError {\n    \n    @BeforeTemplate\n    Duration before(StepVerifier.LastStep step) {\n        return step.expectError().verify();\n    }\n    \n    @AfterTemplate\n    Duration after(StepVerifier.LastStep step) {\n        return step.verifyError();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.expectError().verify()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.verifyError()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.time.Duration", true),
                        new UsesMethod<>("reactor.test.StepVerifier verify(..)"),
                        new UsesMethod<>("reactor.test.StepVerifier.LastStep expectError(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.StepVerifierLastStepVerifyErrorMatches}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class StepVerifierLastStepVerifyErrorMatchesRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public StepVerifierLastStepVerifyErrorMatchesRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `StepVerifier.LastStep#verifyErrorMatches(Predicate)` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class StepVerifierLastStepVerifyErrorMatches {\n    \n    @BeforeTemplate\n    Duration before(StepVerifier.LastStep step, Predicate<Throwable> predicate) {\n        return step.expectErrorMatches(predicate).verify();\n    }\n    \n    @AfterTemplate\n    Duration after(StepVerifier.LastStep step, Predicate<Throwable> predicate) {\n        return step.verifyErrorMatches(predicate);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.expectErrorMatches(#{predicate:any(java.util.function.Predicate<java.lang.Throwable>)}).verify()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.verifyErrorMatches(#{predicate:any(java.util.function.Predicate<java.lang.Throwable>)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.time.Duration", true),
                        new UsesType<>("java.util.function.Predicate", true),
                        new UsesMethod<>("reactor.test.StepVerifier verify(..)"),
                        new UsesMethod<>("reactor.test.StepVerifier.LastStep expectErrorMatches(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.StepVerifierLastStepVerifyErrorSatisfies}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class StepVerifierLastStepVerifyErrorSatisfiesRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public StepVerifierLastStepVerifyErrorSatisfiesRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `StepVerifier.LastStep#verifyErrorSatisfies(Consumer)` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class StepVerifierLastStepVerifyErrorSatisfies {\n    \n    @BeforeTemplate\n    Duration before(StepVerifier.LastStep step, Consumer<Throwable> consumer) {\n        return step.expectErrorSatisfies(consumer).verify();\n    }\n    \n    @AfterTemplate\n    Duration after(StepVerifier.LastStep step, Consumer<Throwable> consumer) {\n        return step.verifyErrorSatisfies(consumer);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.expectErrorSatisfies(#{consumer:any(java.util.function.Consumer<java.lang.Throwable>)}).verify()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.verifyErrorSatisfies(#{consumer:any(java.util.function.Consumer<java.lang.Throwable>)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.time.Duration", true),
                        new UsesType<>("java.util.function.Consumer", true),
                        new UsesMethod<>("reactor.test.StepVerifier verify(..)"),
                        new UsesMethod<>("reactor.test.StepVerifier.LastStep expectErrorSatisfies(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.StepVerifierLastStepVerifyErrorMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class StepVerifierLastStepVerifyErrorMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public StepVerifierLastStepVerifyErrorMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `StepVerifier.LastStep#verifyErrorMessage(String)` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class StepVerifierLastStepVerifyErrorMessage {\n    \n    @BeforeTemplate\n    Duration before(StepVerifier.LastStep step, String message) {\n        return step.expectErrorMessage(message).verify();\n    }\n    \n    @AfterTemplate\n    Duration after(StepVerifier.LastStep step, String message) {\n        return step.verifyErrorMessage(message);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.expectErrorMessage(#{message:any(java.lang.String)}).verify()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.verifyErrorMessage(#{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.time.Duration", true),
                        new UsesMethod<>("reactor.test.StepVerifier verify(..)"),
                        new UsesMethod<>("reactor.test.StepVerifier.LastStep expectErrorMessage(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code ReactorRules.StepVerifierLastStepVerifyTimeout}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class StepVerifierLastStepVerifyTimeoutRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public StepVerifierLastStepVerifyTimeoutRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `StepVerifier.LastStep#verifyTimeout(Duration)` over more verbose alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class StepVerifierLastStepVerifyTimeout {\n    \n    @BeforeTemplate\n    Duration before(StepVerifier.LastStep step, Duration duration) {\n        return step.expectTimeout(duration).verify();\n    }\n    \n    @AfterTemplate\n    Duration after(StepVerifier.LastStep step, Duration duration) {\n        return step.verifyTimeout(duration);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.expectTimeout(#{duration:any(java.time.Duration)}).verify()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("#{step:any(reactor.test.StepVerifier.LastStep)}.verifyTimeout(#{duration:any(java.time.Duration)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("reactor-test"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.time.Duration", true),
                        new UsesMethod<>("reactor.test.StepVerifier verify(..)"),
                        new UsesMethod<>("reactor.test.StepVerifier.LastStep expectTimeout(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
