package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.SuggestedFixRules}.
 */
@SuppressWarnings("all")
public class SuggestedFixRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public SuggestedFixRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules related to expressions dealing with `SuggestedFix`es";
    }

    @Override
    public String getDescription() {
        return "Refaster template recipes for `tech.picnic.errorprone.refasterrules.SuggestedFixRules`. [Source](https://error-prone.picnic.tech/refasterrules/SuggestedFixRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new SuggestedFixDeleteRecipe(),
                new SuggestedFixReplaceTreeRecipe(),
                new SuggestedFixReplaceStartEndRecipe(),
                new SuggestedFixReplaceTreeStartEndRecipe(),
                new SuggestedFixSwapRecipe(),
                new SuggestedFixPrefixWithRecipe(),
                new SuggestedFixPostfixWithRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixDelete}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixDeleteRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixDeleteRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#delete(Tree)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixDelete {\n    \n    @BeforeTemplate\n    SuggestedFix before(Tree tree) {\n        return SuggestedFix.builder().delete(tree).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(Tree tree) {\n        return SuggestedFix.delete(tree);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().delete(#{tree:any(com.sun.source.tree.Tree)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.delete(#{tree:any(com.sun.source.tree.Tree)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesType<>("com.sun.source.tree.Tree", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder delete(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixReplaceTree}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixReplaceTreeRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixReplaceTreeRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#replace(Tree, String)`} over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixReplaceTree {\n    \n    @BeforeTemplate\n    SuggestedFix before(Tree tree, String replaceWith) {\n        return SuggestedFix.builder().replace(tree, replaceWith).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(Tree tree, String replaceWith) {\n        return SuggestedFix.replace(tree, replaceWith);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().replace(#{tree:any(com.sun.source.tree.Tree)}, #{replaceWith:any(java.lang.String)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.replace(#{tree:any(com.sun.source.tree.Tree)}, #{replaceWith:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesType<>("com.sun.source.tree.Tree", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder replace(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixReplaceStartEnd}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixReplaceStartEndRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixReplaceStartEndRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#replace(int, int, String)`} over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixReplaceStartEnd {\n    \n    @BeforeTemplate\n    SuggestedFix before(int start, int end, String replaceWith) {\n        return SuggestedFix.builder().replace(start, end, replaceWith).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(int start, int end, String replaceWith) {\n        return SuggestedFix.replace(start, end, replaceWith);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().replace(#{start:any(int)}, #{end:any(int)}, #{replaceWith:any(java.lang.String)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.replace(#{start:any(int)}, #{end:any(int)}, #{replaceWith:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder replace(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixReplaceTreeStartEnd}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixReplaceTreeStartEndRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixReplaceTreeStartEndRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#replace(Tree, String, int, int)`} over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixReplaceTreeStartEnd {\n    \n    @BeforeTemplate\n    SuggestedFix before(Tree tree, String replaceWith, int start, int end) {\n        return SuggestedFix.builder().replace(tree, replaceWith, start, end).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(Tree tree, String replaceWith, int start, int end) {\n        return SuggestedFix.replace(tree, replaceWith, start, end);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().replace(#{tree:any(com.sun.source.tree.Tree)}, #{replaceWith:any(java.lang.String)}, #{start:any(int)}, #{end:any(int)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.replace(#{tree:any(com.sun.source.tree.Tree)}, #{replaceWith:any(java.lang.String)}, #{start:any(int)}, #{end:any(int)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2), matcher.parameter(3)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesType<>("com.sun.source.tree.Tree", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder replace(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixSwap}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixSwapRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixSwapRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#swap(Tree, Tree)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixSwap {\n    \n    @BeforeTemplate\n    SuggestedFix before(Tree tree1, Tree tree2) {\n        return SuggestedFix.builder().swap(tree1, tree2).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(Tree tree1, Tree tree2) {\n        return SuggestedFix.swap(tree1, tree2);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().swap(#{tree1:any(com.sun.source.tree.Tree)}, #{tree2:any(com.sun.source.tree.Tree)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.swap(#{tree1:any(com.sun.source.tree.Tree)}, #{tree2:any(com.sun.source.tree.Tree)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesType<>("com.sun.source.tree.Tree", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder swap(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixPrefixWith}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixPrefixWithRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixPrefixWithRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#prefixWith(Tree, String)` over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixPrefixWith {\n    \n    @BeforeTemplate\n    SuggestedFix before(Tree tree, String prefix) {\n        return SuggestedFix.builder().prefixWith(tree, prefix).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(Tree tree, String prefix) {\n        return SuggestedFix.prefixWith(tree, prefix);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().prefixWith(#{tree:any(com.sun.source.tree.Tree)}, #{prefix:any(java.lang.String)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.prefixWith(#{tree:any(com.sun.source.tree.Tree)}, #{prefix:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesType<>("com.sun.source.tree.Tree", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder prefixWith(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code SuggestedFixRules.SuggestedFixPostfixWith}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class SuggestedFixPostfixWithRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public SuggestedFixPostfixWithRecipe() {}

        @Override
        public String getDisplayName() {
            return "Prefer `SuggestedFix#postfixWith(Tree, String)`} over more contrived alternatives";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class SuggestedFixPostfixWith {\n    \n    @BeforeTemplate\n    SuggestedFix before(Tree tree, String postfix) {\n        return SuggestedFix.builder().postfixWith(tree, postfix).build();\n    }\n    \n    @AfterTemplate\n    SuggestedFix after(Tree tree, String postfix) {\n        return SuggestedFix.postfixWith(tree, postfix);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.builder().postfixWith(#{tree:any(com.sun.source.tree.Tree)}, #{postfix:any(java.lang.String)}).build()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("com.google.errorprone.fixes.SuggestedFix.postfixWith(#{tree:any(com.sun.source.tree.Tree)}, #{postfix:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("error_prone_check_api"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("com.google.errorprone.fixes.SuggestedFix", true),
                        new UsesType<>("com.sun.source.tree.Tree", true),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder build(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix.Builder postfixWith(..)"),
                        new UsesMethod<>("com.google.errorprone.fixes.SuggestedFix builder(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
