package tech.picnic.errorprone.refasterrules;

import org.openrewrite.ExecutionContext;
import org.openrewrite.Preconditions;
import org.openrewrite.Recipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.internal.lang.NonNullApi;
import org.openrewrite.java.JavaParser;
import org.openrewrite.java.JavaTemplate;
import org.openrewrite.java.JavaVisitor;
import org.openrewrite.java.search.*;
import org.openrewrite.java.template.Primitive;
import org.openrewrite.java.template.function.*;
import org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor;
import org.openrewrite.java.tree.*;

import java.util.*;

import static org.openrewrite.java.template.internal.AbstractRefasterJavaVisitor.EmbeddingOption.*;

/**
 * OpenRewrite recipes created for Refaster template {@code tech.picnic.errorprone.refasterrules.TestNGToAssertJRules}.
 */
@SuppressWarnings("all")
public class TestNGToAssertJRulesRecipes extends Recipe {
    /**
     * Instantiates a new instance.
     */
    public TestNGToAssertJRulesRecipes() {}

    @Override
    public String getDisplayName() {
        return "Refaster rules that replace TestNG assertions with equivalent AssertJ assertions";
    }

    @Override
    public String getDescription() {
        return "<p>Some of the classes below have TestNG `@BeforeTemplate`s that reference wildcard type\n bounds (`<?>`), while the associated AssertJ `@AfterTemplate`s reference stricter\n type bounds. This introduces the risk of producing invalid code. We do this anyway, because\n TestNG's wildcard types can cause javac to infer less specific types than AssertJ requires, while\n the appropriate (more specific) types _will_ be inferred properly when plugged into AssertJ's\n API.\n\n <p>The following is an example of a TestNG statement, which would not be rewritten if it weren't\n for the wildcard matching (note that the type parameters of the map on the right-hand side will\n be inferred to be `<Object, Object>` rather than `<String, Object>`).\n\n <pre>{@code\n List<Map<String, Object>> myMaps = new ArrayList<>();\n assertEquals(myMaps, ImmutableList.of(ImmutableMap.of()));\n }</pre>. [Source](https://error-prone.picnic.tech/refasterrules/TestNGToAssertJRules).";
    }

    @Override
    public List<Recipe> getRecipeList() {
        return Arrays.asList(
                new FailRecipe(),
                new FailWithMessageRecipe(),
                new FailWithMessageAndThrowableRecipe(),
                new AssertTrueRecipe(),
                new AssertTrueWithMessageRecipe(),
                new AssertFalseRecipe(),
                new AssertFalseWithMessageRecipe(),
                new AssertNullRecipe(),
                new AssertNullWithMessageRecipe(),
                new AssertNotNullRecipe(),
                new AssertNotNullWithMessageRecipe(),
                new AssertSameRecipe(),
                new AssertSameWithMessageRecipe(),
                new AssertNotSameRecipe(),
                new AssertNotSameWithMessageRecipe(),
                new AssertEqualRecipe(),
                new AssertEqualWithMessageRecipe(),
                new AssertEqualFloatsWithDeltaRecipe(),
                new AssertEqualFloatsWithDeltaWithMessageRecipe(),
                new AssertEqualDoublesWithDeltaRecipe(),
                new AssertEqualDoublesWithDeltaWithMessageRecipe(),
                new AssertEqualArrayIterationOrderRecipe(),
                new AssertEqualArrayIterationOrderWithMessageRecipe(),
                new AssertEqualArraysIrrespectiveOfOrderRecipe(),
                new AssertEqualArraysIrrespectiveOfOrderWithMessageRecipe(),
                new AssertUnequalRecipe(),
                new AssertUnequalWithMessageRecipe(),
                new AssertUnequalFloatsWithDeltaRecipe(),
                new AssertUnequalFloatsWithDeltaWithMessageRecipe(),
                new AssertUnequalDoublesWithDeltaRecipe(),
                new AssertUnequalDoublesWithDeltaWithMessageRecipe(),
                new AssertThrowsRecipe()
        );
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.Fail}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class FailRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public FailRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.Fail`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class Fail {\n    \n    @BeforeTemplate\n    void before() {\n        Assert.fail();\n    }\n    \n    @AfterTemplate\n    @DoNotCall\n    void after() {\n        throw new AssertionError();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.fail()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("throw new AssertionError();")
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace()),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.testng.Assert", true),
                        new UsesMethod<>("org.testng.Assert fail(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.FailWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class FailWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public FailWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.FailWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class FailWithMessage {\n    \n    @BeforeTemplate\n    void before(String message) {\n        Assert.fail(message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(String message) {\n        fail(message);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.fail(#{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.fail(#{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.testng.Assert", true),
                        new UsesMethod<>("org.testng.Assert fail(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.FailWithMessageAndThrowable}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class FailWithMessageAndThrowableRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public FailWithMessageAndThrowableRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.FailWithMessageAndThrowable`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class FailWithMessageAndThrowable {\n    \n    @BeforeTemplate\n    void before(String message, Throwable throwable) {\n        Assert.fail(message, throwable);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(String message, Throwable throwable) {\n        fail(message, throwable);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.fail(#{message:any(java.lang.String)}, #{throwable:any(java.lang.Throwable)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.fail(#{message:any(java.lang.String)}, #{throwable:any(java.lang.Throwable)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.testng.Assert", true),
                        new UsesMethod<>("org.testng.Assert fail(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertTrue}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertTrueRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertTrueRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertTrue`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertTrue {\n    \n    @BeforeTemplate\n    void before(boolean condition) {\n        assertTrue(condition);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(boolean condition) {\n        assertThat(condition).isTrue();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertTrue(#{condition:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{condition:any(boolean)}).isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertTrue");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertTrue(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertTrueWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertTrueWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertTrueWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertTrueWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertTrueWithMessage {\n    \n    @BeforeTemplate\n    void before(boolean condition, String message) {\n        assertTrue(condition, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(boolean condition, String message) {\n        assertThat(condition).withFailMessage(message).isTrue();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertTrue(#{condition:any(boolean)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{condition:any(boolean)}).withFailMessage(#{message:any(java.lang.String)}).isTrue()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertTrue");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertTrue(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertFalse}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertFalseRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertFalseRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertFalse`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertFalse {\n    \n    @BeforeTemplate\n    void before(boolean condition) {\n        assertFalse(condition);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(boolean condition) {\n        assertThat(condition).isFalse();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertFalse(#{condition:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{condition:any(boolean)}).isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertFalse");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertFalse(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertFalseWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertFalseWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertFalseWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertFalseWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertFalseWithMessage {\n    \n    @BeforeTemplate\n    void before(boolean condition, String message) {\n        assertFalse(condition, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(boolean condition, String message) {\n        assertThat(condition).withFailMessage(message).isFalse();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertFalse(#{condition:any(boolean)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{condition:any(boolean)}).withFailMessage(#{message:any(java.lang.String)}).isFalse()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertFalse");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertFalse(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertNull}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertNullRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertNullRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertNull`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertNull {\n    \n    @BeforeTemplate\n    void before(Object object) {\n        assertNull(object);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object object) {\n        assertThat(object).isNull();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNull(#{object:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{object:any(java.lang.Object)}).isNull()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNull");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNull(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertNullWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertNullWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertNullWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertNullWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertNullWithMessage {\n    \n    @BeforeTemplate\n    void before(Object object, String message) {\n        assertNull(object, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object object, String message) {\n        assertThat(object).withFailMessage(message).isNull();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNull(#{object:any(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{object:any(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).isNull()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNull");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNull(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertNotNull}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertNotNullRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertNotNullRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertNotNull`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertNotNull {\n    \n    @BeforeTemplate\n    void before(Object object) {\n        assertNotNull(object);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object object) {\n        assertThat(object).isNotNull();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotNull(#{object:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{object:any(java.lang.Object)}).isNotNull()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotNull");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotNull(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertNotNullWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertNotNullWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertNotNullWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertNotNullWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertNotNullWithMessage {\n    \n    @BeforeTemplate\n    void before(Object object, String message) {\n        assertNotNull(object, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object object, String message) {\n        assertThat(object).withFailMessage(message).isNotNull();\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotNull(#{object:any(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{object:any(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).isNotNull()")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotNull");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotNull(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertSame}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertSameRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertSameRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertSame`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertSame {\n    \n    @BeforeTemplate\n    void before(Object actual, Object expected) {\n        assertSame(actual, expected);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, Object expected) {\n        assertThat(actual).isSameAs(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertSame(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).isSameAs(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertSame");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertSame(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertSameWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertSameWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertSameWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertSameWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertSameWithMessage {\n    \n    @BeforeTemplate\n    void before(Object actual, String message, Object expected) {\n        assertSame(actual, expected, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, String message, Object expected) {\n        assertThat(actual).withFailMessage(message).isSameAs(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertSame(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).isSameAs(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertSame");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertSame(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertNotSame}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertNotSameRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertNotSameRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertNotSame`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertNotSame {\n    \n    @BeforeTemplate\n    void before(Object actual, Object expected) {\n        assertNotSame(actual, expected);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, Object expected) {\n        assertThat(actual).isNotSameAs(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotSame(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).isNotSameAs(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotSame");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotSame(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertNotSameWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertNotSameWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertNotSameWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertNotSameWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertNotSameWithMessage {\n    \n    @BeforeTemplate\n    void before(Object actual, String message, Object expected) {\n        assertNotSame(actual, expected, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, String message, Object expected) {\n        assertThat(actual).withFailMessage(message).isNotSameAs(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotSame(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).isNotSameAs(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotSame");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotSame(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqual}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqual`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqual {\n    \n    @BeforeTemplate\n    void before(boolean actual, boolean expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(byte actual, byte expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(char actual, char expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(short actual, short expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(int actual, int expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(long actual, long expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(float actual, float expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(double actual, double expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(Object actual, Object expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(String actual, String expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(Map<?, ?> actual, Map<?, ?> expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, Object expected) {\n        assertThat(actual).isEqualTo(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(boolean)}, #{expected:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(byte)}, #{expected:any(byte)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(char)}, #{expected:any(char)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before2 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(short)}, #{expected:any(short)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before3 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(int)}, #{expected:any(int)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before4 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(long)}, #{expected:any(long)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before5 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(float)}, #{expected:any(float)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before6 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(double)}, #{expected:any(double)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before7 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before8 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(java.lang.String)}, #{expected:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before9 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(java.util.Map<?,?>)}, #{expected:any(java.util.Map<?,?>)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).isEqualTo(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before2.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before3.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before4.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before5.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before6.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before7.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before8.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before9.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Map");
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.Map", true),
                        new UsesMethod<>("org.testng.Assert assertEquals(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualWithMessage {\n    \n    @BeforeTemplate\n    void before(boolean actual, String message, boolean expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(byte actual, String message, byte expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(char actual, String message, char expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(short actual, String message, short expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(int actual, String message, int expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(long actual, String message, long expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(float actual, String message, float expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(double actual, String message, double expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(Object actual, String message, Object expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(String actual, String message, String expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(Map<?, ?> actual, String message, Map<?, ?> expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, String message, Object expected) {\n        assertThat(actual).withFailMessage(message).isEqualTo(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(boolean)}, #{expected:any(boolean)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(byte)}, #{expected:any(byte)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(char)}, #{expected:any(char)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before2 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(short)}, #{expected:any(short)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before3 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(int)}, #{expected:any(int)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before4 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(long)}, #{expected:any(long)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before5 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(float)}, #{expected:any(float)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before6 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(double)}, #{expected:any(double)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before7 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before8 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(java.lang.String)}, #{expected:any(java.lang.String)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before9 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(java.util.Map<?,?>)}, #{expected:any(java.util.Map<?,?>)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).isEqualTo(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before2.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before3.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before4.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before5.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before6.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before7.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before8.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before9.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Map");
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.Map", true),
                        new UsesMethod<>("org.testng.Assert assertEquals(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualFloatsWithDelta}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualFloatsWithDeltaRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualFloatsWithDeltaRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualFloatsWithDelta`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualFloatsWithDelta {\n    \n    @BeforeTemplate\n    void before(float actual, float expected, float delta) {\n        assertEquals(actual, expected, delta);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(float actual, float expected, float delta) {\n        assertThat(actual).isCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(float)}, #{expected:any(float)}, #{delta:any(float)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(float)}).isCloseTo(#{expected:any(float)}, org.assertj.core.data.Offset.offset(#{delta:any(float)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualFloatsWithDeltaWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualFloatsWithDeltaWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualFloatsWithDeltaWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualFloatsWithDeltaWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualFloatsWithDeltaWithMessage {\n    \n    @BeforeTemplate\n    void before(float actual, String message, float expected, float delta) {\n        assertEquals(actual, expected, delta, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(float actual, String message, float expected, float delta) {\n        assertThat(actual).withFailMessage(message).isCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(float)}, #{expected:any(float)}, #{delta:any(float)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(float)}).withFailMessage(#{message:any(java.lang.String)}).isCloseTo(#{expected:any(float)}, org.assertj.core.data.Offset.offset(#{delta:any(float)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2), matcher.parameter(3)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualDoublesWithDelta}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualDoublesWithDeltaRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualDoublesWithDeltaRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualDoublesWithDelta`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualDoublesWithDelta {\n    \n    @BeforeTemplate\n    void before(double actual, double expected, double delta) {\n        assertEquals(actual, expected, delta);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(double actual, double expected, double delta) {\n        assertThat(actual).isCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(double)}, #{expected:any(double)}, #{delta:any(double)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(double)}).isCloseTo(#{expected:any(double)}, org.assertj.core.data.Offset.offset(#{delta:any(double)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualDoublesWithDeltaWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualDoublesWithDeltaWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualDoublesWithDeltaWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualDoublesWithDeltaWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualDoublesWithDeltaWithMessage {\n    \n    @BeforeTemplate\n    void before(double actual, String message, double expected, double delta) {\n        assertEquals(actual, expected, delta, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(double actual, String message, double expected, double delta) {\n        assertThat(actual).withFailMessage(message).isCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:any(double)}, #{expected:any(double)}, #{delta:any(double)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(double)}).withFailMessage(#{message:any(java.lang.String)}).isCloseTo(#{expected:any(double)}, org.assertj.core.data.Offset.offset(#{delta:any(double)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2), matcher.parameter(3)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualArrayIterationOrder}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualArrayIterationOrderRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualArrayIterationOrderRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualArrayIterationOrder`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualArrayIterationOrder {\n    \n    @BeforeTemplate\n    void before(boolean[] actual, boolean[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(byte[] actual, byte[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(char[] actual, char[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(short[] actual, short[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(int[] actual, int[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(long[] actual, long[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(float[] actual, float[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(double[] actual, double[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(Object[] actual, Object[] expected) {\n        assertEquals(actual, expected);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object[] actual, Object[] expected) {\n        assertThat(actual).containsExactly(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(boolean)}, #{expected:anyArray(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(byte)}, #{expected:anyArray(byte)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(char)}, #{expected:anyArray(char)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before2 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(short)}, #{expected:anyArray(short)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before3 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(int)}, #{expected:anyArray(int)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before4 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(long)}, #{expected:anyArray(long)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before5 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(float)}, #{expected:anyArray(float)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before6 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(double)}, #{expected:anyArray(double)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before7 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(java.lang.Object)}, #{expected:anyArray(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:anyArray(java.lang.Object)}).containsExactly(#{expected:anyArray(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before2.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before3.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before4.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before5.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before6.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before7.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualArrayIterationOrderWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualArrayIterationOrderWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualArrayIterationOrderWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualArrayIterationOrderWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualArrayIterationOrderWithMessage {\n    \n    @BeforeTemplate\n    void before(boolean[] actual, String message, boolean[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(byte[] actual, String message, byte[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(char[] actual, String message, char[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(short[] actual, String message, short[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(int[] actual, String message, int[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(long[] actual, String message, long[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(float[] actual, String message, float[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(double[] actual, String message, double[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(Object[] actual, String message, Object[] expected) {\n        assertEquals(actual, expected, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object[] actual, String message, Object[] expected) {\n        assertThat(actual).withFailMessage(message).containsExactly(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(boolean)}, #{expected:anyArray(boolean)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(byte)}, #{expected:anyArray(byte)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(char)}, #{expected:anyArray(char)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before2 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(short)}, #{expected:anyArray(short)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before3 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(int)}, #{expected:anyArray(int)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before4 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(long)}, #{expected:anyArray(long)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before5 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(float)}, #{expected:anyArray(float)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before6 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(double)}, #{expected:anyArray(double)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before7 = JavaTemplate
                        .builder("org.testng.Assert.assertEquals(#{actual:anyArray(java.lang.Object)}, #{expected:anyArray(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:anyArray(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).containsExactly(#{expected:anyArray(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before2.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before3.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before4.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before5.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before6.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before7.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualArraysIrrespectiveOfOrder}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualArraysIrrespectiveOfOrderRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualArraysIrrespectiveOfOrderRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualArraysIrrespectiveOfOrder`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualArraysIrrespectiveOfOrder {\n    \n    @BeforeTemplate\n    void before(Object[] actual, Object[] expected) {\n        assertEqualsNoOrder(actual, expected);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object[] actual, Object[] expected) {\n        assertThat(actual).containsExactlyInAnyOrder(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEqualsNoOrder(#{actual:anyArray(java.lang.Object)}, #{expected:anyArray(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:anyArray(java.lang.Object)}).containsExactlyInAnyOrder(#{expected:anyArray(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEqualsNoOrder");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEqualsNoOrder(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertEqualArraysIrrespectiveOfOrderWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertEqualArraysIrrespectiveOfOrderWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertEqualArraysIrrespectiveOfOrderWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertEqualArraysIrrespectiveOfOrderWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertEqualArraysIrrespectiveOfOrderWithMessage {\n    \n    @BeforeTemplate\n    void before(Object[] actual, String message, Object[] expected) {\n        assertEqualsNoOrder(actual, expected, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object[] actual, String message, Object[] expected) {\n        assertThat(actual).withFailMessage(message).containsExactlyInAnyOrder(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertEqualsNoOrder(#{actual:anyArray(java.lang.Object)}, #{expected:anyArray(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:anyArray(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).containsExactlyInAnyOrder(#{expected:anyArray(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertEqualsNoOrder");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertEqualsNoOrder(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertUnequal}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertUnequalRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertUnequalRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertUnequal`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertUnequal {\n    \n    @BeforeTemplate\n    void before(boolean actual, boolean expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(byte actual, byte expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(char actual, char expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(short actual, short expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(int actual, int expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(long actual, long expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(float actual, float expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(double actual, double expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(Object actual, Object expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(String actual, String expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(Set<?> actual, Set<?> expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @BeforeTemplate\n    void before(Map<?, ?> actual, Map<?, ?> expected) {\n        assertNotEquals(actual, expected);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, Object expected) {\n        assertThat(actual).isNotEqualTo(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(boolean)}, #{expected:any(boolean)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(byte)}, #{expected:any(byte)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(char)}, #{expected:any(char)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before2 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(short)}, #{expected:any(short)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before3 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(int)}, #{expected:any(int)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before4 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(long)}, #{expected:any(long)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before5 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(float)}, #{expected:any(float)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before6 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(double)}, #{expected:any(double)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before7 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before8 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.lang.String)}, #{expected:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before9 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.util.Set<?>)}, #{expected:any(java.util.Set<?>)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before10 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.util.Map<?,?>)}, #{expected:any(java.util.Map<?,?>)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).isNotEqualTo(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before2.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before3.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before4.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before5.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before6.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before7.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before8.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before9.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Set");
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before10.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Map");
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.Map", true),
                        new UsesMethod<>("org.testng.Assert assertNotEquals(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertUnequalWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertUnequalWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertUnequalWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertUnequalWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertUnequalWithMessage {\n    \n    @BeforeTemplate\n    void before(boolean actual, String message, boolean expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(byte actual, String message, byte expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(char actual, String message, char expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(short actual, String message, short expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(int actual, String message, int expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(long actual, String message, long expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(float actual, String message, float expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(double actual, String message, double expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(Object actual, String message, Object expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(String actual, String message, String expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(Set<?> actual, String message, Set<?> expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @BeforeTemplate\n    void before(Map<?, ?> actual, String message, Map<?, ?> expected) {\n        assertNotEquals(actual, expected, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(Object actual, String message, Object expected) {\n        assertThat(actual).withFailMessage(message).isNotEqualTo(expected);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(boolean)}, #{expected:any(boolean)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before0 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(byte)}, #{expected:any(byte)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before1 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(char)}, #{expected:any(char)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before2 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(short)}, #{expected:any(short)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before3 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(int)}, #{expected:any(int)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before4 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(long)}, #{expected:any(long)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before5 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(float)}, #{expected:any(float)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before6 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(double)}, #{expected:any(double)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before7 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.lang.Object)}, #{expected:any(java.lang.Object)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before8 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.lang.String)}, #{expected:any(java.lang.String)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before9 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.util.Set<?>)}, #{expected:any(java.util.Set<?>)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate before10 = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(java.util.Map<?,?>)}, #{expected:any(java.util.Map<?,?>)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(java.lang.Object)}).withFailMessage(#{message:any(java.lang.String)}).isNotEqualTo(#{expected:any(java.lang.Object)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before0.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before1.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before2.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before3.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before4.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before5.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before6.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before7.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before8.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before9.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Set");
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    if ((matcher = before10.matcher(getCursor())).find()) {
                        maybeRemoveImport("java.util.Map");
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("java.util.Map", true),
                        new UsesMethod<>("org.testng.Assert assertNotEquals(..)")
                    ),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertUnequalFloatsWithDelta}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertUnequalFloatsWithDeltaRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertUnequalFloatsWithDeltaRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertUnequalFloatsWithDelta`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertUnequalFloatsWithDelta {\n    \n    @BeforeTemplate\n    void before(float actual, float expected, float delta) {\n        assertNotEquals(actual, expected, delta);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(float actual, float expected, float delta) {\n        assertThat(actual).isNotCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(float)}, #{expected:any(float)}, #{delta:any(float)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(float)}).isNotCloseTo(#{expected:any(float)}, org.assertj.core.data.Offset.offset(#{delta:any(float)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertUnequalFloatsWithDeltaWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertUnequalFloatsWithDeltaWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertUnequalFloatsWithDeltaWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertUnequalFloatsWithDeltaWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertUnequalFloatsWithDeltaWithMessage {\n    \n    @BeforeTemplate\n    void before(float actual, String message, float expected, float delta) {\n        assertNotEquals(actual, expected, delta, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(float actual, String message, float expected, float delta) {\n        assertThat(actual).withFailMessage(message).isNotCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(float)}, #{expected:any(float)}, #{delta:any(float)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(float)}).withFailMessage(#{message:any(java.lang.String)}).isNotCloseTo(#{expected:any(float)}, org.assertj.core.data.Offset.offset(#{delta:any(float)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2), matcher.parameter(3)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertUnequalDoublesWithDelta}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertUnequalDoublesWithDeltaRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertUnequalDoublesWithDeltaRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertUnequalDoublesWithDelta`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertUnequalDoublesWithDelta {\n    \n    @BeforeTemplate\n    void before(double actual, double expected, double delta) {\n        assertNotEquals(actual, expected, delta);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(double actual, double expected, double delta) {\n        assertThat(actual).isNotCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(double)}, #{expected:any(double)}, #{delta:any(double)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(double)}).isNotCloseTo(#{expected:any(double)}, org.assertj.core.data.Offset.offset(#{delta:any(double)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertUnequalDoublesWithDeltaWithMessage}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertUnequalDoublesWithDeltaWithMessageRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertUnequalDoublesWithDeltaWithMessageRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertUnequalDoublesWithDeltaWithMessage`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertUnequalDoublesWithDeltaWithMessage {\n    \n    @BeforeTemplate\n    void before(double actual, String message, double expected, double delta) {\n        assertNotEquals(actual, expected, delta, message);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(double actual, String message, double expected, double delta) {\n        assertThat(actual).withFailMessage(message).isNotCloseTo(expected, offset(delta));\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertNotEquals(#{actual:any(double)}, #{expected:any(double)}, #{delta:any(double)}, #{message:any(java.lang.String)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThat(#{actual:any(double)}).withFailMessage(#{message:any(java.lang.String)}).isNotCloseTo(#{expected:any(double)}, org.assertj.core.data.Offset.offset(#{delta:any(double)}))")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.assertNotEquals");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0), matcher.parameter(1), matcher.parameter(2), matcher.parameter(3)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    new UsesMethod<>("org.testng.Assert assertNotEquals(..)"),
                    javaVisitor
            );
        }
    }

    /**
     * OpenRewrite recipe created for Refaster template {@code TestNGToAssertJRules.AssertThrows}.
     */
    @SuppressWarnings("all")
    @NonNullApi
    public static class AssertThrowsRecipe extends Recipe {

        /**
         * Instantiates a new instance.
         */
        public AssertThrowsRecipe() {}

        @Override
        public String getDisplayName() {
            return "Refaster template `TestNGToAssertJRules.AssertThrows`";
        }

        @Override
        public String getDescription() {
            return "Recipe created for the following Refaster template:\n```java\nstatic final class AssertThrows {\n    \n    @BeforeTemplate\n    void before(ThrowingRunnable runnable) {\n        assertThrows(runnable);\n    }\n    \n    @AfterTemplate\n    @UseImportPolicy(value = STATIC_IMPORT_ALWAYS)\n    void after(ThrowingCallable runnable) {\n        assertThatThrownBy(runnable);\n    }\n}\n```\n.";
        }

        @Override
        public TreeVisitor<?, ExecutionContext> getVisitor() {
            JavaVisitor<ExecutionContext> javaVisitor = new AbstractRefasterJavaVisitor() {
                final JavaTemplate before = JavaTemplate
                        .builder("org.testng.Assert.assertThrows(#{runnable:any(org.testng.Assert.ThrowingRunnable)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("testng"))
                        .build();
                final JavaTemplate after = JavaTemplate
                        .builder("org.assertj.core.api.Assertions.assertThatThrownBy(#{runnable:any(org.assertj.core.api.ThrowableAssert.ThrowingCallable)})")
                        .javaParser(JavaParser.fromJavaVersion().classpath("assertj-core"))
                        .build();

                @Override
                public J visitMethodInvocation(J.MethodInvocation elem, ExecutionContext ctx) {
                    JavaTemplate.Matcher matcher;
                    if ((matcher = before.matcher(getCursor())).find()) {
                        maybeRemoveImport("org.testng.Assert.ThrowingRunnable");
                        maybeRemoveImport("org.testng.Assert.assertThrows");
                        return embed(
                                after.apply(getCursor(), elem.getCoordinates().replace(), matcher.parameter(0)),
                                getCursor(),
                                ctx,
                                SHORTEN_NAMES
                        );
                    }
                    return super.visitMethodInvocation(elem, ctx);
                }

            };
            return Preconditions.check(
                    Preconditions.and(
                        new UsesType<>("org.testng.Assert.ThrowingRunnable", true),
                        new UsesMethod<>("org.testng.Assert assertThrows(..)")
                    ),
                    javaVisitor
            );
        }
    }

}
