package tech.prodigio.core.libeventproducer.factory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.core.ApiFuture;
import com.google.cloud.pubsub.v1.Publisher;
import com.google.protobuf.ByteString;
import com.google.pubsub.v1.PubsubMessage;
import lombok.extern.log4j.Log4j2;
import tech.prodigio.core.libcorebase.config.BaseConfig;
import tech.prodigio.core.libeventproducer.event.PubSubEvent;
import tech.prodigio.core.libeventproducer.exception.PublishEventException;
import tech.prodigio.core.libeventproducer.handler.PubSubPublisherEventHandler;
import tech.prodigio.core.libeventproducer.handler.PublisherEventHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;

@Log4j2
public abstract class PubSubFactory implements PublisherEventHandlerFactory {

    private final ObjectMapper objectMapper = new BaseConfig().getObjectMapper();

    @Override
    public PublisherEventHandler buildHandler() {
        return PubSubPublisherEventHandler.eventBuilder()
                .publisher(getPublisher())
                .publishFunction(this::publishEvent)
                .build();
    }


    public abstract Publisher getPublisher();

    public abstract PubSubEvent event();

    private String publishEvent(Publisher publisher) {
        try {
            String message = toJson(event().getData());
            log.info("Starting message publishing to Pub/Sub, Body : {}", message);
            ByteString data = ByteString.copyFromUtf8(message);
            PubsubMessage pubsubMessage = PubsubMessage.newBuilder()
                    .setData(data)
                    .putAllAttributes(buildAttributes(event().getAttributes()))
                    .build();

            long startTime = System.nanoTime();

            ApiFuture<String> result = publisher.publish(pubsubMessage);
            String messageId = result.get();

            publisher.shutdown();
            long endTime = System.nanoTime();
            long duration = endTime - startTime; //nanoseconds
            double milliseconds = duration / 1_000_000.0;

            log.info("Published in {} milliseconds", milliseconds);


            return messageId;
        } catch (InterruptedException e) {
            log.error("InterruptedException : {0}", e);
            throw new PublishEventException(e);
        } catch (ExecutionException e) {
            log.error("ExecutionException : {0}", e);
            throw new PublishEventException(e);
        }
    }

    private Map<String, String> buildAttributes(Map<String, String> messageAttributes) {
        Map<String, String> attributes = new HashMap<>();

        attributes.put("event-id", event().getEventId());

        if (Objects.nonNull(messageAttributes)) {
            attributes.putAll(messageAttributes);
        }
        return attributes;
    }

    public String toJson(Object data) {
        try {
            return objectMapper.writeValueAsString(data);
        } catch (JsonProcessingException e) {
            log.error("Could not convert object to json, Error : {}", e.getMessage());
            throw new PublishEventException(e);
        }
    }
}
