package tech.prodigio.core.libeventproducer.service.impl;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import tech.prodigio.core.libeventproducer.core.Publisher;
import tech.prodigio.core.libeventproducer.enums.EventSourceType;
import tech.prodigio.core.libeventproducer.event.Event;
import tech.prodigio.core.libeventproducer.event.EventHandlerPlatform;
import tech.prodigio.core.libeventproducer.event.PubSubEvent;
import tech.prodigio.core.libeventproducer.exception.NoSupportedEventException;
import tech.prodigio.core.libeventproducer.handler.PublisherEventHandler;
import tech.prodigio.core.libeventproducer.service.IEventService;

import java.util.List;
import java.util.Map;
import java.util.Objects;

@Service
@RequiredArgsConstructor
public class EventService implements IEventService {

    private final List<Publisher> publishers;
    private final List<Class<? extends Event>> supportedEvents = List.of(PubSubEvent.class, EventHandlerPlatform.class);
    private final Map<EventSourceType, List<Class<? extends Event>>> publisherOptions = Map.of(
            EventSourceType.GCP_PUBLISHER, supportedEvents
    );


    @Override
    public PublisherEventHandler buildPublisherHandler(Event event) {
        return getPublisher(event)
                .buildHandler(event);
    }

    private Publisher getPublisher(Event event) {
        EventSourceType eventSourceType = getPublisherType(event);
        return publishers.stream().filter(publisher1 -> eventSourceType.equals(publisher1.getEventSourceType()))
                .findFirst()
                .orElseThrow();
    }

    private EventSourceType getPublisherType(Event event) {
        return publisherOptions.entrySet().stream()
                .filter(entry -> entry.getValue().stream().anyMatch(option -> validClass(event, option)))
                .findFirst()
                .map(Map.Entry::getKey)
                .orElseThrow(() -> new NoSupportedEventException(supportedEvents));
    }


    private static boolean validClass(Event event, Class<? extends Event> option) {
        return Objects.equals(option.getName(), event.getClass().getName());
    }


}
