package top.binfast.common.desensitise.handler;

import cn.hutool.extra.spring.SpringUtil;
import top.binfast.common.desensitise.annotation.Desensitise;
import top.binfast.common.desensitise.core.DesensitiseService;
import top.binfast.common.desensitise.core.DesensitiseStrategy;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.ser.ContextualSerializer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;

import java.io.IOException;
import java.util.Objects;

/**
 * 数据脱敏json序列化工具
 *
 * @author liubin
 */
@Slf4j
public class DesensitiseHandler extends JsonSerializer<String> implements ContextualSerializer {

    private DesensitiseStrategy strategy;

    @Override
    public void serialize(String value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
        try {
            if (strategy == null) {
                gen.writeString(value);
            } else {
                DesensitiseService desensitiseService = SpringUtil.getBean(DesensitiseService.class);
                boolean isDesensitise = desensitiseService != null && desensitiseService.isDesensitise();
                if (!isDesensitise) {
                    gen.writeString(value);
                } else {
                    gen.writeString(strategy.desensitizer().apply(value));
                }
            }
        } catch (BeansException e) {
            log.error("脱敏实现不存在, 采用默认处理 => {}", e.getMessage());
            gen.writeString(value);
        }
    }

    @Override
    public JsonSerializer<?> createContextual(SerializerProvider prov, BeanProperty property) throws JsonMappingException {
        Desensitise annotation = property.getAnnotation(Desensitise.class);
        if (Objects.nonNull(annotation) && Objects.equals(String.class, property.getType().getRawClass())) {
            this.strategy = annotation.value();
            return this;
        }
        return prov.findValueSerializer(property.getType(), property);
    }
}
