package top.coos.extra.es;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.elasticsearch.action.ActionFuture;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsRequest;
import org.elasticsearch.action.admin.indices.exists.types.TypesExistsRequest;
import org.elasticsearch.action.admin.indices.flush.FlushRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.admin.indices.open.OpenIndexAction;
import org.elasticsearch.action.admin.indices.open.OpenIndexRequest;
import org.elasticsearch.action.admin.indices.open.OpenIndexRequestBuilder;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.action.support.WriteRequest.RefreshPolicy;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.transport.NoNodeAvailableException;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.transport.client.PreBuiltTransportClient;

import top.coos.util.StringUtil;

/**
 * ES工具类
 * 
 * @author ZRH
 *
 */
public class ESUtils {

	static Pattern badChars = Pattern.compile("\\s*[\\s~!\\^&\\(\\)\\-\\+:\\|\\\\\"\\\\$]+\\s*");

	private ESUtils() {

	}

	/**
	 * 关闭对应client
	 * 
	 * @param client
	 */
	public static void close(Client client) {

		if (client != null) {
			try {
				client.close();
			} catch (Exception e) {
			}
			client = null;
		}
	}

	public static void flush(Client client, String indexName, String indexType) {

		try {
			client.admin().indices().flush(new FlushRequest(indexName.toLowerCase(), indexType)).actionGet();
		} catch (Exception e) {
		}
	}

	/**
	 * 初始化并连接elasticsearch集群，返回连接后的client
	 * 
	 * @param clusterName
	 *            中心节点名称
	 * @param port
	 *            节点端口
	 * @param hostname
	 *            集群节点所在服务器IP，支持多个
	 * @return 返回连接的集群的client
	 */
	public static Client create(String clusterName, String ip, int port) throws UnknownHostException {

		Settings esSettings = Settings.builder()
				.put("cluster.name", clusterName) // 设置ES实例的名称
				.put("client.transport.sniff", false) // 自动嗅探整个集群的状态，把集群中其他ES节点的ip添加到本地的客户端列表中
				.build();
		TransportClient client = new PreBuiltTransportClient(esSettings);// 初始化client较老版本发生了变化，此方法有几个重载方法，初始化插件等。

		client.addTransportAddress(new TransportAddress(InetAddress.getByName(ip), port));
		return client;
	}

	/**
	 * 验证index是否创建
	 * 
	 * @param client
	 * @param indexName
	 * @return
	 */
	public static boolean indicesExists(Client client, String indexName) {

		IndicesExistsRequest ier = new IndicesExistsRequest();
		ier.indices(new String[] { indexName.toLowerCase() });

		return client.admin().indices().exists(ier).actionGet().isExists();
	}

	/**
	 * 验证type是否存在
	 * 
	 * @param client
	 * @param indexName
	 * @param indexType
	 * @return
	 */
	public static boolean typesExists(Client client, String indexName, String indexType) {

		if (indicesExists(client, indexName)) {
			TypesExistsRequest ter = new TypesExistsRequest(new String[] { indexName.toLowerCase() }, indexType);
			return client.admin().indices().typesExists(ter).actionGet().isExists();
		}
		return false;
	}

	/**
	 * 根据索引数据id删除索引
	 * 
	 * @param indexName
	 *            索引名称
	 * @param indexType
	 *            索引类型
	 * @param id
	 *            对应数据ID
	 */
	public static void deleteIndex(Client client, String indexName, String indexType, String id) {

		try {
			client.prepareDelete(indexName.toLowerCase(), indexType.toLowerCase(), id)
					.setRefreshPolicy(RefreshPolicy.IMMEDIATE).execute().actionGet();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * 根据索引名称删除索引
	 * 
	 * @param indexName
	 *            索引名称
	 */
	public static void deleteIndex(Client client, String indexName) {

		IndicesExistsRequest ier = new IndicesExistsRequest();
		ier.indices(new String[] { indexName.toLowerCase() });

		boolean exists = client.admin().indices().exists(ier).actionGet().isExists();
		if (exists) {
			client.admin().indices().prepareDelete(indexName.toLowerCase()).execute().actionGet();
		}

	}

	public static SearchHits search(Client client, String indexName, List<String> indexTypes, QueryBuilder query,
			List<FieldSortBuilder> sortBuilders, int from, int size) throws NoNodeAvailableException {

		if (client == null) {
			return null;
		}
		indexName = indexName.toLowerCase();

		// 去掉不存在的索引
		IndicesExistsRequest ier = new IndicesExistsRequest();
		ier.indices(new String[] { indexName });
		boolean exists = client.admin().indices().exists(ier).actionGet().isExists();
		if (exists) {
			client.admin().indices().open(new OpenIndexRequest(indexName)).actionGet();
		} else {
			return null;
		}

		client.admin().indices().refresh(new RefreshRequest(indexName)).actionGet();

		SearchRequestBuilder searchRequestBuilder = client.prepareSearch(indexName);

		if (indexTypes != null && indexTypes.size() > 0) {
			String[] types = new String[indexTypes.size()];
			for (int i = 0; i < indexTypes.size(); i++) {
				types[i] = indexTypes.get(i).toLowerCase();
			}
			searchRequestBuilder.setTypes(types);
		}

		searchRequestBuilder.setSearchType(SearchType.DFS_QUERY_THEN_FETCH);
		searchRequestBuilder.setFrom(from);
		searchRequestBuilder.setSize(size);
		searchRequestBuilder.setExplain(false);
		searchRequestBuilder.setQuery(query);
		if (sortBuilders != null && sortBuilders.size() > 0) {
			for (FieldSortBuilder sortBuilder : sortBuilders) {
				searchRequestBuilder.addSort(sortBuilder);
			}
		}

		return searchRequestBuilder.execute().actionGet().getHits();
	}

	public static List<Map<String, Object>> queryAll(Client client, String indexName, String indexType, QueryBean bean)
	{

		List<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
		queryFull(client, indexName, indexType, bean, result);
		return result;
	}

	public static void queryFull(Client client, String indexName, String indexType, QueryBean bean,
			List<Map<String, Object>> result) {

		List<Map<String, Object>> list = queryPage(client, indexName, indexType, bean);
		if (list != null) {
			result.addAll(list);
			if (list.size() == bean.getPagesize()) {
				bean.setCurrentpage(bean.getCurrentpage() + 1);
				queryFull(client, indexName, indexType, bean, result);
			}
		}

	}

	public static List<Map<String, Object>> queryPage(Client client, String indexName, String indexType, QueryBean bean) {

		if (!ESUtils.typesExists(client, indexName, indexType)) {
			return null;
		}
		SearchRequestBuilder builder = client.prepareSearch(indexName);
		BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
		builder.setTypes(indexType);

		if (bean != null) {
			for (String key : bean.getWhere().keySet()) {
				Object value = bean.getWhere().get(key);
				if (value != null) {
					QueryBuilder queryBuilder = null;
					if (value instanceof String) {
						queryBuilder = QueryBuilders.matchQuery(key, value);
					} else {
						queryBuilder = QueryBuilders.termQuery(key, value);
					}
					boolQueryBuilder.must(queryBuilder);
				}
			}

			builder.setQuery(boolQueryBuilder);
			int pagesize = bean.getPagesize();
			int currentpage = bean.getCurrentpage();
			if (pagesize <= 0) {
				pagesize = 50;
			}
			if (currentpage <= 0) {
				currentpage = 1;
			}
			bean.setPagesize(pagesize);
			bean.setCurrentpage(currentpage);
			int from = (pagesize * (currentpage - 1));
			builder.setFrom(from);
			builder.setSize(pagesize);
		}
		SearchResponse response =
				builder.execute()
						.actionGet();
		SearchHits searchHits = response.getHits();
		SearchHit[] hits = searchHits.getHits();
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		for (SearchHit searchHit : hits) {
			Map<String, Object> source = searchHit.getSourceAsMap();
			list.add(source);

		}
		return list;
	}

	public static long queryCount(Client client, String indexName, String indexType, Map<String, Object> where) {

		if (!ESUtils.typesExists(client, indexName, indexType)) {
			return 0;
		}
		SearchRequestBuilder searchBuilder = client.prepareSearch(indexName)
				.setTypes(indexType)
				.setSize(0);// 设置返回结果集为0
		BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
		for (String key : where.keySet()) {
			boolQueryBuilder.must(QueryBuilders.termsQuery(key, where.get(key)));
		}
		searchBuilder.setQuery(boolQueryBuilder);
		SearchResponse response = searchBuilder.get();
		long length = response.getHits().getTotalHits();
		return length;
	}

	/**
	 * 查询数据
	 * 
	 * @param indexName
	 *            索引名称
	 * @param indexType
	 *            索引类型
	 * @param id
	 *            数据id
	 * @return 如果不存在，返回<code>null</code>
	 */
	public static Map<String, Object> query(Client client, String indexName, String indexType, String id) {

		if (client == null) {
			return null;
		}
		if (StringUtil.isEmpty(indexName) || StringUtil.isEmpty(indexType) || StringUtil.isEmpty(id)) {
			return null;
		}
		indexName = indexName.toLowerCase();
		indexType = indexType.toLowerCase();

		IndicesExistsRequest ier = new IndicesExistsRequest();
		ier.indices(new String[] { indexName });
		boolean exists = client.admin().indices().exists(ier).actionGet().isExists();
		if (!exists) {
			// 索引不存在
			return null;
		}

		client.admin().indices().open(new OpenIndexRequest(indexName)).actionGet();
		client.admin().indices().refresh(new RefreshRequest(indexName)).actionGet();

		GetRequest gr = new GetRequest(indexName, indexType, id);

		ActionFuture<GetResponse> future = client.get(gr);
		GetResponse response = future.actionGet();
		return swapResult(response);
	}

	/**
	 * 添加数据到Elasticsearch
	 * 
	 * @param index
	 *            索引
	 * @param type
	 *            类型
	 * @param idName
	 *            Id字段名称
	 * @param listData
	 *            一个对象集合
	 * @return
	 */
	public static boolean insertOrUpdate(Client client, String indexName, String indexType, String id,
			Map<String, Object> data) {

		try {
			XContentBuilder jsonBuilder = XContentFactory.jsonBuilder();
			jsonBuilder.startObject();
			for (String name : data.keySet()) {
				jsonBuilder.field(name, data.get(name));
			}
			jsonBuilder.endObject();
			IndexRequest indexRequest = new IndexRequest(indexName, indexType, id).source(jsonBuilder);

			UpdateRequest updateRequest = new UpdateRequest(indexName, indexType, id);

			updateRequest.setRefreshPolicy(RefreshPolicy.IMMEDIATE);
			updateRequest.doc(jsonBuilder);
			updateRequest.upsert(indexRequest);
			client.update(updateRequest).actionGet();
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * 初始化索引
	 * 
	 * @param client
	 * @param indexName
	 * @param indexType
	 * @param cols
	 * @return 初始化成功,返回true；否则返回false
	 * @throws Exception
	 */
	public static boolean initIndex(Client client, String indexName, String
			indexType) throws Exception {

		if (StringUtil.isEmpty(indexName) || StringUtil.isEmpty(indexType)) {
			return false;
		}

		indexName = indexName.toLowerCase();
		indexType = indexType.toLowerCase();

		if (indicesExists(client, indexName)) {
			OpenIndexRequestBuilder openIndexBuilder = new
					OpenIndexRequestBuilder(client.admin().indices(),
							OpenIndexAction.INSTANCE);
			openIndexBuilder.setIndices(indexName).execute().actionGet();
		} else {
			client.admin().indices().prepareCreate(indexName).execute().actionGet();
		}

		TypesExistsRequest ter = new TypesExistsRequest(new String[] {
				indexName.toLowerCase() }, indexType);
		boolean typeExists =
				client.admin().indices().typesExists(ter).actionGet().isExists();

		if (typeExists) {
			return true;
		}

		PutMappingRequest mappingRequest =
				Requests.putMappingRequest(indexName).type(indexType);
		XContentBuilder builder = XContentFactory.jsonBuilder().startObject();
		builder.startObject(indexName).startObject("analysis").startObject("analyzer").startObject("default")
				.field("type", "keyword");
		mappingRequest.source(builder);
		PutMappingResponse response =
				client.admin().indices().putMapping(mappingRequest).actionGet();

		return response.isAcknowledged();
	}

	// public static SearchHits search(String indexName, String indexType,
	// String[] keywords, String[] channelIdArr, int from, int size) throws
	// NoNodeAvailableException, IndexMissingException {
	// if(client == null ) {
	// return null;
	// }
	//
	// // 去掉不存在的索引
	// IndicesExistsRequest ier = new IndicesExistsRequest();
	// ier.indices(new String[]{indexName});
	// boolean exists =
	// client.admin().indices().exists(ier).actionGet().isExists();
	// if(exists){
	// client.admin().indices().open(new
	// OpenIndexRequest(indexName)).actionGet();
	// }else{
	// Index index = new Index(indexName);
	// throw new IndexMissingException(index);
	// }
	//
	// try {
	// client.admin().indices().refresh(new
	// RefreshRequest(indexName)).actionGet();
	// } catch (IndexMissingException e) {
	// e.printStackTrace();
	// }
	//
	// SearchRequestBuilder searchRequestBuilder =
	// client.prepareSearch(indexName);
	//
	// searchRequestBuilder.setSearchType(SearchType.DFS_QUERY_THEN_FETCH);
	// searchRequestBuilder.setFrom(from);
	// searchRequestBuilder.setSize(size);
	// searchRequestBuilder.setExplain(true);
	//
	// BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
	//
	// StringBuffer totalKeys = new StringBuffer();
	// for(String keyword: keywords) {
	// totalKeys.append(keyword);
	// }
	//
	// if(!totalKeys.toString().equals("*")){
	// for(String keyword: keywords) {
	// if( keyword == null || keyword.trim().length() == 0 ) {
	// continue;
	// }
	// keyword = badChars.matcher(keyword).replaceAll("");
	// if( keyword == null || keyword.trim().length() == 0 ) {
	// continue;
	// }
	//
	// if(keyword.indexOf("*")!=-1 || keyword.indexOf("×")!=-1 ||
	// keyword.indexOf("?")!=-1 || keyword.indexOf("？")!=-1){
	// keyword = keyword.replaceAll("×", "*").replaceAll("？", "?");
	// BoolQueryBuilder subBoolQuery = QueryBuilders.boolQuery();
	// for(String indexColumnName: Content.indexColumnNames) {
	// subBoolQuery.should(QueryBuilders.wildcardQuery(indexColumnName.toLowerCase(),
	// keyword));
	// }
	// boolQuery.must(subBoolQuery);
	// }else{
	// QueryStringQueryBuilder qb =
	// QueryBuilders.queryString("\""+keyword+"\"");
	// boolQuery.must(qb);
	// }
	// }
	// }else {
	// //boolQuery.should(QueryBuilders.queryString("*"));
	// }
	//
	// if(channelIdArr!=null && channelIdArr.length>0){
	// TermsQueryBuilder inQuery = QueryBuilders.inQuery("channelid_",
	// channelIdArr);
	// boolQuery.must(inQuery);
	// }
	//
	// searchRequestBuilder.setQuery(boolQuery);
	//
	//
	// return searchRequestBuilder.execute().actionGet().getHits();
	// }

	public static String preReadString(String read, int maxLength) {

		if (read == null || read.trim().length() == 0) {
			return "";
		}
		read = read.trim();

		if (read.length() <= maxLength) {
			return read;
		}

		// if(keywords!=null && keywords.length>0){
		// for(String keyword: keywords) {
		// if( keyword == null || keyword.trim().length() == 0 ) {
		// continue;
		// }
		// keyword = badChars.matcher(keyword).replaceAll("");
		// int loc = read.indexOf(keyword);
		// if(loc != -1){
		// if(loc <= maxLength) {
		// return read.substring(0, maxLength);
		// }else{
		// int aft = read.length()-loc;
		// if(aft>(maxLength/2)){
		// return read.substring(loc-maxLength/2, loc+maxLength/2+1);
		// }else{
		// return read.substring(loc-maxLength+aft, loc+aft);
		// }
		// }
		// }
		// }
		// }
		return read.substring(0, maxLength);

	}

	public static List<Map<String, Object>> swapResult(SearchHits hits) {

		List<Map<String, Object>> datas = new ArrayList<Map<String, Object>>();

		if (hits == null || hits.getTotalHits() <= 0) {
			return datas;
		}

		for (int i = 0; i < hits.getHits().length; i++) {
			SearchHit hit = hits.getAt(i);

			Map<String, Object> rowData = hit.getSourceAsMap();
			rowData.put("_index", hit.getIndex());
			rowData.put("_type", hit.getType());
			rowData.put("_id", hit.getId());

			datas.add(rowData);
		}

		return datas;
	}

	public static Map<String, Object> swapResult(GetResponse response) {

		if (response == null || !response.isExists()) {
			return null;
		}

		Map<String, Object> rowData = response.getSourceAsMap();
		rowData.put("_index", response.getIndex());
		rowData.put("_type", response.getType());
		rowData.put("_id", response.getId());

		return rowData;
	}

}