package top.crystalx.sms.strategy.tencent;

import com.tencentcloudapi.common.Credential;
import com.tencentcloudapi.common.exception.TencentCloudSDKException;
import com.tencentcloudapi.common.profile.ClientProfile;
import com.tencentcloudapi.common.profile.HttpProfile;
import com.tencentcloudapi.sms.v20210111.SmsClient;
import com.tencentcloudapi.sms.v20210111.models.SendSmsRequest;
import com.tencentcloudapi.sms.v20210111.models.SendSmsResponse;
import com.tencentcloudapi.sms.v20210111.models.SendStatus;
import top.crystalx.sms.props.TencentProperty;
import top.crystalx.sms.core.SmsStrategy;
import top.crystalx.sms.models.R.SendResult;
import top.crystalx.sms.models.R.SmsResponse;
import top.crystalx.sms.models.tencent.TencentModel;

import java.util.HashMap;

import static top.crystalx.sms.constant.TencentConst.HMAC_SHA_256;

/**
 * @author SIYE
 */
public class TencentSms implements SmsStrategy<TencentProperty, TencentModel> {


    /**
     * 发送信息
     */
    @Override
    public SmsResponse sendSms(TencentModel model) {
        return this.sendSms(new TencentProperty(), model);
    }

    /**
     * 发送信息
     * 自定义 腾讯云配置
     */
    @Override
    public SmsResponse sendSms(TencentProperty config, TencentModel model) {
        SmsClient client = this.createClient(config);
        SendSmsRequest request = this.createRequest(model);
        SendSmsResponse response = this.getResponse(client, request);
        SmsResponse.Builder builder = SmsResponse.builder()
                .params(model.getParamsMap())
                .signName(model.getSignName());
        for (SendStatus sendStatus : response.getSendStatusSet()) {
            HashMap<String, String> sendStatusMap = new HashMap<>();
            sendStatus.toMap(sendStatusMap, "");
            SendResult.Builder resultBuilder = SendResult.builder()
                    .phoneNumber(sendStatusMap.get("PhoneNumber"))
                    .message(sendStatusMap.get("Message"));
            if (sendStatusMap.get("Code").equals("Ok")) {
                resultBuilder.statusCode(200);
            } else {
                resultBuilder.statusCode(400);
            }
            builder.sendResults(resultBuilder.build());
        }
        return builder.build();
    }

    /**
     * 通过腾讯云SecretId&SecretKey创建Client
     */
    private SmsClient createClient(TencentProperty config) {
        Credential cred = new Credential(config.getSecretId(), config.getSecretKey());
        HttpProfile httpProfile = new HttpProfile();
        if (!config.getMethod().isEmpty()) {
            httpProfile.setReqMethod(config.getMethod());
        }
        if (config.getConnectTimeout() > 0) {
            httpProfile.setConnTimeout(config.getConnectTimeout());
        }
        httpProfile.setEndpoint(config.getEndpoint().endpoint());
        ClientProfile clientProfile = new ClientProfile();
        clientProfile.setSignMethod(HMAC_SHA_256);
        clientProfile.setHttpProfile(httpProfile);
        return new SmsClient(cred, config.getEndpoint().region(), clientProfile);
    }

    /**
     * 创建request
     */
    private SendSmsRequest createRequest(TencentModel model) {
        SendSmsRequest request = new SendSmsRequest();
        request.setSmsSdkAppId(model.getSdkAppId());
        request.setSignName(model.getSignName());
        request.setTemplateId(model.getTemplateId());
        request.setTemplateParamSet(model.getParams().toArray(new String[0]));
        request.setPhoneNumberSet(model.getPhoneNumbers().toArray(new String[0]));
        String sessionContext = model.getSessionContext();
        String extendCode = model.getExtendCode();
        String senderId = model.getSenderId();
        if (sessionContext != null && !sessionContext.isEmpty()) {
            request.setSessionContext(model.getSessionContext());
        }
        if (extendCode != null && !extendCode.isEmpty()) {
            request.setExtendCode(model.getExtendCode());
        }
        if (senderId != null && !senderId.isEmpty()) {
            request.setSenderId(model.getSenderId());
        }
        return request;
    }


    /**
     * 获取Response
     */
    private SendSmsResponse getResponse(SmsClient client, SendSmsRequest request) {
        try {
            return client.SendSms(request);
        } catch (TencentCloudSDKException e) {
            throw new RuntimeException(e);
        }
    }


}
