package top.xtcoder.clove.common.utils;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

/**
 * 类操作工具包
 * @author XiangTao
 * @date 2021年3月8日
 *
 */
public class ClassUtil {
	private static final Logger log = Logger.getGlobal();
	
	/**
	 * file形式url协议
	 */
	public static final String FILE_PROTOCOL = "file";
	
	/**
	 * jar形式url协议
	 */
	public static final String JAR_PROTOCOL = "jar";
	
	/**
	 * 获取classLoader
	 * @return
	 */
	public static ClassLoader getClassLoader() {
		return Thread.currentThread().getContextClassLoader();
	}
	
	/**
	 * 获取Class
	 * @param className
	 * @return
	 */
	public static Class<?> loadClass(String className){
		try {
			return Class.forName(className);
		}catch(ClassNotFoundException e) {
			log.severe("ClassNotFoundException: " + e.getMessage());
			e.printStackTrace();
			throw new RuntimeException();
		}
	}
	
	/**
	 * 实例化class
	 * @param <T>
	 * @param className
	 * @return
	 */
	public static <T> T newInstance(String className) {
		try {
			Class<?> clazz = loadClass(className);
			return (T) clazz.newInstance();
		}catch(Exception e) {
			log.severe(e.getMessage());
			throw new RuntimeException(e);
		}
	}
	
	public static <T> T newInstance(Class<?> cls) {
		try {
			return (T) cls.newInstance();
		}catch(Exception e) {
			log.severe(e.getMessage());
			throw new RuntimeException(e);
		}
	}
	
	
	/**
	 * 设置类的属性值
	 * @param field
	 * @param target
	 * @param value
	 * @param accessible
	 */
	public static void setField(Field field, Object target, Object value, boolean accessible) {
		field.setAccessible(accessible);
		try {
			field.set(target, value);
		} catch(IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * 设置类的属性值
	 * @param field
	 * @param value
	 * @param target
	 */
	public static void setField(Field field, Object target, Object value) {
		setField(field, target, value, true);
	}
	
	/**
	 * 从Path中获取class
	 * @param classPath
	 * @param basePath
	 * @param basePackage
	 * @return
	 */
	public static Class<?> getClassByPath(Path classPath, Path basePath, String basePackage){
		String packageName = classPath.toString().replace(basePath.toString(), "");
		String className =  (basePackage + packageName)
				.replace("/", ".")
				.replace("\\", ".")
				.replace(".class", "");
		return loadClass(className);
	}
	
	/**
	 * 从jar包中获取class
	 * @param jarEntry
	 * @return
	 */
	public static Class<?> getClassByJar(JarEntry jarEntry){
		String jarEntryName = jarEntry.getName();
		String className = jarEntryName.substring(0, jarEntryName.lastIndexOf("."))
				.replace("/", ".");
		return loadClass(className);
	}
	
	public static Set<Class<?>> getPackageClass(String basePackage) {
		URL url = getClassLoader().getResource(basePackage.replace(".", "/"));
		if(url == null) {
			throw new RuntimeException("无法获取项目路径文件");
		}
		
		try {
			if(url.getProtocol().equalsIgnoreCase(FILE_PROTOCOL)) {
				File file = new File(url.getFile());
				Path basePath = file.toPath();
				return Files.walk(basePath)
						.filter(path -> path.toFile().getName().endsWith(".class"))
						.map(path -> getClassByPath(path, basePath, basePackage))
						.collect(Collectors.toSet());
			}else if(url.getProtocol().equalsIgnoreCase(JAR_PROTOCOL)){
				JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
				return jarURLConnection.getJarFile()
						.stream()
						.filter(jarEntry -> jarEntry.getName().endsWith(".class"))
						.map(ClassUtil::getClassByJar)
						.collect(Collectors.toSet());
			}
			return Collections.emptySet();
		}catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	public static void main(String[] args) throws IOException {
		Set<Class<?>> clsSet = ClassUtil.getPackageClass("org.clove");
		clsSet.forEach(s -> {
			System.out.println(s.getName());
		});
	}
}
