package uk.m0nom.adifproc.adif3.transform;

import org.apache.commons.lang3.StringUtils;
import org.gavaghan.geodesy.GlobalCoordinates;
import org.marsik.ham.adif.Adif3Record;
import org.marsik.ham.adif.enums.Band;
import org.marsik.ham.adif.enums.Propagation;
import org.marsik.ham.adif.types.Sota;
import org.marsik.ham.adif.types.Wwff;
import org.springframework.stereotype.Service;
import uk.m0nom.adifproc.activity.Activity;
import uk.m0nom.adifproc.activity.ActivityDatabase;
import uk.m0nom.adifproc.activity.ActivityDatabaseService;
import uk.m0nom.adifproc.activity.ActivityType;
import uk.m0nom.adifproc.adif3.contacts.Qso;
import uk.m0nom.adifproc.adif3.contacts.Qsos;
import uk.m0nom.adifproc.adif3.control.TransformControl;
import uk.m0nom.adifproc.adif3.transform.comment.CommentTransformer;
import uk.m0nom.adifproc.adif3.transform.comment.FieldParserCommentTransformer;
import uk.m0nom.adifproc.coords.GlobalCoords3D;
import uk.m0nom.adifproc.coords.LocationSource;
import uk.m0nom.adifproc.geocoding.GeocodingProvider;
import uk.m0nom.adifproc.geocoding.GeocodingResult;
import uk.m0nom.adifproc.geocoding.NominatimGeocodingProvider;
import uk.m0nom.adifproc.location.FromLocationDeterminer;
import uk.m0nom.adifproc.location.ToLocationDeterminer;
import uk.m0nom.adifproc.maidenheadlocator.MaidenheadLocatorConversion;
import uk.m0nom.adifproc.qrz.CachingQrzXmlService;
import uk.m0nom.adifproc.qrz.QrzCallsign;
import uk.m0nom.adifproc.satellite.ApSatellite;
import uk.m0nom.adifproc.satellite.ApSatelliteService;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import static uk.m0nom.adifproc.adif3.transform.comment.parsers.FieldParseUtils.parseAlt;

@Service
public class CommentParsingAdifRecordTransformer implements Adif3RecordTransformer {
    private static final Logger logger = Logger.getLogger(CommentParsingAdifRecordTransformer.class.getName());

    private final ActivityDatabaseService activities;
    private final CachingQrzXmlService qrzXmlService;
    private final AdifQrzEnricher enricher;
    private final FromLocationDeterminer fromLocationDeterminer;
    private final ToLocationDeterminer toLocationDeterminer;
    private final ActivityProcessor activityProcessor;
    private final GeocodingProvider geocodingProvider;
    private final CommentTransformer commentTransformer;
    private final ApSatelliteService apSatelliteService;


    public CommentParsingAdifRecordTransformer(ActivityDatabaseService activities,
                                               CachingQrzXmlService qrzXmlService,
                                               FieldParserCommentTransformer commentTransformer,
                                               FromLocationDeterminer fromLocationDeterminer,
                                               ToLocationDeterminer toLocationDeterminer,
                                               ActivityProcessor activityProcessor,
                                               NominatimGeocodingProvider geocodingProvider,
                                               ApSatelliteService apSatelliteService) {
        this.activities = activities;
        this.qrzXmlService = qrzXmlService;
        this.enricher = new AdifQrzEnricher(qrzXmlService);
        this.apSatelliteService = apSatelliteService;
        this.fromLocationDeterminer = fromLocationDeterminer;
        this.toLocationDeterminer = toLocationDeterminer;
        this.activityProcessor = activityProcessor;
        this.geocodingProvider = geocodingProvider;
        this.commentTransformer = commentTransformer;
    }

    private void processSotaRef(Qso qso, TransformResults results) {
        Adif3Record rec = qso.getRecord();

        if (rec.getSotaRef() != null && StringUtils.isNotBlank(rec.getSotaRef().getValue())) {
            String sotaId = rec.getSotaRef().getValue();
            Activity activity = activities.getDatabase(ActivityType.SOTA).get(sotaId);
            if (activity != null) {
                qso.getTo().addActivity(activity);
                String result = toLocationDeterminer.setTheirLocationFromActivity(qso, ActivityType.SOTA, sotaId);
                if (result != null) {
                    results.addContactWithDubiousLocation(result);
                }
            } else {
                results.addContactWithDubiousLocation(String.format("%s (SOTA %s invalid)", qso.getTo().getCallsign(), sotaId));
            }
        }
    }

    private void processWwffRef(Qso qso, TransformResults results) {
        Adif3Record rec = qso.getRecord();

        if (rec.getWwffRef() != null && StringUtils.isNotBlank(rec.getWwffRef().getValue())) {
            String wwffId = rec.getWwffRef().getValue();
            Activity activity = activities.getDatabase(ActivityType.WWFF).get(wwffId);
            if (activity != null) {
                qso.getTo().addActivity(activity);
                String result = toLocationDeterminer.setTheirLocationFromActivity(qso, ActivityType.WWFF, wwffId);
                if (result != null) {
                    results.addContactWithDubiousLocation(result);
                }
            } else {
                results.addContactWithDubiousLocation(String.format("%s (WWFF %s invalid)", qso.getTo().getCallsign(), wwffId));
            }
        }
    }

    private void processRailwaysOnTheAirCallsign(Qso qso, TransformResults results) {
        Adif3Record rec = qso.getRecord();
        // Check the callsign for a Railways on the Air
        Activity rotaInfo = activities.getDatabase(ActivityType.ROTA).get(rec.getCall().toUpperCase());
        if (rotaInfo != null) {
            String result = toLocationDeterminer.setTheirLocationFromActivity(qso, ActivityType.ROTA, rotaInfo.getRef());
            if (result != null) {
                results.addContactWithDubiousLocation(result);
            }
            qso.getTo().addActivity(rotaInfo);
        }
    }

    private void setMyInfoFromQrz(TransformControl control, Qso qso) {
        // Attempt a lookup from QRZ.com
        QrzCallsign myQrzData = qrzXmlService.getCallsignData(qso.getRecord().getStationCallsign());

        fromLocationDeterminer.setMyLocation(control, qso, myQrzData);

        qso.getFrom().setQrzInfo(myQrzData);
        enricher.enrichAdifForMe(qso.getRecord(), myQrzData);
    }

    private QrzCallsign setTheirInfoFromQrz(Qso qso) {
        /* Load QRZ.COM info for the worked station as a fixed station, for information */
        QrzCallsign theirQrzData = qrzXmlService.getCallsignData(qso.getTo().getCallsign());
        qso.getTo().setQrzInfo(theirQrzData);
        enricher.enrichAdifForThem(qso.getRecord(), theirQrzData);
        return theirQrzData;
    }

    private void processSatelliteInfo(TransformControl control,  Qso qso) {
        Adif3Record rec = qso.getRecord();
        if (StringUtils.isBlank(control.getSatelliteBand()) || rec.getBand() == Band.findByCode(control.getSatelliteBand().toLowerCase())) {
            if (StringUtils.isNotBlank(control.getSatelliteMode())) {
                rec.setSatMode(control.getSatelliteMode().toUpperCase());
            }
            if (StringUtils.isNotBlank(control.getSatelliteName())) {
                rec.setSatName(control.getSatelliteName().toUpperCase());
                // Set Propagation Mode Automagically
                rec.setPropMode(Propagation.SATELLITE);
            }
        }
    }

    private boolean hasValidGridsquareNoCoords(Adif3Record rec) {
        return rec.getCoordinates() == null && rec.getGridsquare() != null &&
                !MaidenheadLocatorConversion.isADubiousGridSquare(rec.getGridsquare());
    }

    private boolean setCoordinatesFromGridsquare(Qso qso) {
        Adif3Record rec = qso.getRecord();
        // Set Coordinates from GridSquare that has been supplied in the input file
        try {
            // Only set the gridsquare if it is a valid maidenhead locator
            GlobalCoords3D coords = MaidenheadLocatorConversion.locatorToCoords(LocationSource.OVERRIDE, rec.getGridsquare());
            rec.setCoordinates(coords);
            qso.getTo().setCoordinates(coords);
            qso.getTo().setGrid(rec.getGridsquare());
        } catch (UnsupportedOperationException e) {
            logger.warning(String.format("For QSO with %s: %s", qso.getTo().getCallsign(), e.getMessage()));
            return false;
        }
        return true;
    }

    private boolean hasNoValidGridsquareOrCoords(Adif3Record rec) {
        return rec.getCoordinates() == null || MaidenheadLocatorConversion.isADubiousGridSquare(rec.getGridsquare());
    }

    private void setTheirLocationFromGeocodedAddress(Qso qso, QrzCallsign theirQrzData) {
        Adif3Record rec = qso.getRecord();
        try {
            GeocodingResult result = geocodingProvider.getLocationFromAddress(theirQrzData);
            rec.setCoordinates(result.getCoordinates());
            qso.getTo().setCoordinates(result.getCoordinates());
            if (rec.getCoordinates() != null) {
                logger.info(String.format("Location for %s set based on geolocation data to: %s", rec.getCall(), rec.getCoordinates()));
                if (MaidenheadLocatorConversion.isEmptyOrInvalid(rec.getGridsquare())) {
                    rec.setGridsquare(MaidenheadLocatorConversion.coordsToLocator(rec.getCoordinates()));
                }
            }
        } catch (Exception e) {
            logger.severe(String.format("Caught Exception from Geolocation Provider looking up %s: %s", rec.getCall(), e.getMessage()));
        }
    }

    private boolean coordsAreZero(GlobalCoordinates coords) {
        return coords.getLatitude() == 0.0 && coords.getLongitude() == 0.0;
    }

    private void nullCoordsIfZero(Adif3Record rec) {
        if (rec.getCoordinates() != null && coordsAreZero(rec.getCoordinates())) {
            rec.setCoordinates(null);
        }
        if (rec.getMyCoordinates() != null && coordsAreZero(rec.getMyCoordinates())) {
            rec.setMyCoordinates(null);
        }
    }

    @Override
    public void transform(TransformControl control, TransformResults results, Qsos qsos, Adif3Record rec, int index) {
        Map<String, String> unmapped = new HashMap<>();
        results.getSatelliteActivity().setSatellites(apSatelliteService);

        // A HAM Radio Log ADI input file had both my/their coords set to 0/0 - clearly these aren't right!
        nullCoordsIfZero(rec);

        /* Add Adif3Record details to the Qsos meta structure */
        Qso qso = new Qso(rec, index);
        qsos.addQso(qso);

        activityProcessor.processActivities(control, qso.getFrom(), rec);

        qso.getFrom().setAntenna(control.getAntenna());
        setMyInfoFromQrz(control, qso);
        QrzCallsign theirQrzData = setTheirInfoFromQrz(qso);
        processSotaRef(qso, results);
        processWwffRef(qso, results);
        processRailwaysOnTheAirCallsign(qso, results);
        processSatelliteInfo(control, qso);
        commentTransformer.transformComment(qso, rec.getComment(), unmapped, results);

        if (rec.getCoordinates() == null && rec.getGridsquare() == null) {
            enricher.lookupLocationFromQrz(qso);
        }

        // IF qrz.com can't fill in the coordinates, and the gridsquare is set, fill in coordinates from that
        if (hasValidGridsquareNoCoords(rec)) {
            if (!setCoordinatesFromGridsquare(qso)) {
                results.addContactWithDubiousLocation(String.format("%s (Locator %s invalid)", qso.getTo().getCallsign(), rec.getGridsquare()));
            }
        }

        // Last resort, attempt to find location from qrz.com address data via geolocation provider
        if (hasNoValidGridsquareOrCoords(rec) && theirQrzData != null) {
            setTheirLocationFromGeocodedAddress(qso, theirQrzData);
        }

        // Look to see if there is anything in the SIG/SIGINFO fields
        if (StringUtils.isNotBlank(rec.getSig())) {
            processSig(qso, unmapped);
        }

        if (control.isStripComment()) {
            if (!unmapped.isEmpty()) {
                addUnmappedToRecord(rec, unmapped);
            } else {
                // done a good job and slotted all the key/value pairs in the right place
                rec.setComment("");
            }
        }

        // Add the SOTA Microwave Award data to the end of the comment field
        if (control.isSotaMicrowaveAwardComment()) {
            SotaMicrowaveAward.addSotaMicrowaveAwardToComment(rec);
        }

        if (rec.getSatName() != null) {
            if (apSatelliteService.isAKnownSatellite(rec.getSatName())) {
                ApSatellite satellite = apSatelliteService.getSatellite(rec.getSatName());
                if (satellite.isGeostationary() || apSatelliteService.getEarliestDataAvailable().isBefore(rec.getQsoDate())) {
                    results.getSatelliteActivity().recordSatelliteActivity(qso);
                } else {
                    results.addUnknownSatellitePass(String.format("%s: %s", rec.getSatName(), rec.getQsoDate()));
                }
            } else {
                results.addUnknownSatellite(rec.getSatName());
            }
        }

        // Override your altitude if defined
        if (rec.getApplicationDefinedField(ApplicationDefinedFields.MY_ALT) != null) {
            double alt = parseAlt(rec.getApplicationDefinedField(ApplicationDefinedFields.MY_ALT));
            if (qso.getFrom().getCoordinates() != null) {
                qso.getFrom().getCoordinates().setAltitude(alt);
            }
        }

        // Override their altitude if defined
        if (rec.getApplicationDefinedField(ApplicationDefinedFields.ALT) != null) {
            double alt = parseAlt(rec.getApplicationDefinedField(ApplicationDefinedFields.ALT));
            if (qso.getTo().getCoordinates() != null) {
                qso.getTo().getCoordinates().setAltitude(alt);
            }
        }

        // Set DXCC Entities
        qso.getFrom().setDxccEntity(control.getDxccEntities().findDxccEntityFromCallsign(qso.getFrom().getCallsign(), qso.getRecord().getQsoDate()));
        qso.getTo().setDxccEntity(control.getDxccEntities().findDxccEntityFromCallsign(qso.getTo().getCallsign(), qso.getRecord().getQsoDate()));
    }

    private void processSig(Qso qso, Map<String, String> unmapped) {
        Adif3Record rec = qso.getRecord();
        String activityType = rec.getSig().toUpperCase();
        String activityLocation = rec.getSigInfo().toUpperCase();

        if (StringUtils.isNotBlank(activityType)) {
            // See if it is an activity we support
            ActivityDatabase database = activities.getDatabase(activityType);
            if (database != null) {
                Activity activity = database.get(activityLocation);
                if (activity != null) {
                    qso.getTo().addActivity(activity);

                    boolean sotaFieldEmpty = rec.getSotaRef() == null || StringUtils.isBlank(rec.getSotaRef().getValue());
                    boolean wwffFieldEmpty = rec.getWwffRef() == null || StringUtils.isBlank(rec.getWwffRef().getValue());

                    // Make sure if they have a SOTA or WWFF reference specified
                    // in their specific fields that they take precedence over any other reference
                    // hence why this code is only executed if these specific references are null
                    if (sotaFieldEmpty && wwffFieldEmpty) {
                        toLocationDeterminer.setTheirLocationFromActivity(qso, activity);
                    }

                    // If activity in SIG_INFO/SIG_REF is SOTA and SOTA specific field isn't set, set it now
                    if (sotaFieldEmpty && activity.getType() == ActivityType.SOTA) {
                        rec.setSotaRef(Sota.valueOf(activity.getRef()));
                    }
                    // If activity in SIG_INFO/SIG_REF is WWFF and WWFF specific field isn't set, set it now
                    if (wwffFieldEmpty && activity.getType() == ActivityType.WWFF) {
                        rec.setWwffRef(Wwff.valueOf(activity.getRef()));
                    }

                    unmapped.put(activityType, activityLocation);
                }
            }
        }
    }

    /**
     * Any key/value pairs in the fast log entry comment string that can't be mapped into a specific ADIF field
     * are added to the comment string in the ADIF file
     *
     * @param rec      ADIF record
     * @param unmapped unmapped parameters
     */
    private void addUnmappedToRecord(Adif3Record rec, Map<String, String> unmapped) {
        StringBuilder sb = new StringBuilder();
        Set<String> keySet = unmapped.keySet();
        int keySetLen = keySet.size();
        int i = 1;
        for (String key : unmapped.keySet()) {
            if (StringUtils.isNotEmpty(key)) {
                sb.append(String.format("%s: %s", key, unmapped.get(key)));
                if (i++ < keySetLen) {
                    sb.append(", ");
                }
            } else {
                sb.append(String.format("%s ", key));
            }
        }
        rec.setComment(sb.toString());
    }
}
