package com.netacom.base.chat.android_utils

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.res.Configuration
import android.content.res.Resources
import android.os.Build
import android.os.LocaleList
import android.text.TextUtils
import android.util.Log
import java.util.Locale

/**Created by vantoan on 16,MAR,2021
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/

class LanguageUtils private constructor() {
    companion object {
        private const val KEY_LOCALE = "KEY_LOCALE"
        private const val VALUE_FOLLOW_SYSTEM = "VALUE_FOLLOW_SYSTEM"
        /**
         * Apply the system language.
         *
         * @param isRelaunchApp True to relaunch app, false to recreate all activities.
         */
        /**
         * Apply the system language.
         */
        @JvmOverloads
        fun applySystemLanguage(isRelaunchApp: Boolean = false) {
            applyLanguageReal(null, isRelaunchApp)
        }
        /**
         * Apply the language.
         *
         * @param locale        The language of locale.
         * @param isRelaunchApp True to relaunch app, false to recreate all activities.
         */
        /**
         * Apply the language.
         *
         * @param locale The language of locale.
         */
        @JvmOverloads
        fun applyLanguage(locale: Locale, isRelaunchApp: Boolean = false) {
            applyLanguageReal(locale, isRelaunchApp)
        }

        private fun applyLanguageReal(locale: Locale?, isRelaunchApp: Boolean) {
            if (locale == null) {
                UtilsBridge.spUtils4Utils.put(KEY_LOCALE, VALUE_FOLLOW_SYSTEM, true)
            } else {
                UtilsBridge.spUtils4Utils.put(KEY_LOCALE, locale2String(locale), true)
            }
            val destLocal = locale ?: getLocal(Resources.getSystem().configuration)
            updateAppContextLanguage(destLocal) { success ->
                if (success == true) {
                    restart(isRelaunchApp)
                } else {
                    // use relaunch app
                    UtilsBridge.relaunchApp()
                }
            }
        }

        private fun restart(isRelaunchApp: Boolean) {
            if (isRelaunchApp) {
                UtilsBridge.relaunchApp()
            } else {
                for (activity in UtilsBridge.activityList) {
                    activity.recreate()
                }
            }
        }

        val isAppliedLanguage: Boolean
            get() = getAppliedLanguage() != null

        fun isAppliedLanguage(locale: Locale): Boolean {
            val appliedLocale = getAppliedLanguage() ?: return false
            return isSameLocale(locale, appliedLocale)
        }

        /**
         * Return the applied locale.
         *
         * @return the applied locale
         */
        fun getAppliedLanguage(): Locale? {
            val spLocaleStr = UtilsBridge.spUtils4Utils.getString(KEY_LOCALE)
            return if (TextUtils.isEmpty(spLocaleStr) || VALUE_FOLLOW_SYSTEM == spLocaleStr) {
                null
            } else string2Locale(spLocaleStr)
        }

        /**
         * Return the locale of context.
         *
         * @return the locale of context
         */
        private fun getContextLanguage(context: Context): Locale {
            return getLocal(context.resources.configuration)
        }

        /**
         * Return the locale of applicationContext.
         *
         * @return the locale of applicationContext
         */
        val appContextLanguage: Locale
            get() = getContextLanguage(Utils.getApp())

        /**
         * Return the locale of system
         *
         * @return the locale of system
         */
        val systemLanguage: Locale
            get() = getLocal(Resources.getSystem().configuration)

        /**
         * Update the locale of applicationContext.
         *
         * @param destLocale The dest locale.
         * @param consumer   The consumer.
         */
        private fun updateAppContextLanguage(destLocale: Locale, consumer: Utils.Consumer<Boolean?>?) {
            pollCheckAppContextLocal(destLocale, 0, consumer)
        }

        private fun pollCheckAppContextLocal(destLocale: Locale, index: Int, consumer: Utils.Consumer<Boolean?>?) {
            val appResources = Utils.getApp().resources
            val appConfig = appResources.configuration
            val appLocal = getLocal(appConfig)
            setLocal(appConfig, destLocale)
            Utils.getApp().resources.updateConfiguration(appConfig, appResources.displayMetrics)
            if (consumer == null) return
            if (isSameLocale(appLocal, destLocale)) {
                consumer.accept(true)
            } else {
                if (index < 20) {
                    UtilsBridge.runOnUiThreadDelayed({ pollCheckAppContextLocal(destLocale, index + 1, consumer) }, 16)
                    return
                }
                Log.e("LanguageUtils", "appLocal didn't update.")
                consumer.accept(false)
            }
        }

        /**
         * If applyLanguage not work, try to call it in [Activity.attachBaseContext].
         *
         * @param context The baseContext.
         * @return the context with language
         */
        fun attachBaseContext(activity: Activity) {
            val spLocale = UtilsBridge.spUtils4Utils.getString(KEY_LOCALE)
            if (TextUtils.isEmpty(spLocale)) {
                return
            }
            val destLocal: Locale? = if (VALUE_FOLLOW_SYSTEM == spLocale) {
                getLocal(Resources.getSystem().configuration)
            } else {
                string2Locale(spLocale)
            }
            if (destLocal == null) return
            updateConfiguration(activity, destLocal)
            updateConfiguration(Utils.getApp(), destLocal)
        }

        @Suppress("DEPRECATION")
        private fun updateConfiguration(context: Context, destLocal: Locale) {
            val resources = context.resources
            val config = resources.configuration
            setLocal(config, destLocal)
            resources.updateConfiguration(config, resources.displayMetrics)
        }

        private fun locale2String(locale: Locale): String {
            val localLanguage = locale.language // this may be empty
            val localCountry = locale.country // this may be empty
            return "$localLanguage$$localCountry"
        }

        private fun string2Locale(str: String): Locale? {
            val locale = string2LocaleReal(str)
            if (locale == null) {
                Log.e("LanguageUtils", "The string of $str is not in the correct format.")
                UtilsBridge.spUtils4Utils.remove(KEY_LOCALE)
            }
            return locale
        }

        private fun string2LocaleReal(str: String): Locale? {
            return if (!isRightFormatLocalStr(str)) {
                null
            } else try {
                val splitIndex = str.indexOf("$")
                Locale(str.substring(0, splitIndex), str.substring(splitIndex + 1))
            } catch (ignore: Exception) {
                null
            }
        }

        private fun string2Language(str: String): String? {
            return if (!isRightFormatLocalStr(str)) {
                null
            } else try {
                val splitIndex = str.indexOf("$")
                str.substring(0, splitIndex)
            } catch (ignore: Exception) {
                null
            }
        }

        private fun isRightFormatLocalStr(localStr: String): Boolean {
            val chars = localStr.toCharArray()
            var count = 0
            for (c in chars) {
                if (c == '$') {
                    if (count >= 1) {
                        return false
                    }
                    ++count
                }
            }
            return count == 1
        }

        private fun isSameLocale(l0: Locale, l1: Locale): Boolean {
            return (
                UtilsBridge.equals(l1.language, l0.language) &&
                    UtilsBridge.equals(l1.country, l0.country)
                )
        }

        private fun getLocal(configuration: Configuration): Locale {
            return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                configuration.locales[0]
            } else {
                configuration.locale
            }
        }

        @SuppressLint("ObsoleteSdkInt", "AnnotateVersionCheck")
        private fun setLocal(configuration: Configuration, locale: Locale) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                configuration.setLocale(locale)
            } else {
                configuration.locale = locale
            }
        }

        fun getLanguage(): String = string2Language(UtilsBridge.spUtils4Utils.getString(KEY_LOCALE)) ?: Locale.getDefault().language

        fun updateLocale(
            c: Context,
            languageToSwitchTo: String
        ): Context {
            val locale = Locale(languageToSwitchTo)
            Locale.setDefault(locale)
            var context = c
            val resources = context.resources

            val configuration: Configuration = resources.configuration
            configuration.setLayoutDirection(locale)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                val localeList = LocaleList(locale)
                LocaleList.setDefault(localeList)
                configuration.setLocales(localeList)
            } else {
                configuration.locale = locale
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N_MR1) {
                context = context.createConfigurationContext(configuration)
            }

            resources.updateConfiguration(configuration, resources.displayMetrics)

            return context
        }
    }

    init {
        throw UnsupportedOperationException("u can't instantiate me...")
    }
}
