/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.base

import android.os.Bundle
import android.view.View
import androidx.annotation.LayoutRes
import androidx.appcompat.app.AppCompatActivity
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.LiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import com.google.android.material.snackbar.Snackbar
import com.netacom.base.chat.R
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.livedata.EventObserver
import com.netacom.base.chat.type.ScreenState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancelChildren
import kotlin.coroutines.CoroutineContext

abstract class BaseDataActivity<DB : ViewDataBinding?, VM : ViewModel>(
    @LayoutRes val layoutID: Int?,
    private val viewModelClass: Class<VM>
) :
    AppCompatActivity(), CoroutineScope {

    private val job = SupervisorJob()
    override val coroutineContext: CoroutineContext
        get() = Dispatchers.Main + job
    private var _binding: DB? = null
    val binding get() = _binding!!
    val viewModel by lazy {
        ViewModelProvider(this).get(viewModelClass)
    }

    protected abstract fun setViewModel(): Int
    protected abstract fun initViews()
    protected abstract fun initData()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        layoutID?.let { layout ->
            _binding = DataBindingUtil.setContentView(this, layout) as? DB
            _binding?.apply {
                lifecycleOwner = this@BaseDataActivity
                setVariable(setViewModel(), viewModel)
            }
        }
        initViews()
        initData()
        // checkError()
    }

    private fun checkError() {
        (viewModel as? BaseViewModel)?.screenStateErrorListen?.observeOnce {
            when (it.state) {
                ScreenState.NO_INTERNET -> showSnackBar(R.string.not_have_internet_please_check_your_connection)
                ScreenState.ERROR -> showSnackBar(it.message)
                else -> showSnackBar(it.message)
            }
        }
    }

    fun <T> showSnackBar(text: T, textAction: Int? = null, action: View.OnClickListener? = null) {
        findViewById<View>(android.R.id.content)?.apply {
            val snackBar = when (text) {
                is Int -> Snackbar.make(this, getString(text), Snackbar.LENGTH_LONG)
                else -> Snackbar.make(this, text.toString(), Snackbar.LENGTH_LONG)
            }
            if (textAction != null && action != null) {
                snackBar.setAction(getString(textAction), action)
            }
            snackBar.show()
        }
    }

    fun <T> LiveData<T>.observe(listener: (T) -> (Unit)) {
        observe(this@BaseDataActivity) { listener.invoke(it) }
    }

    inline fun <T> LiveData<EventLiveData<T>>.observeOnce(crossinline onChanged: (T) -> (Unit)) {
        observe(
            this@BaseDataActivity,
            EventObserver { value ->
                onChanged(value)
            }
        )
    }

    override fun onStop() {
        super.onStop()
        job.cancelChildren()
    }

    override fun onDestroy() {
        // Logger.e("onDestroy::" + this.javaClass.simpleName)
        _binding = null
        super.onDestroy()
    }
}
