/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.base

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.annotation.LayoutRes
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.fragment.app.Fragment
import androidx.lifecycle.LiveData
import androidx.lifecycle.ViewModelProvider
import com.google.android.material.snackbar.Snackbar
import com.netacom.base.chat.R
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.livedata.EventObserver
import com.netacom.base.chat.type.ScreenState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlin.coroutines.CoroutineContext

abstract class BaseDataFragment<DB : ViewDataBinding, VM : BaseViewModel>(
    @LayoutRes val layoutID: Int,
    private val viewModelClass: Class<VM>
) : Fragment(), CoroutineScope {
    protected var job = SupervisorJob()
    override val coroutineContext: CoroutineContext
        get() = Dispatchers.Main + job

    lateinit var viewModel: VM
    private var _binding: DB? = null
    val binding get() = _binding!!
    private fun getViewVM(): VM = ViewModelProvider(requireActivity()).get(viewModelClass)
    protected abstract fun setViewModel(): Int
    protected abstract fun initViews()
    protected abstract fun initData()
    protected abstract fun syncEvent()
    protected abstract fun setupTheme()

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        super.onCreateView(inflater, container, savedInstanceState)
        viewModel = getViewVM()
        _binding = DataBindingUtil.inflate(inflater, layoutID, container, false)
        _binding?.apply {
            lifecycleOwner = this@BaseDataFragment
            setVariable(setViewModel(), viewModel)
        }
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        initViews()
        initData()
        syncEvent()
    }

    override fun onResume() {
        super.onResume()
        setupTheme()
        checkError()
    }

    private fun checkError() {
        (viewModel as? BaseViewModel)?.screenStateErrorListen?.observeOnce {
            when (it.state) {
                ScreenState.NO_INTERNET -> showSnackBar(R.string.not_have_internet_please_check_your_connection)
                ScreenState.ERROR -> showSnackBar(it.message)
                else -> showSnackBar(it.message)
            }
        }
    }

    fun <T> showSnackBar(text: T, textAction: Int? = null, action: View.OnClickListener? = null) {
        activity?.findViewById<View>(android.R.id.content)?.apply {
            val snackBar = when (text) {
                is Int -> Snackbar.make(this, getString(text), Snackbar.LENGTH_LONG)
                else -> Snackbar.make(this, text.toString(), Snackbar.LENGTH_LONG)
            }
            if (textAction != null && action != null) {
                snackBar.setAction(getString(textAction), action)
            }
            snackBar.show()
        }
    }

    fun <T> LiveData<T>.observe(listener: (T) -> (Unit)) {
        observe(this@BaseDataFragment) { listener.invoke(it) }
    }

    inline fun <T> LiveData<EventLiveData<T>>.observeOnce(crossinline onChanged: (T) -> (Unit)) {
        observe(
            viewLifecycleOwner,
            EventObserver { value ->
                onChanged(value)
            }
        )
    }

    override fun onDestroyView() {
        job.cancel()
        _binding = null
        super.onDestroyView()
    }
}
