/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.base

import android.app.Dialog
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import androidx.annotation.LayoutRes
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.fragment.app.DialogFragment
import androidx.lifecycle.LiveData
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProvider
import com.google.android.material.snackbar.Snackbar
import com.netacom.base.chat.R
import com.netacom.base.chat.android_utils.ScreenUtils
import com.netacom.base.chat.define.DialogDef
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.livedata.EventObserver
import com.netacom.base.chat.util.dimen

abstract class BaseDialogFragment<VM : BaseViewModel, DB : ViewDataBinding>(
    @LayoutRes val layoutID: Int,
    private val viewModelClass: Class<VM>
) : DialogFragment() {
    private fun getViewVM(): VM = ViewModelProvider(requireActivity()).get(viewModelClass)
    protected abstract fun setLayoutHeight(): Int
    protected abstract fun setViewModel(): Int
    private var _binding: DB? = null
    val binding get() = _binding!!
    lateinit var viewModel: VM
    var isLayout: Int = DialogDef.LAYOUT_NORMAL

    override fun onStart() {
        super.onStart()
        dialog?.window?.apply {
            setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE)
            isLayout = setLayoutHeight()
            when (isLayout) {
                DialogDef.LAYOUT_SMALL -> setLayout(dimen(R.dimen.dp_300), dimen(R.dimen.dp_160))
                DialogDef.LAYOUT_NORMAL -> setLayout(ScreenUtils.getScreenWidth() / 2, ScreenUtils.getScreenHeight() / 2)
                DialogDef.LAYOUT_NEARLY_FULL -> setLayout(dimen(R.dimen.dp_300) * 2, dimen(R.dimen.dp_160) * 2)
                DialogDef.LAYOUT_FULL -> setLayout(ScreenUtils.getScreenWidth(), ScreenUtils.getScreenHeight())
            }
        }
    }
    override fun getTheme(): Int = R.style.BottomSheetDialogTheme

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        super.onCreateView(inflater, container, savedInstanceState)
        _binding = DataBindingUtil.inflate(inflater, layoutID, container, false)
        viewModel = getViewVM()
        _binding?.apply {
            setVariable(setViewModel(), viewModel)
            lifecycleOwner = this@BaseDialogFragment
            executePendingBindings()
        }
        return binding.root
    }

    override fun onCreateDialog(savedInstanceState: Bundle?): Dialog = Dialog(requireContext(), theme)

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        initViews()
        initData()
        setupTheme()
    }

    protected abstract fun initViews()

    protected abstract fun initData()

    protected abstract fun setupTheme()

    fun <T> showSnackBar(text: T, textAction: Int? = null, action: View.OnClickListener? = null) {
        view?.apply {
            val snackBar = when (text) {
                is Int -> Snackbar.make(this, getString(text), Snackbar.LENGTH_LONG)
                else -> Snackbar.make(this, text.toString(), Snackbar.LENGTH_LONG)
            }
            if (textAction != null && action != null) {
                snackBar.setAction(getString(textAction), action)
            }
            snackBar.show()
        }
    }

    fun <T> LiveData<T>.observe(listener: (T) -> (Unit)) {
        observe(this@BaseDialogFragment, Observer { listener.invoke(it) })
    }

    inline fun <T> LiveData<EventLiveData<T>>.observeOnce(crossinline onChanged: (T) -> (Unit)) {
        observe(
            viewLifecycleOwner,
            EventObserver { value ->
                onChanged(value)
            }
        )
    }

    override fun onDestroyView() {
        dialog?.dismiss()
        _binding = null
        super.onDestroyView()
    }
}
