/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.base

import androidx.lifecycle.MutableLiveData
import com.netacom.base.chat.model.ScreenStateObj
import com.netacom.base.chat.network.ApiResponse
import com.netacom.base.chat.network.BasePage
import com.netacom.base.chat.type.RequestStatus
import com.netacom.base.chat.type.ScreenState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

abstract class BaseListViewModel<T> : BaseViewModel() {

    val state = MutableLiveData<ScreenStateObj>()
    val list = MutableLiveData<List<T>>()
    val keyword: MutableLiveData<String> = MutableLiveData()
    var page: Int = FIRST_PAGE
    var allowLoadMore = true
    var fetchDataJob: Job? = null
    var delayJob: Job? = null

    companion object {
        const val FIRST_PAGE = 0
        const val PAGE_SIZE = 20
        const val DELAY_SEARCH_MS = 500L
    }

    init {
        state.value = ScreenStateObj(state = ScreenState.INIT)
    }

    fun refresh(scope: CoroutineScope) {
        page = FIRST_PAGE
        state.value = ScreenStateObj(state = ScreenState.REFRESH)
        allowLoadMore = true
        fetchData(scope)
    }

    fun loadmore(scope: CoroutineScope) {
        if (allowLoadMore) {
            page++
            state.value = ScreenStateObj(state = ScreenState.LOAD_MORE)
            fetchData(scope)
        }
    }

    abstract suspend fun callApi(): ApiResponse<BasePage<T>>

    protected open fun handleListDataBeforeDisplay(items: List<T>?): List<T>? {
        return items
    }

    fun setKeyword(scope: CoroutineScope, text: String?) {
        keyword.value = text.toString()
        delayJob?.cancel()
        delayJob = scope.launch {
            delay(DELAY_SEARCH_MS)
            page = FIRST_PAGE
            allowLoadMore = true
            state.value = ScreenStateObj(state = ScreenState.INIT)
            fetchData(scope)
        }
    }

    open fun handleError(scope: CoroutineScope, response: ApiResponse<BasePage<T>>) {
        state.postValue(ScreenStateObj.ERROR(response) { fetchData(scope) })
    }

    fun fetchData(scope: CoroutineScope) {
        if (state.value?.state == ScreenState.INIT ||
            state.value?.state == ScreenState.ERROR ||
            state.value?.state == ScreenState.NO_INTERNET
        ) {
            state.setDoing()
        }

        fetchDataJob?.cancel()
        fetchDataJob = scope.launch {
            callApi().apply {
                if (code == RequestStatus.SUCCESS.value) {
                    if (data?.data.isNullOrEmpty()) {
                        when (state.value?.state) {
                            ScreenState.LOAD_MORE -> state.postValue(ScreenStateObj(state = ScreenState.LOAD_MORE_EMPTY))
                            ScreenState.REFRESH -> {
                                state.postValue(ScreenStateObj(state = ScreenState.REFRESH_EMPTY))
                                list.postValue(listOf())
                            }
                            else -> {
                                state.postEmpty()
                                list.postValue(listOf())
                            }
                        }
                    } else {
                        val handledList = handleListDataBeforeDisplay(data?.data) ?: listOf()

                        when (state.value?.state) {
                            ScreenState.LOAD_MORE -> {
                                list.postValue(
                                    (list.value ?: listOf()) + handledList
                                )
                                state.postData()
                            }
                            else -> {
                                list.postValue(handledList)
                                if (handledList.isEmpty()) {
                                    state.postEmpty()
                                } else {
                                    state.postData()
                                }
                            }
                        }
                    }

                    if ((data?.data?.size ?: 0) < PAGE_SIZE) {
                        allowLoadMore = false
                    }
                } else {
                    when (state.value?.state) {
                        ScreenState.LOAD_MORE -> {
                            page--
                            state.postValue(
                                ScreenStateObj(
                                    state = ScreenState.LOAD_MORE_ERROR,
                                    message = this.message
                                )
                            )
                        }
                        ScreenState.REFRESH -> {
                            state.postValue(
                                ScreenStateObj(
                                    state = ScreenState.REFRESH_ERROR,
                                    message = this.message
                                )
                            )
                        }
                        else -> if (errorCode != RequestStatus.CANCEL_REQUEST.value) {
                            handleError(scope, this)
                        }
                    }
                }
            }
        }
    }
}
