/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.base

import androidx.annotation.Keep
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.liveData
import androidx.lifecycle.viewModelScope
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.model.ScreenStateObj
import com.netacom.base.chat.network.ApiResponse
import com.netacom.base.chat.type.ScreenState
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.launch
import java.net.UnknownHostException
import kotlin.coroutines.CoroutineContext

@Keep
open class BaseViewModel : ViewModel() {
    val screenStateError: LiveData<EventLiveData<ScreenStateObj>> = MutableLiveData()
    val screenStateErrorListen: LiveData<EventLiveData<ScreenStateObj>> = screenStateError

    fun <T> LiveData<T>.post(data: T) = (this as MutableLiveData<T>).postValue(data)
    fun <T> LiveData<T>.sendAction(data: T) = (this as ActionLiveData<T>).postAction(data)

    fun <T> LiveData<T>.set(data: T) {
        (this as MutableLiveData<T>).value = data
    }

    fun <T> LiveData<T>.setAction(data: T) {
        (this as ActionLiveData<T>).sendAction(data)
    }

    fun LiveData<String?>.trimValue(): String {
        return (value)?.trim() ?: ""
    }

    fun LiveData<ScreenStateObj>.setDoing() {
        this.set(ScreenStateObj(state = ScreenState.LOADING))
    }

    fun LiveData<ScreenStateObj>.postDoing() {
        this.post(ScreenStateObj(state = ScreenState.LOADING))
    }

    fun LiveData<ScreenStateObj>.postLoadMore() {
        this.post(ScreenStateObj(state = ScreenState.LOAD_MORE))
    }

    fun LiveData<ScreenStateObj>.postData() {
        this.post(ScreenStateObj(state = ScreenState.DATA))
    }

    fun LiveData<ScreenStateObj>.postEmpty() {
        this.post(ScreenStateObj(state = ScreenState.EMPTY))
    }

    fun <T> LiveData<EventLiveData<ScreenStateObj>>.postErrorOnce(message: T? = null) {
        this.post(EventLiveData(ScreenStateObj(state = ScreenState.ERROR, message = message)))
    }

    fun <T> LiveData<ScreenStateObj>.postError(message: T? = null) {
        this.post(ScreenStateObj(state = ScreenState.ERROR, message = message))
    }

    fun <T> ActionLiveData<ActionData<T>>.setDoing() {
        value = ActionData.DOING()
    }

    fun <T> ActionLiveData<ActionData<T>>.postFinish() {
        this.post(ActionData(isDoing = false))
    }

    fun <T> ActionLiveData<ActionData<T>>.postSuccess(data: T? = null) {
        post(ActionData.SUCCESS(data))
    }

    fun <T> ActionLiveData<ActionData<T>>.postResult(response: ApiResponse<T>) {
        if (response.data != null) {
            postSuccess(response.data)
        } else {
            postError(response)
        }
    }

    fun <T> ActionLiveData<ActionData<T>>.setError(message: String? = null) {
        value = ActionData.ERROR(message)
    }

    fun <T> ActionLiveData<ActionData<T>>.postErrorMsg(message: String? = null) {
        post(ActionData.ERROR(message))
    }

    fun <T> ActionLiveData<ActionData<T>>.postError(response: ApiResponse<T>? = null) {
        post(response?.let { ActionData.ERROR(it) } ?: ActionData.ERROR())
    }

    val handlerError = CoroutineExceptionHandler { _, exception ->
        exception.printStackTrace()
        when (exception) {
            is UnknownHostException -> {
                Logger.e("NetworkErrorException:NO_INTERNET")
                screenStateError.post(EventLiveData(ScreenStateObj(ScreenState.NO_INTERNET)))
            }
            else -> {
                Logger.e("CoroutineExceptionHandler")
            }
        }
    }

    inline fun <T> launchOnViewModelScopeLiveData(scope: CoroutineContext, crossinline block: suspend () -> LiveData<T>): LiveData<T> {
        return liveData(context = viewModelScope.coroutineContext + scope + handlerError) {
            emitSource(block())
        }
    }

    inline fun launchOnViewModelScope(scope: CoroutineContext, crossinline block: suspend () -> Unit) {
        viewModelScope.launch(context = viewModelScope.coroutineContext + scope + handlerError) {
            block()
        }
    }
}
