/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.imageloader

import android.content.Context
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.Drawable
import android.view.ViewGroup
import android.widget.ImageView
import androidx.swiperefreshlayout.widget.CircularProgressDrawable
import com.bumptech.glide.load.MultiTransformation
import com.bumptech.glide.load.Transformation
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.load.resource.bitmap.CenterCrop
import com.bumptech.glide.load.resource.bitmap.CircleCrop
import com.bumptech.glide.load.resource.bitmap.RoundedCorners
import com.bumptech.glide.request.target.CustomTarget
import com.bumptech.glide.request.target.Target
import com.bumptech.glide.request.transition.Transition
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.jackandphantom.blurimage.BlurImage
import com.netacom.base.chat.R
import com.netacom.base.chat.android_utils.SizeUtils
import com.netacom.base.chat.android_utils.ThreadUtils
import com.netacom.base.chat.util.getThemeColor
import com.netacom.base.chat.util.screenWidth

/**
 * Class for global image functions (load image, transform image, ...)
 */

fun <T> ImageView.loadAvatar(imgUrl: T?) {
    imgUrl?.let { url ->
        GlideApp.with(this.context)
            .load(url)
            .centerInside()
            .placeholder(CircleProgress(this.context))
            .error(R.drawable.ic_avatar_default)
            .diskCacheStrategy(DiskCacheStrategy.ALL)
            .transform(MultiTransformation(CircleCrop()))
            .into(this)
        /*loadAny(it) {
            placeholder(CircleProgress(this@loadAvatar.context))
            error(R.drawable.ic_avatar_default)
            transformations(CircleCropTransformation())
        }*/
    } ?: setImageResource(R.drawable.ic_avatar_default)
}

fun <T> ImageView.loadImage(
    imgUrl: T?,
    drawableError: Int = R.drawable.ic_no_image,
    corner: Float? = 0f,
    transformation: Transformation<Bitmap>? = null,
    allowFitCenter: Boolean = true,
    allowCenterCrop: Boolean = false,
    isWhiteTheme: Boolean = false
) {
    imgUrl?.let { url ->
        val options = GlideApp.with(this.context)
            .load(url)
            .placeholder(CircleProgress(this.context, isWhiteTheme))
            .error(drawableError)
            .diskCacheStrategy(DiskCacheStrategy.ALL)
            .thumbnail(0.1f)
        if (corner ?: 0f > 0) {
            options.transform(
                MultiTransformation(
                    CenterCrop(),
                    RoundedCorners(corner?.toInt() ?: 1)
                )
            )
        }
        if (allowFitCenter && !allowCenterCrop) {
            options.fitCenter()
        }
        if (allowCenterCrop) {
            options.centerCrop()
        }
        transformation?.apply {
            options.transform(this)
        }
        options.into(this)
    }
}

fun <T> ImageView.loadImage(imgUrl: T?, imgWidth: Int, imgHeight: Int) {
    // Logger.e("loadLargeImage:imgWidth=$imgWidth imgHeight=$imgHeight imgPath=$imgUrl")
    if (imgUrl == null) {
        setImageResource(R.drawable.ic_no_image)
    } else {
        val size = if (imgWidth > imgHeight) {
            imgWidth
        } else {
            imgHeight
        }
        val scale = when {
            size >= 1200 -> 5f
            size >= 1000 -> 4f
            size >= 720 -> 3f
            size >= 500 -> 3.5f
            size >= 300 -> 2f
            else -> 1f
        }
        var width = SizeUtils.dp2px(imgWidth.div(scale))
        val height = SizeUtils.dp2px(imgHeight.div(scale))
        // Calculator ratio for image width and screen width
        val screenWidth = this.context.screenWidth
        if (width > screenWidth) {
            width = screenWidth - 200
        } else if (width < screenWidth && width > (screenWidth - 100)) {
            width -= 150
        } else {
            width -= 100
        }
        GlideApp.with(this.context)
            .load(imgUrl)
            .placeholder(CircleProgress(this.context))
            .error(R.drawable.ic_no_image)
            .centerInside()
            .override(width, height)
            .diskCacheStrategy(DiskCacheStrategy.ALL)
            .into(this)
        /*loadAny(imgUrl) {
            error(R.drawable.ic_no_image)
            size(width, height)
        }*/
    }
}

fun <T> loadBitmapSync(
    context: Context,
    t: T?,
    callback: (Bitmap?) -> Unit,
    drawableError: Int = R.drawable.ic_no_image
) {
    try {
        if (t != null) {
            val futureTarget = GlideApp.with(context)
                .asBitmap()
                .load(t)
                .centerInside()
                .error(drawableError)
                .submit(Target.SIZE_ORIGINAL, Target.SIZE_ORIGINAL)

            callback.invoke(futureTarget.get())
            GlideApp.with(context).clear(futureTarget)
            /*val request = ImageRequest.Builder(context)
                .data(t)
                .error(drawableError)
                .allowHardware(false)
                .build()
            val drawable = context.imageLoader.execute(request).drawable
            val bitmap = (drawable as? BitmapDrawable)?.bitmap
            callback.invoke(bitmap)*/
        } else {
            callback.invoke(null)
        }
    } catch (e: Exception) {
        FirebaseCrashlytics.getInstance().recordException(e)
        e.printStackTrace()
        callback.invoke(null)
    }
}

fun <T> loadAvatarNotification(
    context: Context,
    t: T?,
    callback: (Bitmap?) -> Unit,
    drawableError: Int = R.drawable.ic_no_image,
    size: Int? = null
) {
    try {
        if (t != null) {
            val width = size?.let {
                size
            } ?: Target.SIZE_ORIGINAL
            val height = size?.let {
                size
            } ?: Target.SIZE_ORIGINAL

            val futureTarget = GlideApp.with(context)
                .asBitmap()
                .load(t)
                .transform(jp.wasabeef.transformers.glide.CropCircleTransformation())
                .error(drawableError)
                .submit(width, height)
            callback.invoke(futureTarget.get())
            GlideApp.with(context).clear(futureTarget)
        } else {
            callback.invoke(null)
        }
    } catch (e: Exception) {
        FirebaseCrashlytics.getInstance().recordException(e)
        e.printStackTrace()
        callback.invoke(null)
    }
}

fun <T> ViewGroup.loadImageBackground(imgUrl: T?, isBlur: Boolean = true) {
    if (imgUrl == null) {
        setBackgroundResource(R.drawable.ic_no_image)
    } else {
        val option = GlideApp.with(this.context)
            .load(imgUrl)
            .placeholder(CircleProgress(this.context))
            .error(R.drawable.ic_no_image)
        // No use because performance
        // if (isBlur) option.apply(bitmapTransform(BlurTransformation(100)))
        option.into(
            object : CustomTarget<Drawable>() {
                override fun onResourceReady(resource: Drawable, transition: Transition<in Drawable>?) {
                    if (isBlur) {
                        this@loadImageBackground.post {
                            val bitmap = BlurImage.with(this@loadImageBackground.context)
                                .load(drawableToBitmap(resource))
                                .intensity(20f)
                                .Async(false).imageBlur
                            this@loadImageBackground.background = BitmapDrawable(resources, bitmap)
                        }
                    } else {
                        this@loadImageBackground.background = resource
                    }
                }

                override fun onLoadCleared(placeholder: Drawable?) {
                }
            }
        )
    }
}

private fun drawableToBitmap(drawable: Drawable): Bitmap? {
    if (drawable is BitmapDrawable) {
        if (drawable.bitmap != null) {
            return drawable.bitmap
        }
    }
    val bitmap: Bitmap = if (drawable.intrinsicWidth <= 0 || drawable.intrinsicHeight <= 0) {
        Bitmap.createBitmap(1, 1, Bitmap.Config.ARGB_8888) // Single color bitmap will be created of 1x1 pixel
    } else {
        Bitmap.createBitmap(drawable.intrinsicWidth, drawable.intrinsicHeight, Bitmap.Config.ARGB_8888)
    }
    val canvas = Canvas(bitmap)
    drawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight())
    drawable.draw(canvas)
    return bitmap
}

private class CircleProgress(context: Context, isWhiteTheme: Boolean = false) : CircularProgressDrawable(context) {
    init {
        strokeWidth = 5f
        centerRadius = 30f
        if (isWhiteTheme) setColorSchemeColors(getThemeColor(R.color.colorWhite50))
        ThreadUtils.runOnUiThread { start() }
    }
}
