/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */
package com.netacom.base.chat.library.mention

import android.content.Context
import android.text.Editable
import android.text.TextWatcher
import android.text.style.ClickableSpan
import android.util.AttributeSet
import android.widget.EditText
import androidx.annotation.AttrRes
import androidx.appcompat.widget.AppCompatEditText

/**
 * Create on 8/4/20
 * @author duonghd
 */

class DEditText : AppCompatEditText {

    constructor(context: Context) : super(context) {
        init(context)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        init(context)
    }

    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        init(context)
    }

    private var externalTextWatcher: TextWatcher? = null
    private var _removeMention: RemoveMention? = null
    private var _blockCompletion = false

    private fun init(context: Context) {
        addTextChangedListener(
            object : TextWatcher {
                override fun beforeTextChanged(
                    text: CharSequence,
                    start: Int,
                    count: Int,
                    after: Int
                ) {
                    if (_blockCompletion) return
                    removeIfMarkSpans(count = count, after = after)
                    externalTextWatcher?.beforeTextChanged(text, start, count, after)
                }

                override fun onTextChanged(
                    charSequence: CharSequence,
                    start: Int,
                    before: Int,
                    count: Int
                ) {
                    if (_blockCompletion) return
                    if (_removeMention != null) {
                        _blockCompletion = true
                        text?.let {
                            it.removeSpan(_removeMention!!.clickableSpan)
                            text = it.replace(_removeMention!!.start, _removeMention!!.end - 1, "@")
                            setSelection(_removeMention!!.start + 1)
                            externalTextWatcher?.onTextChanged(text, 0, 0, 0)
                        }
                        _removeMention = null
                        _blockCompletion = false
                    } else {
                        externalTextWatcher?.onTextChanged(charSequence, start, before, count)
                    }
                }

                override fun afterTextChanged(editable: Editable) {
                    if (_blockCompletion) return
                    externalTextWatcher?.afterTextChanged(editable)
                }
            }
        )
    }

    // --------------------------------------------------
    // Cursor & Selection Event Methods
    // --------------------------------------------------
    /**
     * Called whenever the selection within the [EditText] has changed.
     *
     * @param selStart starting position of the selection
     * @param selEnd ending position of the selection
     */
    override fun onSelectionChanged(selStart: Int, selEnd: Int) {
        /**
         * TODO refactor selection with selStart != selEnd
         * good luck (-.*")
         * */
        if (selStart == selEnd) {
            val text = text ?: return
            text.getSpans(selStart, selStart, ClickableSpan::class.java)?.firstOrNull()?.let {
                val start = text.getSpanStart(it)
                val end = text.getSpanEnd(it)
                if (selStart in (start + 1) until end) {
                    super.setSelection(end)
                }
            }
        }

        super.onSelectionChanged(selStart, selEnd)
    }

    /**
     * Marks a span for deletion later if necessary by checking if the last character in a MentionSpan
     * is deleted by this change. If so, mark the span to be deleted later when
     * {@link #ensureMentionSpanIntegrity(Editable)} is called in {@link #handleTextChanged()}.
     *
     * @param count length of affected text before change starting at start in text
     * @param after length of affected text after change
     * @return true if there is a span before the cursor that is going to change state
     */
    private fun removeIfMarkSpans(count: Int, after: Int) {
        if (count - after == 1) {
            val editable = text ?: return
            val cursor = selectionStart
            editable.getSpans(cursor - 1, cursor - 1, ClickableSpan::class.java)?.firstOrNull()?.let {
                val start = editable.getSpanStart(it)
                val end = editable.getSpanEnd(it)
                if (cursor - 1 in (start + 1) until end) {
                    _removeMention = RemoveMention(clickableSpan = it, start = start, end = end)
                }
            }
        }
    }

    fun addExternalTextChangedListener(externalTextWatcher: TextWatcher) {
        this.externalTextWatcher = externalTextWatcher
    }

    class RemoveMention(val clickableSpan: ClickableSpan, val start: Int, val end: Int)
}
