/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.logger

import android.os.Environment
import android.os.Handler
import android.os.HandlerThread
import androidx.annotation.NonNull
import androidx.annotation.Nullable
import com.orhanobut.logger.DiskLogStrategy
import java.io.File
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale

/**
 * CSV formatted file logging for Android.
 * Writes to CSV the following data:
 * epoch timestamp, ISO8601 timestamp (human-readable), log level, tag, log message.
 */
class CsvFormatStrategy private constructor(@NonNull builder: Builder) : FormatStrategy {
    @NonNull
    private val date: Date?

    @NonNull
    private val dateFormat: SimpleDateFormat?

    @NonNull
    private val logStrategy: LogStrategy?

    @Nullable
    private val tag: String
    override fun log(priority: Int, @Nullable onceOnlyTag: String?, @NonNull message: String?) {
        if (message == null || onceOnlyTag == null) return
        var message = message
        Utils.checkNotNull(message)
        val tag = formatTag(onceOnlyTag)
        date!!.time = System.currentTimeMillis()
        val builder = StringBuilder()

        // machine-readable date/time
        builder.append(java.lang.Long.toString(date.time))

        // human-readable date/time
        builder.append(SEPARATOR)
        builder.append(dateFormat!!.format(date))

        // level
        builder.append(SEPARATOR)
        builder.append(Utils.logLevel(priority))

        // tag
        builder.append(SEPARATOR)
        builder.append(tag)

        // message
        if (message.contains(NEW_LINE)) {
            // a new line would break the CSV format, so we replace it here
            message = message.replace(NEW_LINE.toRegex(), NEW_LINE_REPLACEMENT)
        }
        builder.append(SEPARATOR)
        builder.append(message)

        // new line
        builder.append(NEW_LINE)
        logStrategy?.log(priority, tag, builder.toString())
    }

    @Nullable
    private fun formatTag(@Nullable tag: String): String {
        return if (!Utils.isEmpty(tag) && !Utils.equals(this.tag, tag)) {
            this.tag + "-" + tag
        } else this.tag
    }

    class Builder {
        var date: Date? = null
        var dateFormat: SimpleDateFormat? = null
        var logStrategy: LogStrategy? = null
        var tag = "PRETTY_LOGGER"

        @NonNull
        fun date(@Nullable `val`: Date?): Builder {
            date = `val`
            return this
        }

        @NonNull
        fun dateFormat(@Nullable `val`: SimpleDateFormat?): Builder {
            dateFormat = `val`
            return this
        }

        @NonNull
        fun logStrategy(@Nullable `val`: LogStrategy?): Builder {
            logStrategy = `val`
            return this
        }

        @NonNull
        fun tag(@Nullable tag: String): Builder {
            this.tag = tag
            return this
        }

        @NonNull
        fun build(): CsvFormatStrategy {
            if (date == null) {
                date = Date()
            }
            if (dateFormat == null) {
                dateFormat = SimpleDateFormat("yyyy.MM.dd HH:mm:ss.SSS", Locale.UK)
            }
            if (logStrategy == null) {
                val diskPath = Environment.getExternalStorageDirectory().absolutePath
                val folder = diskPath + File.separatorChar + "com/chat/logger"
                val ht = HandlerThread("AndroidFileLogger.$folder")
                ht.start()
                val handler: Handler = DiskLogStrategy.WriteHandler(ht.looper, folder, MAX_BYTES)
                logStrategy = DiskLogStrategy(handler)
            }
            return CsvFormatStrategy(this)
        }

        companion object {
            private const val MAX_BYTES = 500 * 1024 // 500K averages to a 4000 lines per file
        }
    }

    companion object {
        private val NEW_LINE = System.getProperty("line.separator")
        private const val NEW_LINE_REPLACEMENT = " <br> "
        private const val SEPARATOR = ","

        @JvmStatic
        @NonNull
        fun newBuilder(): Builder {
            return Builder()
        }
    }

    init {
        Utils.checkNotNull(builder)
        date = builder.date
        dateFormat = builder.dateFormat
        logStrategy = builder.logStrategy
        tag = builder.tag
    }
}
