/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.logger

import androidx.annotation.Keep
import androidx.annotation.NonNull
import androidx.annotation.Nullable

/**
 * <pre>
 * ┌────────────────────────────────────────────
 * │ LOGGER
 * ├┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄
 * │ Standard logging mechanism
 * ├┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄
 * │ But more pretty, simple and powerful
 * └────────────────────────────────────────────
</pre> *
 *
 * <h3>How to use it</h3>
 * Initialize it first
 * <pre>`
 * Logger.addLogAdapter(new AndroidLogAdapter());
`</pre> *
 *
 * And use the appropriate static Logger methods.
 *
 * <pre>`
 * Logger.d("debug");
 * Logger.e("error");
 * Logger.w("warning");
 * Logger.v("verbose");
 * Logger.i("information");
 * Logger.wtf("What a Terrible Failure");
`</pre> *
 *
 * <h3>String format arguments are supported</h3>
 * <pre>`
 * Logger.d("hello %s", "world");
`</pre> *
 *
 * <h3>Collections are support ed(only available for debug logs)</h3>
 * <pre>`
 * Logger.d(MAP);
 * Logger.d(SET);
 * Logger.d(LIST);
 * Logger.d(ARRAY);
`</pre> *
 *
 * <h3>Json and Xml support (output will be in debug level)</h3>
 * <pre>`
 * Logger.json(JSON_CONTENT);
 * Logger.xml(XML_CONTENT);
`</pre> *
 *
 * <h3>Customize Logger</h3>
 * Based on your needs, you can change the following settings:
 *
 *  * Different [LogAdapter]
 *  * Different [FormatStrategy]
 *  * Different [LogStrategy]
 *
 *
 * @see LogAdapter
 *
 * @see FormatStrategy
 *
 * @see LogStrategy
 */

@Keep
object Logger {
    const val VERBOSE = 2
    const val DEBUG = 3
    const val INFO = 4
    const val WARN = 5
    const val ERROR = 6
    const val ASSERT = 7

    @NonNull
    private var printer: Printer = LoggerPrinter()
    fun printer(@NonNull printer: Printer?) {
        Logger.printer = Utils.checkNotNull(printer)
    }

    fun addLogAdapter(@NonNull adapter: LogAdapter?) {
        printer.addAdapter(Utils.checkNotNull(adapter))
    }

    fun clearLogAdapters() {
        printer.clearLogAdapters()
    }

    /**
     * Given tag will be used as tag only once for this method call regardless of the tag that's been
     * set during initialization. After this invocation, the general tag that's been set will
     * be used for the subsequent log calls
     */
    fun t(@Nullable tag: String?): Printer? {
        return printer.t(tag)
    }

    /**
     * General log function that accepts all configurations as parameter
     */
    fun log(
        priority: Int,
        @Nullable tag: String?,
        @Nullable message: String?,
        @Nullable throwable: Throwable?
    ) {
        printer.log(priority, tag, message, throwable)
    }

    fun d(@NonNull message: String?, @Nullable vararg args: Any?) {
        printer.d(message, args)
    }

    fun d(@Nullable `object`: Any?) {
        printer.d(`object`)
    }

    fun e(@NonNull message: String?, @Nullable vararg args: Any?) {
        printer.e(message, args)
    }

    fun e(@Nullable throwable: Throwable?, @NonNull message: String?, @Nullable vararg args: Any?) {
        printer.e(throwable, message, args)
    }

    fun i(@NonNull message: String?, @Nullable vararg args: Any?) {
        printer.i(message, args)
    }

    fun v(@NonNull message: String?, @Nullable vararg args: Any?) {
        printer.v(message, args)
    }

    fun w(@NonNull message: String?, @Nullable vararg args: Any?) {
        printer.w(message, args)
    }

    /**
     * Tip: Use this for exceptional situations to log
     * ie: Unexpected errors etc
     */
    fun wtf(@NonNull message: String?, @Nullable vararg args: Any?) {
        printer.wtf(message, args)
    }

    /**
     * Formats the given json content and print it
     */
    fun json(@Nullable json: String?) {
        printer.json(json)
    }

    /**
     * Formats the given xml content and print it
     */
    fun xml(@Nullable xml: String?) {
        printer.xml(xml)
    }
}
