/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.logger

import androidx.annotation.NonNull
import androidx.annotation.Nullable

/**
 * Draws borders around the given log message along with additional information such as :
 *
 *
 *  * Thread information
 *  * Method stack trace
 *
 *
 * <pre>
 * ┌──────────────────────────
 * │ Method stack history
 * ├┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄
 * │ Thread information
 * ├┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄
 * │ Log message
 * └──────────────────────────
</pre> *
 *
 * <h3>Customize</h3>
 * <pre>`
 * FormatStrategy formatStrategy = PrettyFormatStrategy.newBuilder()
 * .showThreadInfo(false)  // (Optional) Whether to show thread info or not. Default true
 * .methodCount(0)         // (Optional) How many method line to show. Default 2
 * .methodOffset(7)        // (Optional) Hides internal method calls up to offset. Default 5
 * .logStrategy(customLog) // (Optional) Changes the log strategy to print out. Default LogCat
 * .tag("My custom tag")   // (Optional) Global tag for every log. Default PRETTY_LOGGER
 * .build();
`</pre> *
 */
class PrettyFormatStrategy private constructor(@NonNull builder: Builder) : FormatStrategy {
    private val methodCount: Int
    private val methodOffset: Int
    private val showThreadInfo: Boolean

    @NonNull
    private val logStrategy: LogStrategy?

    @Nullable
    private val tag: String
    override fun log(priority: Int, @Nullable onceOnlyTag: String?, @NonNull message: String?) {
        Utils.checkNotNull(message)
        val tag = formatTag(onceOnlyTag)
        logTopBorder(priority, tag)
        logHeaderContent(priority, tag, methodCount)

        // get bytes of message with system's default charset (which is UTF-8 for Android)
        val bytes = message!!.toByteArray()
        val length = bytes.size
        if (length <= CHUNK_SIZE) {
            if (methodCount > 0) {
                logDivider(priority, tag)
            }
            logContent(priority, tag, message)
            logBottomBorder(priority, tag)
            return
        }
        if (methodCount > 0) {
            logDivider(priority, tag)
        }
        var i = 0
        while (i < length) {
            val count = Math.min(length - i, CHUNK_SIZE)
            // create a new String with system's default charset (which is UTF-8 for Android)
            logContent(priority, tag, String(bytes, i, count))
            i += CHUNK_SIZE
        }
        logBottomBorder(priority, tag)
    }

    private fun logTopBorder(logType: Int, @Nullable tag: String) {
        logChunk(logType, tag, TOP_BORDER)
    }

    private fun logHeaderContent(logType: Int, @Nullable tag: String, methodCount: Int) {
        var methodCount = methodCount
        val trace = Thread.currentThread().stackTrace
        if (showThreadInfo) {
            logChunk(
                logType,
                tag,
                HORIZONTAL_LINE.toString() + " Thread: " + Thread.currentThread().name
            )
            logDivider(logType, tag)
        }
        var level = ""
        val stackOffset = getStackOffset(trace) + methodOffset

        // corresponding method count with the current stack may exceeds the stack trace. Trims the count
        if (methodCount + stackOffset > trace.size) {
            methodCount = trace.size - stackOffset - 1
        }
        for (i in methodCount downTo 1) {
            val stackIndex = i + stackOffset
            if (stackIndex >= trace.size) {
                continue
            }
            val builder = StringBuilder()
            builder.append(HORIZONTAL_LINE)
                .append(' ')
                .append(level)
                .append(getSimpleClassName(trace[stackIndex].className))
                .append(".")
                .append(trace[stackIndex].methodName)
                .append(" ")
                .append(" (")
                .append(trace[stackIndex].fileName)
                .append(":")
                .append(trace[stackIndex].lineNumber)
                .append(")")
            level += "   "
            logChunk(logType, tag, builder.toString())
        }
    }

    private fun logBottomBorder(logType: Int, @Nullable tag: String) {
        logChunk(logType, tag, BOTTOM_BORDER)
    }

    private fun logDivider(logType: Int, @Nullable tag: String) {
        logChunk(logType, tag, MIDDLE_BORDER)
    }

    private fun logContent(logType: Int, @Nullable tag: String, @NonNull chunk: String?) {
        Utils.checkNotNull(chunk)
        val lines = chunk!!.split(System.getProperty("line.separator")).toTypedArray()
        for (line in lines) {
            logChunk(logType, tag, "$HORIZONTAL_LINE $line")
        }
    }

    private fun logChunk(priority: Int, @Nullable tag: String, @NonNull chunk: String) {
        Utils.checkNotNull(chunk)
        logStrategy!!.log(priority, tag, chunk)
    }

    private fun getSimpleClassName(@NonNull name: String): String {
        Utils.checkNotNull(name)
        val lastIndex = name.lastIndexOf(".")
        return name.substring(lastIndex + 1)
    }

    /**
     * Determines the starting index of the stack trace, after method calls made by this class.
     *
     * @param trace the stack trace
     * @return the stack offset
     */
    private fun getStackOffset(@NonNull trace: Array<StackTraceElement>): Int {
        Utils.checkNotNull(trace)
        var i = MIN_STACK_OFFSET
        while (i < trace.size) {
            val e = trace[i]
            val name = e.className
            if (name != LoggerPrinter::class.java.name && name != Logger::class.java.name) {
                return --i
            }
            i++
        }
        return -1
    }

    @Nullable
    private fun formatTag(@Nullable tag: String?): String {
        return if (!Utils.isEmpty(tag) && !Utils.equals(this.tag, tag)) {
            this.tag + "-" + tag
        } else this.tag
    }

    class Builder {
        var methodCount = 2
        var methodOffset = 0
        var showThreadInfo = true

        @Nullable
        var logStrategy: LogStrategy? = null

        @Nullable
        var tag = "PRETTY_LOGGER"

        @NonNull
        fun methodCount(`val`: Int): Builder {
            methodCount = `val`
            return this
        }

        @NonNull
        fun methodOffset(`val`: Int): Builder {
            methodOffset = `val`
            return this
        }

        @NonNull
        fun showThreadInfo(`val`: Boolean): Builder {
            showThreadInfo = `val`
            return this
        }

        @NonNull
        fun logStrategy(@Nullable `val`: LogStrategy?): Builder {
            logStrategy = `val`
            return this
        }

        @NonNull
        fun tag(@Nullable tag: String): Builder {
            this.tag = tag
            return this
        }

        @NonNull
        fun build(): PrettyFormatStrategy {
            if (logStrategy == null) {
                logStrategy = LogcatLogStrategy()
            }
            return PrettyFormatStrategy(this)
        }
    }

    companion object {
        /**
         * Android's max limit for a log entry is ~4076 bytes,
         * so 4000 bytes is used as chunk size since default charset
         * is UTF-8
         */
        private const val CHUNK_SIZE = 4000

        /**
         * The minimum stack trace index, starts at this class after two native calls.
         */
        private const val MIN_STACK_OFFSET = 5

        /**
         * Drawing toolbox
         */
        private const val TOP_LEFT_CORNER = '┌'
        private const val BOTTOM_LEFT_CORNER = '└'
        private const val MIDDLE_CORNER = '├'
        private const val HORIZONTAL_LINE = '│'
        private const val DOUBLE_DIVIDER = "────────────────────────────────────────────────────────"
        private const val SINGLE_DIVIDER = "┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄"
        private const val TOP_BORDER =
            TOP_LEFT_CORNER.toString() + DOUBLE_DIVIDER + DOUBLE_DIVIDER
        private const val BOTTOM_BORDER =
            BOTTOM_LEFT_CORNER.toString() + DOUBLE_DIVIDER + DOUBLE_DIVIDER
        private const val MIDDLE_BORDER =
            MIDDLE_CORNER.toString() + SINGLE_DIVIDER + SINGLE_DIVIDER

        @NonNull
        fun newBuilder(): Builder {
            return Builder()
        }
    }

    init {
        Utils.checkNotNull(builder)
        methodCount = builder.methodCount
        methodOffset = builder.methodOffset
        showThreadInfo = builder.showThreadInfo
        logStrategy = builder.logStrategy
        tag = builder.tag
    }
}
