/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.network

import android.annotation.SuppressLint
import androidx.annotation.Keep
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import com.google.firebase.FirebaseApp
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.R
import com.netacom.base.chat.android_utils.NetworkUtils
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.type.RequestStatus
import com.netacom.base.chat.util.IsoDateJsonAdapter
import com.squareup.moshi.JsonAdapter
import com.squareup.moshi.Moshi
import com.squareup.moshi.Types
import com.squareup.moshi.kotlin.reflect.KotlinJsonAdapterFactory
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineExceptionHandler
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody
import okhttp3.RequestBody.Companion.asRequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import retrofit2.Response
import java.io.File
import java.io.InterruptedIOException
import java.net.UnknownHostException
import java.util.Date

@Keep
open class BaseRepository {
    @SuppressLint("MissingPermission")
    suspend fun <T : Any> makeNormalApiCall(call: suspend () -> Response<T>): ApiResponse<T> {
        if (!NetworkUtils.isConnected()) {
            return ApiResponseError(
                errorCode = RequestStatus.NO_INTERNET_CONNECTION.value,
                message = StringUtils.getString(
                    R.string.common_error_connect
                )
            )
        }
        return try {
            val response = call.invoke()
            if (response.isSuccessful) {
                ApiResponseSuccess(data = response.body())
            } else {
                when (response.code()) {
                    RequestStatus.UNAUTHORIZED.value -> {
                        ApiResponseError(
                            errorCode = RequestStatus.UNAUTHORIZED.value,
                            message = StringUtils.getString(R.string.common_error_unknown)
                        )
                    }
                    RequestStatus.GATEWAY_TIMEOUT.value,
                    RequestStatus.SERVICE_UNAVAILABLE.value,
                    RequestStatus.BAD_GATEWAY.value,
                    RequestStatus.INTERNAL_SERVER_ERROR.value -> ApiResponseError(
                        errorCode = response.code(),
                        message = StringUtils.getString(R.string.common_error_unknown)
                    )
                    else -> ApiResponseError(
                        errorCode = response.code(),
                        message = StringUtils.getString(R.string.error_server_msg)
                    )
                }
            }
        } catch (e: CancellationException) {
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseCrashlytics.getInstance().recordException(e)
            }
            ApiResponseError(
                errorCode = RequestStatus.CANCEL_REQUEST.value
            )
        } catch (e: InterruptedIOException) {
            e.printStackTrace()
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseCrashlytics.getInstance().recordException(e)
            }
            ApiResponseError(
                errorCode = RequestStatus.CLIENT_TIMEOUT.value,
                message = StringUtils.getString(R.string.error_server_msg)
            )
        } catch (ex: Exception) {
            ex.printStackTrace()
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseCrashlytics.getInstance().recordException(ex)
            }
            ApiResponseError(
                errorCode = RequestStatus.ERROR_CLIENT.value,
                message = StringUtils.getString(R.string.common_error_unknown)
            )
        }
    }
    @SuppressLint("MissingPermission")
    suspend fun <T : Any> makeApiCall(call: suspend () -> Response<ApiResponse<T>>): ApiResponse<T> {
        if (!NetworkUtils.isConnected()) {
            return ApiResponseError(
                errorCode = RequestStatus.NO_INTERNET_CONNECTION.value,
                message = StringUtils.getString(
                    R.string.common_error_connect
                )
            )
        }
        return try {
            val response = call.invoke()
            if (response.isSuccessful) {
                response.body()?.let {
                    if (it.data != null) {
                        it
                    } else {
                        ApiResponseError(
                            message = StringUtils.getString(R.string.common_error_unknown),
                            errorCode = response.code()
                        )
                    }
                } ?: run {
                    ApiResponseSuccess<T>(data = null)
                }
            } else {
                when (response.code()) {
                    RequestStatus.UNAUTHORIZED.value -> ApiResponseError(
                        errorCode = RequestStatus.UNAUTHORIZED.value,
                        message = StringUtils.getString(R.string.login_required)
                    )
                    RequestStatus.GATEWAY_TIMEOUT.value,
                    RequestStatus.SERVICE_UNAVAILABLE.value,
                    RequestStatus.BAD_GATEWAY.value,
                    RequestStatus.INTERNAL_SERVER_ERROR.value -> ApiResponseError(
                        errorCode = response.code(),
                        message = StringUtils.getString(R.string.error_server_msg)
                    )
                    else -> handleError(response)
                }
            }
        } catch (e: CancellationException) {
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseCrashlytics.getInstance().recordException(e)
            }
            ApiResponseError(
                errorCode = RequestStatus.CANCEL_REQUEST.value
            )
        } catch (e: InterruptedIOException) {
            e.printStackTrace()
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseCrashlytics.getInstance().recordException(e)
            }
            ApiResponseError(
                errorCode = RequestStatus.CLIENT_TIMEOUT.value,
                message = StringUtils.getString(R.string.error_server_msg)
            )
        } catch (ex: Exception) {
            ex.printStackTrace()
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseCrashlytics.getInstance().recordException(ex)
            }
            ApiResponseError(
                errorCode = RequestStatus.ERROR_CLIENT.value,
                message = StringUtils.getString(R.string.common_error_unknown)
            )
        }
    }
    private fun <T : Any> handleError(response: Response<ApiResponse<T>>?): ApiResponse<T> {
        var errorResponse: ApiResponse<Any>? = null
        response?.errorBody()?.let {
            try {
                val json = it.string()
                if (json.isNotEmpty()) {
                    val payloadType = Types.newParameterizedType(
                        ApiResponse::class.java,
                        Any::class.java
                    )
                    val jsonAdapter: JsonAdapter<ApiResponse<Any>> = Moshi.Builder()
                        .addLast(KotlinJsonAdapterFactory())
                        .add(Date::class.java, IsoDateJsonAdapter())
                        .build().adapter(payloadType)
                    errorResponse = jsonAdapter.lenient().nullSafe().fromJson(json)
                }
            } catch (e: Exception) {
                FirebaseCrashlytics.getInstance().recordException(e)
                e.printStackTrace()
            }
        }
        return ApiResponseError(
            message = StringUtils.getString(R.string.error_server_msg), // errorResponse?.message ?:
            errorCode = response?.code(),
            error = errorResponse?.error
        )
    }
    protected fun getBodyPart(file: File, field: String): MultipartBody.Part {
        val fileReqBody = file.asRequestBody("image/*".toMediaTypeOrNull())
        return MultipartBody.Part.createFormData(field, file.name, fileReqBody)
    }
    protected fun toRequestBody(value: String): RequestBody {
        return value.toRequestBody("text/plain".toMediaTypeOrNull())
    }

    fun <T> LiveData<T>.post(data: T) = (this as MutableLiveData<T>).postValue(data)
    val handlerError = CoroutineExceptionHandler { _, exception ->
        exception.printStackTrace()
        when (exception) {
            is UnknownHostException -> {
                Logger.e("NetworkErrorException:NO_INTERNET")
            }
            else -> {
                Logger.e("CoroutineExceptionHandler")
            }
        }
    }
}
