/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.security.biometric

import android.hardware.biometrics.BiometricManager.BIOMETRIC_ERROR_HW_UNAVAILABLE
import android.hardware.biometrics.BiometricManager.BIOMETRIC_ERROR_NONE_ENROLLED
import android.hardware.biometrics.BiometricManager.BIOMETRIC_ERROR_NO_HARDWARE
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.biometric.BiometricPrompt.ERROR_CANCELED
import androidx.biometric.BiometricPrompt.ERROR_HW_NOT_PRESENT
import androidx.biometric.BiometricPrompt.ERROR_HW_UNAVAILABLE
import androidx.biometric.BiometricPrompt.ERROR_LOCKOUT
import androidx.biometric.BiometricPrompt.ERROR_LOCKOUT_PERMANENT
import androidx.biometric.BiometricPrompt.ERROR_NEGATIVE_BUTTON
import androidx.biometric.BiometricPrompt.ERROR_NO_BIOMETRICS
import androidx.biometric.BiometricPrompt.ERROR_NO_DEVICE_CREDENTIAL
import androidx.biometric.BiometricPrompt.ERROR_NO_SPACE
import androidx.biometric.BiometricPrompt.ERROR_TIMEOUT
import androidx.biometric.BiometricPrompt.ERROR_UNABLE_TO_PROCESS
import androidx.biometric.BiometricPrompt.ERROR_USER_CANCELED
import androidx.biometric.BiometricPrompt.ERROR_VENDOR
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import com.netacom.base.chat.coroutine.CoroutineDispatcherProvider
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.channels.sendBlocking
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.callbackFlow
import kotlinx.coroutines.flow.flowOn
import java.security.Signature
import java.util.concurrent.Executor
import javax.crypto.Cipher
import javax.crypto.Mac

class BiometricAuth constructor(
    private val biometricPromptExecutor: Executor,
    private val biometricCipher: BiometricCipher,
    private val biometricManager: BiometricManager,
    private val dispatcherProvider: CoroutineDispatcherProvider
) : Biometric {

    override fun canAuthenticate(): BiometricAvailability =
        when (biometricManager.canAuthenticate()) {
            BIOMETRIC_ERROR_NO_HARDWARE -> BiometricAvailability.NoHardware
            BIOMETRIC_ERROR_NONE_ENROLLED -> BiometricAvailability.UserNotEnrolled
            BIOMETRIC_ERROR_HW_UNAVAILABLE -> BiometricAvailability.Retry
            else -> BiometricAvailability.Success
        }

    @ExperimentalCoroutinesApi
    override fun authenticate(
        fragmentActivity: FragmentActivity,
        authPromptInfo: AuthPromptInfo
    ): Flow<BiometricResult> =
        callbackFlow<BiometricResult> {
            BiometricPrompt(
                fragmentActivity,
                biometricPromptExecutor,
                object : BiometricPrompt.AuthenticationCallback() {
                    override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                        super.onAuthenticationError(errorCode, errString)
                        sendBlocking(
                            BiometricResult.Error(errorCode.biometricError(), errString.toString())
                        )
                    }

                    override fun onAuthenticationFailed() {
                        super.onAuthenticationFailed()
                        sendBlocking(BiometricResult.Failed)
                    }

                    override fun onAuthenticationSucceeded(
                        result: BiometricPrompt.AuthenticationResult
                    ) {
                        super.onAuthenticationSucceeded(result)
                        sendBlocking(
                            biometricAuthResult(
                                result.cryptoObject?.mac,
                                result.cryptoObject?.cipher,
                                result.cryptoObject?.signature
                            )
                        )
                    }
                }
            ).run {
                authenticate(
                    authPromptInfo.biometricPromptInfo(),
                    BiometricPrompt.CryptoObject(biometricCipher.encryptCipher)
                )
            }
            awaitClose()
        }.flowOn(dispatcherProvider.main())

    @ExperimentalCoroutinesApi
    override fun authenticate(
        fragment: Fragment,
        authPromptInfo: AuthPromptInfo
    ): Flow<BiometricResult> =
        authenticate(fragment.requireActivity(), authPromptInfo)

    private fun biometricAuthResult(
        mac: Mac?,
        cipher: Cipher?,
        signature: Signature?
    ): BiometricResult = when {
        mac != null -> BiometricResult.AuthMac(mac = mac)
        cipher != null -> BiometricResult.AuthCipher(cipher = cipher)
        signature != null -> BiometricResult.AuthSignature(signature = signature)
        else -> BiometricResult.Failed
    }

    private fun AuthPromptInfo.biometricPromptInfo() = BiometricPrompt.PromptInfo.Builder().also {
        it.setTitle(promptTitle)
        it.setNegativeButtonText(promptNegativeButtonText)
        if (!promptSubtitle.isNullOrBlank()) {
            it.setSubtitle(promptSubtitle)
        }
        if (deviceCredentialAllowed != null) {
            it.setDeviceCredentialAllowed(deviceCredentialAllowed)
        }
    }.build()

    private fun Int.biometricError() = when (this) {
        ERROR_HW_UNAVAILABLE -> BiometricError.Retry
        ERROR_UNABLE_TO_PROCESS -> BiometricError.UnableToProcess
        ERROR_TIMEOUT -> BiometricError.Timeout
        ERROR_NO_SPACE -> BiometricError.NoSpace
        ERROR_CANCELED -> BiometricError.Canceled
        ERROR_LOCKOUT -> BiometricError.Lockout
        ERROR_VENDOR -> BiometricError.VendorSpecific
        ERROR_LOCKOUT_PERMANENT -> BiometricError.LocoutPermanent
        ERROR_USER_CANCELED -> BiometricError.UserCanceled
        ERROR_NO_BIOMETRICS -> BiometricError.NoBiometricsEnrolled
        ERROR_HW_NOT_PRESENT -> BiometricError.HardweareNotPresent
        ERROR_NEGATIVE_BUTTON -> BiometricError.NegativeButtonPressed
        ERROR_NO_DEVICE_CREDENTIAL -> BiometricError.NoDeviceCredentials
        else -> BiometricError.Unknown
    }
}
