/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.security.biometric

import java.security.Signature
import javax.crypto.Cipher
import javax.crypto.Mac

enum class BiometricAvailability {
    NoHardware,
    UserNotEnrolled,
    Retry,
    Success
}

enum class BiometricError {
    /**
     * The hardware is unavailable. Try again later.
     */
    Retry,

    /**
     * Error state returned when the sensor was unable to process the current image.
     */
    UnableToProcess,

    /**
     * Error state returned when the current request has been running too long. This is intended to
     * prevent programs from waiting for the biometric sensor indefinitely. The timeout is platform
     * and sensor-specific, but is generally on the order of 30 seconds.
     */
    Timeout,

    /**
     * Error state returned for operations like enrollment; the operation cannot be completed
     * because there's not enough storage remaining to complete the operation.
     */
    NoSpace,

    /**
     * The operation was canceled because the biometric sensor is unavailable. For example, this may
     * happen when the user is switched, the device is locked or another pending operation prevents
     * or disables it.
     */
    Canceled,

    /**
     * The operation was canceled because the API is locked out due to too many attempts.
     * This occurs after 5 failed attempts, and lasts for 30 seconds.
     */
    Lockout,

    /**
     * Hardware vendors may extend this list if there are conditions that do not fall under one of
     * the above categories. Vendors are responsible for providing error strings for these errors.
     * These messages are typically reserved for internal operations such as enrollment, but may be
     * used to express vendor errors not otherwise covered. Applications are expected to show the
     * error message string if they happen, but are advised not to rely on the message id since they
     * will be device and vendor-specific
     */
    VendorSpecific,

    /**
     * The operation was canceled because ERROR_LOCKOUT occurred too many times.
     * Biometric authentication is disabled until the user unlocks with strong authentication
     * (PIN/Pattern/Password)
     */
    LocoutPermanent,

    /**
     * The user canceled the operation. Upon receiving this, applications should use alternate
     * authentication (e.g. a password). The application should also provide the means to return to
     * biometric authentication, such as a "use <biometric>" button.
     */
    UserCanceled,

    /**
     * The user does not have any biometrics enrolled.
     */
    NoBiometricsEnrolled,

    /**
     * The device does not have a biometric sensor.
     */
    HardweareNotPresent,

    /**
     * The user pressed the negative button.
     */
    NegativeButtonPressed,

    /**
     * The device does not have pin, pattern, or password set up.
     */
    NoDeviceCredentials,

    /**
     * Anything else
     */
    Unknown
}

sealed class BiometricResult {
    data class Error(val reason: BiometricError, val description: String) : BiometricResult()
    data class AuthMac(val mac: Mac) : BiometricResult()
    data class AuthCipher(val cipher: Cipher) : BiometricResult()
    data class AuthSignature(val signature: Signature) : BiometricResult()
    object Failed : BiometricResult()
}
