/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

@file:Suppress("NULLABILITY_MISMATCH_BASED_ON_JAVA_ANNOTATIONS")

package com.netacom.base.chat.util

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.res.ColorStateList
import android.os.Bundle
import android.util.TypedValue
import android.view.View
import android.widget.EditText
import android.widget.TextView
import android.widget.Toast
import androidx.annotation.ColorInt
import androidx.annotation.ColorRes
import androidx.annotation.DimenRes
import androidx.annotation.DrawableRes
import androidx.core.content.ContextCompat
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.Fragment
import kotlinx.coroutines.suspendCancellableCoroutine
import java.io.Closeable
import java.text.DecimalFormat
import java.text.Normalizer
import java.util.regex.Matcher
import java.util.regex.Pattern

@ColorInt
fun Context.getColorCompat(@ColorRes resourceId: Int) = ContextCompat.getColor(this, resourceId)

fun Context.getDrawableCompat(@DrawableRes resId: Int) = ContextCompat.getDrawable(this, resId)

fun Context.getDimension(@DimenRes resourceId: Int) = resources.getDimension(resourceId)

fun Context.dimen(@DimenRes resourceId: Int): Int = resources.getDimensionPixelSize(resourceId)

fun TextView.setBackgroundTheme(color: Int) {
    this.backgroundTintList = ColorStateList.valueOf(
        color
    )
}

fun Context.resNameById(resId: Int?): String? {
    return resId?.run {
        resources.getResourceEntryName(this)
    }
}

fun Context.resIdByNameColor(resIdName: String?): Int? {
    return resIdName?.run {
        resources.getIdentifier(resIdName, "color", packageName)
    }
}

fun Context.resIdByNameDrawable(resIdName: String?): Int? {
    return resIdName?.run {
        resources.getIdentifier(resIdName, "drawable", packageName)
    }
}

val Context.screenWidth: Int
    get() = resources.displayMetrics.widthPixels

val Context.screenHeight: Int
    get() = resources.displayMetrics.heightPixels

fun Context.toast(msg: String, duration: Int = Toast.LENGTH_SHORT) {
    Toast.makeText(this, msg, duration).show()
}

fun Context.toast(stringId: Int, duration: Int = Toast.LENGTH_SHORT) {
    Toast.makeText(this, stringId, duration).show()
}

fun Context.getPxFromDp(dp: Float) = TypedValue
    .applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, resources.displayMetrics).toInt()

fun Fragment.getStringFormat(idString: Int, param: Int? = 0): String =
    String.format(getString(idString), param)

fun Fragment.getStringFormat(idString: Int, param: String? = ""): String =
    String.format(getString(idString), param)

fun Fragment.dimen(res: Int): Int = resources.getDimensionPixelSize(res)

fun View.OnClickListener.listenToViews(vararg views: View?) =
    views.map { it?.setOnClickListener(this) }

fun View.OnKeyListener.listenKeyToViews(vararg views: View) =
    views.map { it.setOnKeyListener(this) }

fun TextView.OnEditorActionListener.listenEditor(vararg views: EditText) =
    views.forEach { it.setOnEditorActionListener(this) }

fun View.OnFocusChangeListener.focusToViews(vararg views: View) =
    views.map { it.onFocusChangeListener = this }

fun DialogFragment.getHeight(): Int = resources.displayMetrics.heightPixels

fun textEmpty(vararg view: TextView?) = view.map { it?.text = "" }

inline fun <T : Fragment> T.withArgs(argsBuilder: Bundle.() -> Unit): T =
    this.apply {
        arguments = Bundle().apply(argsBuilder)
    }

inline fun <reified T : Any> newIntent(context: Context): Intent =
    Intent(context, T::class.java)

inline fun <reified T : Any> Activity.launchActivity(
    requestCode: Int = -1,
    options: Bundle? = null,
    noinline init: Intent.() -> Unit = {}
) {
    val intent = newIntent<T>(this)
    intent.init()
    startActivityForResult(intent, requestCode, options)
}

inline fun <reified T : Any> Context.launchActivity(
    options: Bundle? = null,
    noinline init: Intent.() -> Unit = {}
) {
    val intent = newIntent<T>(this)
    intent.init()
    startActivity(intent, options)
}

fun <T> List<T>.replace(newValue: T, block: (T) -> Boolean): List<T> {
    return map {
        if (block(it)) newValue else it
    }
}

fun String.unAccentText(): String {
    val temp = Normalizer.normalize(this, Normalizer.Form.NFD)
    return "\\p{InCombiningDiacriticalMarks}+".toRegex().replace(temp, "").replace("đ", "d")
}

/*println("11".toPriceAmount())
println("05".toPriceAmount())
println(12.0.toPriceAmount())*/
fun String.toPriceAmount(): String {
    val dec = DecimalFormat("###,###,###.00")
    return dec.format(this.toDouble())
}

fun Double.toPriceAmount(): String {
    val dec = DecimalFormat("###,###,###.00")
    return dec.format(this)
}

fun getOnlyDigits(s: String?): String {
    val pattern: Pattern = Pattern.compile("[^0-9]")
    val matcher: Matcher = pattern.matcher(s)
    return matcher.replaceAll("")
}

suspend inline fun <T : Closeable?, R> T.useCancelLazy(
    crossinline block: (T) -> R
): R = suspendCancellableCoroutine { cont ->
    cont.invokeOnCancellation { this?.close() }
    cont.resume(use(block)) { error ->
        error.printStackTrace()
    }
}
