/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.util

import android.app.ActivityManager
import android.content.Context
import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.Color
import androidx.core.content.ContextCompat
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.livedata.EventLiveData
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.transform
import kotlinx.coroutines.launch
import java.io.File
import java.security.MessageDigest
import java.util.regex.Pattern

// val md5Hash = "test".md5 // 098f6bcd4621d373cade4e832627b4f6
// val sha1Hash = "test".sha1 // a94a8fe5ccb19ba61c4c0873d391e987982fbbd3
val String.md5: String
    get() {
        val bytes = MessageDigest.getInstance("MD5").digest(this.toByteArray())
        return bytes.joinToString("") {
            "%02x".format(it)
        }
    }

val String.sha1: String
    get() {
        val bytes = MessageDigest.getInstance("SHA-1").digest(this.toByteArray())
        return bytes.joinToString("") {
            "%02x".format(it)
        }
    }

/*val email = "test@email.com"
if (email.isEmailValid()) {
    print("Email is valid. Continue registration")
} else {
    print("Email is not validate. Show error")
}*/
fun String.isEmailValid(): Boolean {
    val expression = "^[\\w.-]+@([\\w\\-]+\\.)+[A-Z]{2,8}$"
    val pattern = Pattern.compile(expression, Pattern.CASE_INSENSITIVE)
    val matcher = pattern.matcher(this)
    return matcher.matches()
}

fun String.isLocalLink(): Boolean {
    return startsWith("/storage") || startsWith("/data/user")
}

/*
val cl = "Contains letters".containsLatinLetter // true
val cnl = "12345".containsLatinLetter // false
val cd = "Contains digits 123".containsDigit // true
val istr = "123".isIntegerNumber // true
val dstr = "12.9".toDecimalNumber // true
*/

val String?.isValid: Boolean
    get() = this?.toLongOrNull() != null

val String.containsLatinLetter: Boolean
    get() = matches(Regex(".*[A-Za-z].*"))

val String.containsDigit: Boolean
    get() = matches(Regex(".*[0-9].*"))

val String.isAlphanumeric: Boolean
    get() = matches(Regex("[A-Za-z0-9]*"))

val String.hasLettersAndDigits: Boolean
    get() = containsLatinLetter && containsDigit

val String.isIntegerNumber: Boolean
    get() = toIntOrNull() != null

val String.toDecimalNumber: Boolean
    get() = toDoubleOrNull() != null

val Any?.isNull
    get() = when (this) {
        this == null -> true
        is Collection<*>? -> this?.size ?: 0 == 0
        is String? -> this?.length == 0 || isNullOrBlank()
        is Long? -> this == 0L
        else -> this == null
    }

val Any?.isNotNull
    get() = !isNull

/*
val lpc1 = "https://google.com/chrome/".lastPathComponent // chrome
val lpc2 = "C:\\Windows\\Fonts\\font.ttf".lastPathComponent // font.ttf
val lpc3 = "/dev/null".lastPathComponent // null
*/
val String.lastPathComponent: String
    get() {
        var path = this
        if (path.endsWith("/")) {
            path = path.substring(0, path.length - 1)
        }
        var index = path.lastIndexOf('/')
        if (index < 0) {
            if (path.endsWith("\\")) {
                path = path.substring(0, path.length - 1)
            }
            index = path.lastIndexOf('\\')
            if (index < 0) {
                return path
            }
        }
        return path.substring(index + 1)
    }

/*
val colorHex = "#010203"
val color = colorHex.asColor // -16711165
val nonColorHex = "abcdef"
val nonColor = nonColorHex.asColor // null
val color1 = Color.parseColor(hexString)
val color2 = hexString.asColor
*/
val String.asColor: Int?
    get() = try {
        Color.parseColor(this)
    } catch (e: java.lang.IllegalArgumentException) {
        null
    }

fun isResource(context: Context?, resId: Int): Boolean {
    if (context != null) {
        try {
            return context.resources.getResourceName(resId) != null
        } catch (ignore: Resources.NotFoundException) {
        }
    }
    return false
}

fun getThemeColor(resId: Int): Int {
    return if (isResource(Utils.getApp(), resId)) ContextCompat.getColor(Utils.getApp(), resId)
    else resId
}

/*
val ccFormatted = "1234567890123456".creditCardFormatted // "1234 5678 9012 3456"
*/
val String.creditCardFormatted: String
    get() {
        val preparedString = replace(" ", "").trim()
        val result = StringBuilder()
        for (i in preparedString.indices) {
            if (i % 4 == 0 && i != 0) {
                result.append(" ")
            }
            result.append(preparedString[i])
        }
        return result.toString()
    }

/**
 * Returns a flow which performs the given [action] on each value of the original flow's [EventLiveData].
 */
fun <T> Flow<EventLiveData<T?>>.onEachEvent(action: suspend (T) -> Unit): Flow<T> = transform { value ->
    value.getContentIfNotHandled()?.let {
        action(it)
        return@transform emit(it)
    }
}

private fun File.writeBitmap(bitmap: Bitmap, format: Bitmap.CompressFormat, quality: Int) {
    outputStream().use { out ->
        bitmap.compress(format, quality, out)
        out.flush()
    }
}

fun Context.isServiceRunning(serviceClass: Class<*>): Boolean {
    val manager = this.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
    return manager.getRunningServices(Integer.MAX_VALUE)
        .any { it.service.className == serviceClass.name }
}

fun <T> debounce(
    delayMillis: Long = 500L,
    scope: CoroutineScope,
    action: (T) -> Unit
): (T) -> Unit {
    var debounceJob: Job? = null
    return { param: T ->
        if (debounceJob == null) {
            debounceJob = scope.launch {
                action(param)
                delay(delayMillis)
                debounceJob = null
            }
        }
    }
}
