/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.base.chat.util

import android.util.Patterns
import com.netacom.base.chat.R
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.type.ValidationResult
import com.netacom.base.chat.type.ValidationType
import java.util.regex.Pattern

/**
 * Class for validation of fields (phone, email, imei, address, password, ...)
 */
class ValidationUtil {
    companion object {
        @JvmStatic
        fun getMessageRes(validationType: ValidationType, validation: ValidationResult): String = StringUtils.getString(
            when (validationType) {
                ValidationType.EMAIL -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_email_invalid_format_msg
                    ValidationResult.EMPTY -> R.string.validate_email_empty_msg
                    ValidationResult.WRONG -> R.string.validate_email_wrong_msg
                    ValidationResult.EXISTED -> R.string.validate_email_existed_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_email_email_not_exist_msg
                    else -> R.string.validate_valid
                }
                ValidationType.PHONE_NUMBER -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_phone_invalid_format_msg
                    ValidationResult.EMPTY -> R.string.validate_phone_empty_msg
                    ValidationResult.WRONG -> R.string.validate_phone_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_phone_not_exist_msg
                    ValidationResult.EXISTED -> R.string.validate_phone_existed_msg
                    else -> R.string.validate_valid
                }
                ValidationType.FULL_NAME -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_name_invalid_format_msg
                    ValidationResult.EMPTY -> R.string.validate_name_empty_msg
                    ValidationResult.WRONG -> R.string.validate_name_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_name_not_exist_msg
                    ValidationResult.EXISTED -> R.string.validate_name_existed_msg
                    else -> R.string.validate_valid
                }
                ValidationType.USER_NAME -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_name_invalid_format_msg
                    ValidationResult.EMPTY -> R.string.validate_name_empty_msg
                    ValidationResult.WRONG -> R.string.validate_name_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_name_not_exist_msg
                    ValidationResult.EXISTED -> R.string.validate_name_existed_msg
                    ValidationResult.INVALID_LENGTH -> R.string.validate_name_invalid_length_msg
                    else -> R.string.validate_valid
                }
                ValidationType.OTP -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_code_invalid_msg
                    ValidationResult.EMPTY -> R.string.validate_code_empty_msg
                    ValidationResult.WRONG -> R.string.validate_code_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_code_not_exist_msg
                    ValidationResult.EXISTED -> R.string.validate_code_existed_msg
                    else -> R.string.validate_valid
                }
                ValidationType.CONFIRM_PASSWORD -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.EMPTY -> R.string.validate_confirm_password_empty_msg
                    ValidationResult.NOT_MATCH -> R.string.validate_confirm_password_not_match_msg
                    else -> R.string.validate_valid
                }
                ValidationType.PASSWORD -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_password_invalid_format_msg
                    ValidationResult.EMPTY -> R.string.validate_password_empty_msg
                    ValidationResult.WRONG -> R.string.validate_password_wrong_msg
                    ValidationResult.INVALID_LENGTH -> R.string.validate_password_invalid_length_msg
                    ValidationResult.MISSING_DIGIT -> R.string.validate_password_invalid_digit_msg
                    ValidationResult.MISSING_SPECIAL_CHARACTER -> R.string.validate_password_invalid_special_character_msg
                    ValidationResult.MISSING_LOWER_CASE_CHARACTER -> R.string.validate_password_invalid_lower_case_msg
                    ValidationResult.MISSING_UPPER_CASE_CHARACTER -> R.string.validate_password_invalid_upper_case_msg
                    else -> R.string.validate_valid
                }
                ValidationType.IMEI -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_imei_invalid_msg
                    ValidationResult.EMPTY -> R.string.validate_imei_empty_msg
                    ValidationResult.WRONG -> R.string.validate_imei_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_imei_not_exist_msg
                    else -> R.string.validate_valid
                }
                ValidationType.ID_CARD_NUMBER -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_id_card_invalid_msg
                    ValidationResult.EMPTY -> R.string.validate_id_card_empty_msg
                    ValidationResult.WRONG -> R.string.validate_id_card_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_id_card_not_exist_msg
                    else -> R.string.validate_valid
                }
                ValidationType.PASSPORT_NUMBER -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.INVALID -> R.string.validate_passport_invalid_msg
                    ValidationResult.EMPTY -> R.string.validate_passport_empty_msg
                    ValidationResult.WRONG -> R.string.validate_passport_wrong_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_passport_not_exist_msg
                    else -> R.string.validate_valid
                }
                ValidationType.ORGANIZATION_CODE -> when (validation) {
                    ValidationResult.VALID -> R.string.validate_valid
                    ValidationResult.EMPTY -> R.string.validate_confirm_org_empty_msg
                    ValidationResult.NOT_EXIST -> R.string.validate_confirm_org_wrong_msg
                    else -> R.string.validate_valid
                }
                else -> R.string.validate_valid
            }
        )

        @JvmStatic
        fun isValid(result: ValidationResult) = (result == ValidationResult.VALID)

        @JvmStatic
        fun validateEmail(email: String?): ValidationResult = when {
            email.isNullOrBlank() -> ValidationResult.EMPTY
            Patterns.EMAIL_ADDRESS.matcher(email).matches() -> ValidationResult.VALID
            else -> ValidationResult.INVALID
        }

        @JvmStatic
        fun validateOrg(org: String?): ValidationResult = when {
            org.isNullOrBlank() -> ValidationResult.EMPTY
            else -> ValidationResult.VALID
        }

        @JvmStatic
        fun validateName(name: String?): ValidationResult = when {
            name.isNullOrBlank() -> ValidationResult.EMPTY
            !Pattern.matches("^[\\p{L} .'-]+$", name) -> ValidationResult.INVALID
            else -> ValidationResult.VALID
        }

        @JvmStatic
        fun validateUserName(name: String?): ValidationResult = when {
            name.isNullOrBlank() -> ValidationResult.EMPTY
            name.length < 3 -> ValidationResult.INVALID_LENGTH
            Pattern.matches("[a-zA-Z0-9_]+", name) -> ValidationResult.VALID
            else -> ValidationResult.INVALID
        }

        @JvmStatic
        fun validatePhone(phone: String?): ValidationResult = when {
            phone.isNullOrBlank() -> ValidationResult.EMPTY
            Patterns.PHONE.matcher(phone).matches() && phone.length in 8..15 -> ValidationResult.VALID
            else -> ValidationResult.INVALID
        }

        @JvmStatic
        fun validatePassword(password: String?): ValidationResult = when {
            password.isNullOrBlank() -> ValidationResult.EMPTY
            password.length < 6 -> ValidationResult.INVALID_LENGTH
            password == password.toLowerCase() -> ValidationResult.MISSING_UPPER_CASE_CHARACTER
            password == password.toUpperCase() -> ValidationResult.MISSING_LOWER_CASE_CHARACTER
            !Pattern.compile(".*\\d+.*").matcher(password).matches() -> ValidationResult.MISSING_DIGIT
            else -> ValidationResult.VALID
        }

        @JvmStatic
        fun validatePasswordNormal(password: String?): ValidationResult = when {
            password.isNullOrBlank() -> ValidationResult.EMPTY
            password.length < 6 -> ValidationResult.INVALID_LENGTH
            else -> ValidationResult.VALID
        }

        @JvmStatic
        fun validateConfirmPassword(password: String?, confirmPassword: String?): ValidationResult = when {
            confirmPassword.isNullOrBlank() -> ValidationResult.EMPTY
            confirmPassword == password -> ValidationResult.VALID
            else -> ValidationResult.NOT_MATCH
        }

        @JvmStatic
        fun validateOTP(otp: String?): ValidationResult = when {
            otp.isNullOrBlank() -> ValidationResult.EMPTY
            otp.length == 6 && otp.matches("\\d+".toRegex()) -> ValidationResult.VALID
            else -> ValidationResult.INVALID
        }

        @JvmStatic
        fun validateIMEI(imei: String?): ValidationResult = when {
            imei.isNullOrBlank() -> ValidationResult.EMPTY
            imei.length > 20 -> ValidationResult.INVALID_LENGTH
            // isValidIMEI(imei) -> ValidationResult.VALID
            else -> ValidationResult.VALID
        }

        @JvmStatic
        fun validateIdCardNumber(id: String?): ValidationResult = when {
            id.isNullOrBlank() -> ValidationResult.EMPTY
            (id.length == 9 || id.length == 12) && id.matches("\\d+".toRegex()) -> ValidationResult.VALID
            else -> ValidationResult.INVALID
        }

        @JvmStatic
        fun validatePassport(passport: String?): ValidationResult = when {
            passport.isNullOrBlank() -> ValidationResult.EMPTY
            passport.length < 6 || passport.length > 20 -> ValidationResult.INVALID
            !passport.matches("[a-zA-Z0-9]+".toRegex()) -> ValidationResult.INVALID
            else -> ValidationResult.VALID
        }

        private fun isValidIMEI(imeiStr: String): Boolean {
            if (!Pattern.compile("\\d+").matcher(imeiStr).matches() || imeiStr.length != 15) return false

            var imeiNumber = imeiStr.toLong()

            var d: Int
            var sum = 0
            for (i in 15 downTo 1) {
                d = (imeiNumber % 10).toInt()
                if (i % 2 == 0) {
                    d *= 2 // Doubling every alternate digit
                }
                sum += sumDig(d) // Finding sum of the digits
                imeiNumber /= 10
            }

            return sum % 10 == 0 && sum != 0
        }

        // Function for finding and returning sum of digits of a number
        private fun sumDig(num: Int): Int {
            var n = num
            var a = 0
            while (n > 0) {
                a += n % 10
                n /= 10
            }
            return a
        }
    }
}
