package com.netacom.full.basechat

import android.annotation.SuppressLint
import android.net.Uri
import android.os.Bundle
import androidx.activity.result.contract.ActivityResultContracts
import androidx.databinding.ViewDataBinding
import androidx.fragment.app.setFragmentResultListener
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.android_utils.ImageUtils
import com.netacom.base.chat.android_utils.UriUtils
import com.netacom.base.chat.base.BaseViewModel
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.network.ResultData
import com.netacom.full.ui.main.profile.SelectMediaFragment
import com.netacom.lite.entity.ui.local.LocalFileModel
import com.netacom.lite.util.CompressUtil
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.File
import java.io.IOException

abstract class BaseCoreCallCameraBottomDialog<DB : ViewDataBinding, VM : BaseViewModel>(
    layoutID: Int,
    viewModelClass: Class<VM>
) : BaseSDKBottomDialog<DB, VM>(layoutID, viewModelClass) {
    private val _capture = MutableLiveData<EventLiveData<Uri>>()
    val capture: LiveData<EventLiveData<Uri>> = _capture
    private var fileUri: Uri? = null
    private val getCameraImage = registerForActivityResult(ActivityResultContracts.TakePicture()) { success ->
        if (success) {
            fileUri?.let { uri ->
                _capture.postValue(EventLiveData(uri))
            }
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        resultListener()
    }

    // Call
    fun callEvent() {
        (viewModel as? BaseSDKViewModel)?.apply {
            startCallingScreen.observeOnce { resultData ->
                when (resultData) {
                    is ResultData.Success -> {
                        resultData.data?.let { _neGroup ->
                            openCall(neGroup = _neGroup, isVideoEnable = _neGroup.isCallVideo)
                        }
                    }
                    else -> Unit
                }
            }
        }
    }

    fun checkCall(receiverId: Long?, isVideoEnable: Boolean) {
        if (receiverId == 0L) return
        checkPermissionCall(
            {
                (viewModel as? BaseSDKViewModel)?.makeNewCall(receiverId ?: 0L, isVideoEnable)
            },
            {
            }
        )
    }

    // Capture
    private fun resultListener() {
        setFragmentResultListener(SelectMediaFragment.OPEN_CAMERA) { _, _ ->
            checkPermissionCamera(
                {
                    requestOpenCamera()
                },
                {
                }
            )
        }
        setFragmentResultListener(SelectMediaFragment.SELECT_PHOTO_VIDEO) { _, bundle ->
            bundle.getParcelableArrayList<LocalFileModel>("data")?.let { media ->
                if (!media.isNullOrEmpty()) {
                    CoroutineScope(Dispatchers.IO).launch {
                        val rawFile = File(media[0].filePath)
                        fileUri = UriUtils.file2Uri(rawFile)
                        fileUri?.let { uri ->
                            _capture.postValue(EventLiveData(uri))
                        }
                    }
                }
            }
        }
    }

    @Suppress("DEPRECATION")
    @SuppressLint("QueryPermissionsNeeded")
    private fun requestOpenCamera() {
        try {
            CoroutineScope(Dispatchers.IO).launch {
                CompressUtil.createTempImageFile()?.also {
                    val imageSize = ImageUtils.getSize(it.path)
                    ImageUtils.rotate(ImageUtils.getBitmap(it.path), ImageUtils.getRotateDegree(it.path), (imageSize[0] / 2).toFloat(), (imageSize[1] / 2).toFloat(), true)
                    fileUri = UriUtils.file2Uri(it)
                    getCameraImage.launch(fileUri)
                }
            }
        } catch (ex: IOException) {
            ex.printStackTrace()
            FirebaseCrashlytics.getInstance().recordException(ex)
        }
    }
}
