/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.basechat

import android.content.Context
import android.database.ContentObserver
import android.net.Uri
import android.os.Handler
import android.os.Looper
import android.provider.ContactsContract
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.navigation.NavController
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.base.BaseViewModel
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.base.chat.network.ResultData
import com.netacom.full.R
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.extensions.navigateIfSafe
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.MessageType
import com.netacom.lite.define.NavigationDef
import com.netacom.lite.entity.socket.Group
import com.netacom.lite.entity.socket.Message
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.repository.GroupRepository
import com.netacom.lite.repository.SocketRepository
import com.netacom.lite.socket.request.BlockUserRequest
import com.netacom.lite.socket.request.CreateMessageRequest
import com.netacom.lite.socket.request.GroupRequest
import com.netacom.lite.socket.request.UpdateBackgroundRequest
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants
import com.netacom.lite.util.TimerSupport
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.withContext
import javax.inject.Inject

/**Created by vantoan on 18/Sep/2020
Company: Netacom.
Email: hvtoan.dev@gmail.com
 **/

@HiltViewModel
open class BaseSDKViewModel @Inject constructor(
    private val socketRepository: SocketRepository,
    private val groupRepository: GroupRepository,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>
) : BaseViewModel() {

    private val isShowLog = true

    private val _contactChange = MutableLiveData<EventLiveData<Boolean>>()
    val contactChange: LiveData<EventLiveData<Boolean>> = _contactChange

    private val _startCallingScreen = MutableLiveData<EventLiveData<ResultData<NeGroup>>>()
    val startCallingScreen: LiveData<EventLiveData<ResultData<NeGroup>>> = _startCallingScreen

    val getIOSocket get() = socketRepository.getIOSocket

    val getPreferences get() = socketRepository.getPreferences

    val getDbManager get() = socketRepository.getDbManager

    val getPostExecutionThread get() = socketRepository.getPostExecutionThread

    val getUser get() = socketRepository.getUser

    val getUserId get() = socketRepository.getUserId

    val getUserName get() = socketRepository.getUserName

    val getUserToken get() = socketRepository.getUserToken

    fun appId() = getPreferences.sdkConfig?.appId

    /**
     * Quick fix to avoid reload list group when open the call screen. Will refactor it later.
     */
    var isCalling = false

    private val _updateGroup = MutableLiveData<EventLiveData<NeGroup>>()
    val updateGroup: LiveData<EventLiveData<NeGroup>> = _updateGroup

    suspend fun createGroup(type: Int, listIDs: List<Long>, groupName: String, avatar: String) = getIOSocket.createGroup(
        ownerId = getPreferences.getUserId,
        type = type,
        listIDs = listIDs,
        groupName = groupName,
        avatar = avatar,
        senderName = getPreferences.user?.getDisplayName,
        senderAvatar = getPreferences.user?.getDisplayAvatar
    )

    suspend fun updateGroup(
        groupId: Long,
        name: String?,
        avatar: String? = null,
        description: String? = null,
        owner: String? = null,
        pullList: List<Long>? = null,
        pushList: List<Long>? = null,
        background: UpdateBackgroundRequest? = null
    ) =
        getIOSocket.updateGroup(
            groupId = groupId,
            name = name,
            avatar = avatar,
            description = description,
            owner = owner,
            pull = pullList,
            push = pushList,
            background = background
        )

    suspend fun checkGroup(partnerUin: Long): ResultData<Group> {
        val listIDs = listOf(getPreferences.getUserId, partnerUin)
        return getIOSocket.checkGroup(
            type = GroupType.GROUP_TYPE_PUBLIC,
            listIDs = listIDs
        )
    }

    fun openChat(neGroup: NeGroup, sdk: Boolean = false) {
        navigationDispatcher.emit {
            if (sdk) {
                it.navigateIfSafe(R.id.open_chat_sdk, NavigationDef.ARGUMENT_NEGROUP to neGroup)
            } else {
                it.navigateIfSafe(R.id.open_chat, NavigationDef.ARGUMENT_NEGROUP to neGroup)
            }
        }
    }

    fun onBack(callbackResult: () -> Unit? = { }) {
        navigationDispatcher.emit {
            if (!it.popBackStack()) {
                callbackResult()
            }
        }
    }

    fun listenerContactChange(context: Context) {
        context.contentResolver?.registerContentObserver(
            ContactsContract.Contacts.CONTENT_URI,
            true,
            object : ContentObserver(Handler(Looper.getMainLooper())) {
                override fun onChange(selfChange: Boolean, uri: Uri?) {
                    super.onChange(selfChange, uri)
                    _contactChange.post(EventLiveData(true))
                }
            }
        )
    }

    /**
     * Call this function to create the group(connection between Caller & Receiver)
     * Success -> make a new call
     * Fail -> stop Call
     */

    fun openCall(neGroup: NeGroup, isVideoEnable: Boolean) {
        Logger.e("openCall==$isVideoEnable")
        navigationDispatcher.emit {
            isCalling = true
            it.navigateIfSafe(
                actionId = R.id.make_call,
                params = arrayOf(NavigationDef.ARGUMENT_NEGROUP to neGroup, NavigationDef.ARGUMENT_CALL to isVideoEnable)
            )
        }
    }

    fun makeNewCall(receiverId: Long, isVideoEnable: Boolean) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val listIDs = listOf(getUserId, receiverId)
            createGroup(
                type = GroupType.GROUP_TYPE_PUBLIC,
                listIDs = listIDs,
                groupName = Constants.EMPTY,
                avatar = Constants.EMPTY
            ).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        updateGroupToDB(it)?.let { neGroup ->
                            neGroup.blockers?.let { listUser ->
                                if (listUser.isNotEmpty()) {
                                    if (listUser[0].id != getUserId) {
                                        if (isShowLog) Logger.d("User Was Blocked!")
                                        screenStateError.postErrorOnce(StringUtils.getString(R.string.str_user_blocked))
                                    } else {
                                        neGroup.members?.firstOrNull { member ->
                                            member.id != getUserId
                                        }?.let {
                                            if (isShowLog) Logger.d("Owner Blocked!")
                                            screenStateError.postErrorOnce(StringUtils.getString(R.string.str_owner_blocked))
                                        }
                                    }
                                } else {
                                    neGroup.isCallVideo = isVideoEnable
                                    _startCallingScreen.post(EventLiveData(ResultData.Success(neGroup)))
                                }
                            } ?: run {
                                neGroup.isCallVideo = isVideoEnable
                                _startCallingScreen.post(EventLiveData(ResultData.Success(neGroup)))
                            }
                        }
                    }
                    is ResultData.Failed -> {
                        if (isShowLog) Logger.d("Can't make a call, need to retry something.....")
                    }
                    else -> {
                        if (isShowLog) Logger.d("Can't make a call, need to retry something.....")
                    }
                }
            }
        }
    }

    suspend fun updateGroupToDB(group: Group, isForce: Boolean = false): NeGroup? = withContext(getPostExecutionThread.io) {
        var neGroup: NeGroup? = null
        if (group.name == null || isForce) {
            groupRepository.getGroupInfoById(group.groupId).collect { state ->
                when (state) {
                    is ApiResponseSuccess -> {
                        state.data?.let { infoGroupResponse ->
                            infoGroupResponse.partnerInfos?.let { groupRepository.saveListPartner(it) }
                            infoGroupResponse.group?.let { group ->
                                groupRepository.saveMessageToDB(group)
                                neGroup = groupRepository.saveGroupToDB(group)
                            }
                        }
                    }
                }
            }
        } else {
            groupRepository.saveMessageToDB(group)
            neGroup = groupRepository.saveGroupToDB(group)
        }
        return@withContext neGroup
    }

    fun startOneByOneChat(neUser: NeUser, sdk: Boolean = false) {
        val partnerUid = neUser.id
        partnerUid?.let {
            launchOnViewModelScope(getPostExecutionThread.io) {
                checkGroup(partnerUin = partnerUid).let { result ->
                    when (result) {
                        is ResultData.Success -> result.data?.let {
                            updateGroupToDB(group = it, isForce = true)?.let { neGroup ->
                                Logger.e("group exits==$neGroup")
                                openChat(neGroup, sdk)
                            }
                        }
                        is ResultData.Failed -> {
                            val tempGroup = NeGroup()
                            tempGroup.owner = NeUser(id = getUserId, username = getUserName, token = getUserToken)
                            tempGroup.name = neUser.getDisplayName
                            tempGroup.members = listOf(neUser)
                            tempGroup.type = GroupType.GROUP_TYPE_PUBLIC
                            Logger.e("group not exits , create temp group$tempGroup")
                            openChat(tempGroup, sdk)
                        }
                        else -> {
                            screenStateError.postErrorOnce("start chat error $result")
                            Logger.e("start chat error $result")
                        }
                    }
                }
            }
        } ?: kotlin.run {
            screenStateError.postErrorOnce(StringUtils.getString(R.string.popup_value_is_null))
            Logger.e(StringUtils.getString(R.string.popup_value_is_null))
        }
    }

    fun updateSeenForLatestMessage(groupId: String?, messageId: String?, status: Int) {
        getIOSocket.updateMessage(
            group_id = groupId,
            message_id = messageId,
            message = Constants.EMPTY,
            status = status
        )
    }

    fun convertToNeGroup(group: Group): NeGroup = groupRepository.convertToNeGroup(group)

    fun showSingleMember(neUser: NeUser) {
        navigationDispatcher.emit {
            it.navigateIfSafe(R.id.open_member, "neUser" to neUser)
        }
    }

    fun checkGroupExitsInDb(neGroup: NeGroup?): Boolean {
        val groupId = neGroup?.id ?: return false
        groupRepository.getGroupDbById(groupId)?.let {
            return true
        } ?: return false
    }

    fun blockUser(neGroup: NeGroup, blockedUid: Long, callbackResult: CallbackResult<Boolean>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val groupId = neGroup.id?.toLongOrNull() ?: 0L
            val blockUserRequest = BlockUserRequest(groupId = groupId, blockedUins = blockedUid)
            getIOSocket.blockUser(blockUserRequest).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let { success ->
                        if (success) {
                            groupRepository.updateBlockUsers(mutableListOf(blockedUid), groupId)?.apply {
                                updateGroup(groupRepository.convertDBToNeGroup(this))
                                callbackResult.callBackSuccess(true)
                            }
                        } else {
                            callbackResult.callBackError(result.data.toString())
                        }
                    }
                    is ResultData.Failed -> {
                        withContext(getPostExecutionThread.main) {
                            callbackResult.callBackError(result.message)
                        }
                    }
                    else -> {
                    }
                }
            }
        }
    }

    fun unblockUser(groupId: Long, unblockedUid: Long, callbackResult: CallbackResult<NeGroup>? = null) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val blockUserRequest = BlockUserRequest(groupId = groupId, blockedUins = unblockedUid)
            getIOSocket.unBlockUser(blockUserRequest).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let { success ->
                        if (success) {
                            groupRepository.updateUnblockUsers(mutableListOf(unblockedUid), groupId)?.apply {
                                groupRepository.convertDBToNeGroup(this)?.let { neGroup ->
                                    updateGroup(neGroup)
                                    callbackResult?.callBackSuccess(neGroup)
                                }
                            }
                        } else {
                            withContext(getPostExecutionThread.main) {
                                callbackResult?.callBackError(result.data.toString())
                            }
                        }
                    }
                    is ResultData.Failed -> {
                        withContext(getPostExecutionThread.main) {
                            callbackResult?.callBackError(result.message)
                        }
                    }
                    else -> {
                    }
                }
            }
        }
    }

    fun groupActionNotification(neGroup: NeGroup, callbackResult: CallbackResult<NeGroup>? = null) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val groupId = neGroup.id?.toLongOrNull() ?: 0L
            val muteGroupRequest = GroupRequest(groupId = groupId)
            if (neGroup.isMute) {
                /**
                 * mute -> unmute
                 * */
                getIOSocket.unMuteGroup(
                    muteGroupRequest
                ).let { result ->
                    when (result) {
                        is ResultData.Success -> result.data?.let {
                            groupRepository.updateMuteGroupById(groupId = groupId, isMute = false)
                                ?.apply {
                                    groupRepository.convertDBToNeGroup(this)?.apply {
                                        updateGroup(this)
                                        callbackResult?.callBackSuccess(result = this)
                                    }
                                }
                        }
                        is ResultData.Failed -> callbackResult?.callBackError()
                        else -> {
                        }
                    }
                }
            } else {
                /**
                 * unmute -> mute
                 * */
                getIOSocket.muteGroup(
                    muteGroupRequest
                ).let { result ->
                    when (result) {
                        is ResultData.Success -> result.data?.let {
                            groupRepository.updateMuteGroupById(groupId = groupId, isMute = true)
                                ?.apply {
                                    groupRepository.convertDBToNeGroup(this)?.apply {
                                        updateGroup(this)
                                        callbackResult?.callBackSuccess(result = this)
                                    }
                                }
                        }
                        is ResultData.Failed -> callbackResult?.callBackError()
                        else -> {
                        }
                    }
                }
            }
        }
    }

    fun updateGroup(neGroup: NeGroup) {
        _updateGroup.post(EventLiveData(neGroup))
    }

    fun openChooseMediaFragment(isChat: Boolean) {
        navigationDispatcher.emit {
            it.navigateIfSafe(R.id.choose_media, NavigationDef.MEDIA_IS_CHAT to isChat)
        }
    }

    fun createUpdateGroup(
        neGroup: NeGroup,
        updatedGroup: NeMessage,
        isAddMember: Boolean,
        callbackResult: CallbackResult<Message>
    ) {
        if (neGroup.id == null) return
        // send message update group
        launchOnViewModelScope(getPostExecutionThread.io) {
            val attachmentMessage = updatedGroup.let { neMessage ->
                groupRepository.convertNeMessageToMessage(neMessage)
            }
            val myName = getPreferences.user?.getDisplayName
            val myAvatar = getPreferences.user?.getDisplayAvatar
            val tempCreatedAt = TimerSupport.nowNano()
            val tempMessageId = (tempCreatedAt + 1).toString()
            val createMessageRequest = CreateMessageRequest(
                groupId = neGroup.id.toString(),
                senderName = myName,
                senderAvatar = myAvatar,
                message = null,
                attachments = StringUtils.handleSpecialChar(
                    text = attachmentMessage.attachments,
                    isTextType = attachmentMessage.type != MessageType.MESSAGE_TYPE_IMAGE
                ),
                nonce = tempMessageId,
                isEncrypted = neGroup.isSecretChat(),
                version = Constants.MESSAGE_VERSION.toInt(),
                type = MessageType.MESSAGE_TYPE_GROUP_UPDATE
            )
            Logger.e("log attachment == ${createMessageRequest.attachments}")
            socketRepository.createMessage(createMessageRequest = createMessageRequest)
                .let { result ->
                    when (result) {
                        is ResultData.Success -> result.data?.let {
                            callbackResult.callBackSuccess(it)
                        }
                        is ResultData.Failed -> {
                        }
                        is ResultData.Loading -> Unit
                        is ResultData.Exception -> {
                        }
                    }
                }
        }
    }

    fun showFailureMessage(message: String? = null) {
        screenStateError.postErrorOnce("${StringUtils.getString(R.string.common_failure)} $message")
    }

    fun isLeaderInGroup(neGroup: NeGroup): Boolean {
        val ownerId = neGroup.owner?.id ?: return false
        return ownerId == getUserId
    }
}
