/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.databinding

import android.annotation.SuppressLint
import android.content.res.ColorStateList
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.view.View
import android.widget.ImageView
import android.widget.ProgressBar
import android.widget.RadioButton
import android.widget.TextView
import androidx.appcompat.widget.AppCompatButton
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import androidx.core.view.ViewCompat
import androidx.core.widget.ImageViewCompat
import androidx.databinding.BindingAdapter
import com.google.android.material.bottomnavigation.BottomNavigationView
import com.google.android.material.tabs.TabLayout
import com.netacom.base.chat.android_utils.SpanUtils
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.imageloader.loadImage
import com.netacom.base.chat.type.Language
import com.netacom.base.chat.util.TimeUtils
import com.netacom.base.chat.util.getColorCompat
import com.netacom.base.chat.util.getThemeColor
import com.netacom.base.chat.util.isNotNull
import com.netacom.full.R
import com.netacom.lite.define.CallDefine.MEDIA_TYPE_AUDIO
import com.netacom.lite.define.CallDefine.MEDIA_TYPE_VIDEO
import com.netacom.lite.define.MessageStatusType
import com.netacom.lite.entity.ui.call.NeCallLogItem
import com.netacom.lite.entity.ui.group.NeGroup

@BindingAdapter("groupLastMessage")
fun TextView.groupLastMessage(content: String?) {
    content?.apply {
        text = if (this == StringUtils.getString(R.string.message_cancel_call) || this == StringUtils.getString(R.string.message_cancel_video)) {
            SpanUtils().append(this).setForegroundColor(Color.RED).create().toString()
        } else {
            SpanUtils().append(this).setForegroundColor(ContextCompat.getColor(context, R.color.color_sub_text_2)).create().toString()
        }
    }
}

@BindingAdapter("setImageColorTheme")
fun ImageView.setImageColorTheme(colorRes: Int) {
    if (colorRes == 0) return
    ImageViewCompat.setImageTintList(
        this,
        ColorStateList.valueOf(getThemeColor(colorRes))
    )
}

@BindingAdapter("setColorTheme")
fun TextView.setTextColorTheme(colorRes: Int) {
    if (colorRes == 0) return
    setTextColor(getThemeColor(colorRes))
}

@BindingAdapter("setBackgroundColorTheme")
fun View.setBackgroundColorTheme(colorRes: Int) {
    if (colorRes == 0) return
    backgroundTintList = ColorStateList.valueOf(
        getThemeColor(colorRes)
    )
}

fun AppCompatButton.setButtonTheme(colorRes: Int) {
    backgroundTintList = ColorStateList.valueOf(
        getThemeColor(colorRes)
    )
}

fun BottomNavigationView.setupNavigationTheme(colorStateList: ColorStateList) {
    itemIconTintList = colorStateList
    itemTextColor = colorStateList
}

fun TabLayout.setTabLayoutTheme(colorRes: Int) {
    // tabTextColors = ColorStateList.valueOf(getThemeColor(colorRes))
    setTabTextColors(ContextCompat.getColor(context, R.color.color_grey_959595), getThemeColor(colorRes))
    setSelectedTabIndicatorColor(getThemeColor(colorRes))
}

fun ProgressBar.setProgressTintListTheme(colorRes: Int) {
    indeterminateDrawable.colorFilter = PorterDuffColorFilter(getThemeColor(colorRes), PorterDuff.Mode.MULTIPLY)
}

@SuppressLint("SetTextI18n")
@BindingAdapter("callLogMessage")
fun TextView.callLogMessage(item: NeCallLogItem?) {
    item?.callLog?.let {
        val callType = when (it.mediaType) {
            MEDIA_TYPE_AUDIO -> {
                context.resources.getString(R.string.voice_call)
            }
            MEDIA_TYPE_VIDEO -> {
                context.resources.getString(R.string.video_call)
            }
            else -> {
                ""
            }
        }
        val duration = getDuration(item)
        val durationText = kotlin.run {
            if (duration > 0L) {
                com.netacom.base.chat.android_utils.TimeUtils.millis2String(duration)
            } else ""
        }
        if (it.callerId?.toLongOrNull() == item.user?.id) {
            /**
             * my call
             * */
            text = if (!it.isMissCall) {
                setCompoundDrawablesWithIntrinsicBounds(R.drawable.ic_calllog_callout, 0, 0, 0)
                setTextColor(ContextCompat.getColor(context, R.color.color_text_time_row_call_log))
                if (durationText.isNotEmpty()) {
                    "$callType ($durationText)"
                } else {
                    callType
                }
            } else {
                setCompoundDrawablesWithIntrinsicBounds(R.drawable.ic_calllog_misscall_out, 0, 0, 0)
                setTextColor(ContextCompat.getColor(context, R.color.color_red_B3D31027))
                callType
            }
        } else {
            /**
             * partner call
             * */
            text = if (!it.isMissCall) {

                /**
                 * accept call
                 * */
                setCompoundDrawablesWithIntrinsicBounds(R.drawable.ic_calllog_callin, 0, 0, 0)
                setTextColor(ContextCompat.getColor(context, R.color.color_text_time_row_call_log))
                if (durationText.isNotEmpty()) {
                    "$callType ($durationText)"
                } else {
                    callType
                }
            } else {
                /**
                 * miss call
                 * */
                setCompoundDrawablesWithIntrinsicBounds(R.drawable.ic_calllog_misscall_in, 0, 0, 0)
                setTextColor(ContextCompat.getColor(context, R.color.color_red_B3D31027))
                callType
            }
        }
    }
}

@BindingAdapter("callLogItemStatus")
fun ImageView.callLogItemStatus(item: NeCallLogItem?) {
    item?.let {
        if (it.callLog?.callerId?.toLongOrNull() == it.user?.id) {
            if (it.callLog?.isMissCall == false) {
                setImageResource(R.drawable.ic_calllog_callout)
            } else {
                setImageResource(R.drawable.ic_calllog_misscall_out)
            }
        } else {
            /**
             * partner call
             * */
            if (it.callLog?.isMissCall == false) {
                /**
                 * accept call
                 * */
                setImageResource(R.drawable.ic_calllog_callin)
            } else {
                /**
                 * miss call
                 * */
                setImageResource(R.drawable.ic_calllog_misscall_in)
            }
        }
    }
}

@BindingAdapter("callLogMessageStatus")
fun TextView.callLogMessageStatus(item: NeCallLogItem?) {
    item?.let {
        if (it.callLog?.isMissCall == false) {
            val duration = getDuration(it)
            setTextColor(ContextCompat.getColor(context, R.color.color_black_B34b4955))
            callLogTime(duration)
        }
    }
}

@BindingAdapter("callLogMessageType")
fun TextView.callLogMessageType(mediaType: Int?) {
    text = when (mediaType) {
        MEDIA_TYPE_AUDIO -> context.resources.getString(R.string.voice_call)
        MEDIA_TYPE_VIDEO -> context.resources.getString(R.string.video_call)
        else -> context.resources.getString(R.string.blank)
    }
}

@BindingAdapter("callLogColor")
fun TextView.callLogColor(isMissCall: Boolean) {
    when (isMissCall) {
        true -> setTextColor(ContextCompat.getColor(context, R.color.color_red_B3D31027))
        false -> setTextColor(ContextCompat.getColor(context, R.color.color_text))
    }
}

@BindingAdapter("callLogTime")
fun TextView.callLogTime(createAt: Long?) {
    createAt?.let { time ->
        text = com.netacom.base.chat.android_utils.TimeUtils.millis2String(time)
    }
}

@BindingAdapter("lastSent")
fun TextView.lastSent(time: String?) {
    setTextColor(ContextCompat.getColor(context, R.color.color_text_time_row_call_log))
    time?.let {
        if (time.length > 10) {
            time.substring(0, 10).toLongOrNull()?.let {
                text = TimeUtils.toDate(it * 1000)
            }
        } else {
            time.toLongOrNull()?.let {
                text = TimeUtils.toDate(it * 1000)
            }
        }
    }
}

@BindingAdapter(value = ["timeAgo", "language"], requireAll = false)
fun TextView.timeAgo(_time: Long, language: String) {
    if (_time == 0L) return
    val time = run {
        if (_time.toString().length > 10) _time.toString().substring(0, 10).toLong()
        else _time
    }

    var result = ""
    if (time > 0L) {
        val new: String
        val labels = if (language == Language.ENGLISH.value) {
            new = "Just now"
            mutableListOf("day", "hour", "min", "second")
        } else {
            new = "mới gửi"
            mutableListOf("ngày", "giờ", "phút", "giây")
        }

        val times = TimeUtils.distanceTime(time)

        for (index in 0 until times.count()) {
            if (times[index] > 0) {
                result = if (index == 0 && times[index] > 7) {
                    TimeUtils.toDate(time * 1000)
                } else if (index == 3) {
                    new
                } else {
                    if (times[index] < 2) {
                        "${times[index]} ${labels[index]}"
                    } else {
                        if (language == Language.ENGLISH.value) "${times[index]} ${labels[index]}s"
                        else "${times[index]} ${labels[index]}"
                    }
                }
                break
            } else {
                if (index == times.count() - 1) {
                    result = new
                }
            }
        }
    }
    text = result
}

@BindingAdapter(value = ["neGroup", "myId"], requireAll = true)
fun ImageView.messageSentStatus(neGroup: NeGroup?, myId: Long) {
    neGroup?.lastMessage?.let { lastMessage ->
        // Logger.e("messageSentStatus==$lastMessage")
        visibility = if (lastMessage.owner?.id == myId) {
            when (lastMessage.status) {
                MessageStatusType.MESSAGE_DELIVERED, MessageStatusType.MESSAGE_SEEN -> setImageResource(R.drawable.ic_double_check)
                MessageStatusType.MESSAGE_SENT -> setImageResource(R.drawable.ic_single_check)
                MessageStatusType.MESSAGE_FAILED -> setImageResource(R.drawable.ic_sent_fail)
                else -> setImageResource(android.R.color.transparent)
            }
            View.VISIBLE
        } else {
            View.GONE
        }
    }
}

@BindingAdapter("setTint")
fun ImageView.setTint(colorRes: Int) {
    if (colorRes == 0) return
    ImageViewCompat.setImageTintList(this, ColorStateList.valueOf(context.getColorCompat(colorRes)))
}

@BindingAdapter("setTint")
fun TextView.setTint(colorRes: Int) {
    if (colorRes == 0) return
    setTextColor(context.getColorCompat(colorRes))
}

@BindingAdapter("setTint")
fun View.setTint(colorRes: Int) {
    if (colorRes == 0) return
    setBackgroundColor(context.getColorCompat(colorRes))
}

@BindingAdapter("setSubBackground")
fun View.setSubBackground(colorRes: Int) {
    if (colorRes == 0) return
    ViewCompat.setBackgroundTintList(
        this,
        ContextCompat.getColorStateList(
            context,
            colorRes
        )
    )
}

@BindingAdapter("setTintMenu")
fun BottomNavigationView.setTint(colorRes: Int) {
    itemIconTintList = ContextCompat.getColorStateList(context, colorRes)
    itemTextColor = ContextCompat.getColorStateList(context, colorRes)
}

@BindingAdapter("setButtonColorTheme")
fun RadioButton.setRadioButtonColorTheme(colorRes: Int) {
    if (colorRes == 0) return
    buttonTintList = ColorStateList.valueOf(
        context.getColorCompat(colorRes)
    )
}

@BindingAdapter("android:src")
fun setImageViewResource(imageView: AppCompatImageView, resource: Int) {
    imageView.setImageResource(resource)
}

@BindingAdapter("android:text")
fun setTextView(textView: AppCompatTextView, resource: String?) {
    resource?.let {
        textView.text = resource
    }
}

@BindingAdapter("loadImage")
fun <T> AppCompatImageView.image(path: T?) {
    loadImage(path)
}

@BindingAdapter("callLogDetailMessageStatus")
fun TextView.callLogDetailMessageStatus(item: NeCallLogItem?) {
    item?.let { callLogItem ->
        val callType = when (callLogItem.callLog?.mediaType) {
            MEDIA_TYPE_AUDIO -> context.resources.getString(R.string.voice_call)
            MEDIA_TYPE_VIDEO -> context.resources.getString(R.string.video_call)
            else -> context.resources.getString(R.string.blank)
        }
        val isCallOut = callLogItem.callLog?.callerId?.toLongOrNull() == callLogItem.user?.id
        val isMissCall = callLogItem.callLog?.isMissCall == true

        val callStatus = if (isMissCall) {
            if (isCallOut) context.resources.getString(R.string.text_cancelled) else context.resources.getString(R.string.text_missed)
        } else {
            if (isCallOut) context.resources.getString(R.string.text_out) else context.resources.getString(R.string.text_in)
        }

        text = "$callType $callStatus"
    }
}

private fun getDuration(item: NeCallLogItem): Long {
    return when {
        item.callLog?.connectedAt.isNotNull -> (item.callLog?.stopAt ?: 0L) - (item.callLog?.connectedAt ?: 0L)
        item.callLog?.acceptAt.isNotNull -> (item.callLog?.stopAt ?: 0L) - (item.callLog?.acceptAt ?: 0L)
        else -> 0
    }
}
