package com.netacom.full.databinding

import android.annotation.SuppressLint
import android.graphics.Typeface
import android.view.View
import android.widget.FrameLayout
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.databinding.BindingAdapter
import com.google.android.material.progressindicator.CircularProgressIndicator
import com.netacom.base.chat.android_utils.FileUtils
import com.netacom.base.chat.android_utils.TimeUtils
import com.netacom.base.chat.imageloader.loadImage
import com.netacom.base.chat.util.DateTimeUtils
import com.netacom.base.chat.util.isNotNull
import com.netacom.full.R
import com.netacom.full.define.MessagePosition
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.MessageStatusType
import com.netacom.lite.define.MessageStatusType.MESSAGE_DELIVERED
import com.netacom.lite.define.MessageStatusType.MESSAGE_FAILED
import com.netacom.lite.define.MessageStatusType.MESSAGE_RECEIVED
import com.netacom.lite.define.MessageStatusType.MESSAGE_SEEN
import com.netacom.lite.define.MessageStatusType.MESSAGE_SENDING
import com.netacom.lite.define.MessageStatusType.MESSAGE_SENT
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_AUDIO
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_FILE
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_FORWARD
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_IMAGE
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_LOCATION
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_REPLY
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_STICKER
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_TEXT
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_VIDEO
import com.netacom.lite.entity.database.attachment.CallAttachment
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.util.AppUtils

@BindingAdapter("messageStatus")
fun AppCompatImageView.messageStatus(statusMessage: Long?) {
    when (statusMessage) {
        MESSAGE_SENDING -> setImageResource(R.drawable.ic_vector_loading)
        MESSAGE_SENT -> setImageResource(R.drawable.ic_single_check_gray)
        MESSAGE_RECEIVED -> setImageResource(R.drawable.ic_double_check_gray)
        MESSAGE_DELIVERED, MESSAGE_SEEN -> setImageResource(R.drawable.ic_double_check)
        MESSAGE_FAILED -> setImageResource(R.drawable.ic_sent_fail)
        else -> setImageResource(android.R.color.transparent)
    }
//    setTint(ThemeUtils.getColorTheme(context))
}

@BindingAdapter("messageContent")
fun AppCompatTextView.messageContent(neMessage: NeMessage) {
    if (neMessage.status == MessageStatusType.MESSAGE_DELETED) {
        setTextColor(ContextCompat.getColor(context, R.color.color_grey_959595))
        setTypeface(null, Typeface.ITALIC)
        text = context.resources.getString(R.string.message_has_been_removed)
        setBackgroundResource(R.drawable.bg_message_deleted)
        backgroundTintList = null
    }
}

@BindingAdapter("separatorStatus")
fun View.separatorStatus(messagePosition: MessagePosition?) {
    messagePosition?.let { _messagePosition ->
        isVisible = _messagePosition == MessagePosition.SINGLE || _messagePosition == MessagePosition.TOP
    }
}

@BindingAdapter(value = ["isSingle", "messagePosition"], requireAll = false)
fun AppCompatTextView.showName(isSingle: Boolean, messagePosition: MessagePosition?) {
    messagePosition?.let { _messagePosition ->
        if (_messagePosition == MessagePosition.SINGLE || _messagePosition == MessagePosition.TOP) {
            this.isVisible = !isSingle
        } else {
            this.isVisible = false
        }
    }
}

@BindingAdapter("senderTimeStatus")
fun AppCompatTextView.senderTimeStatus(messagePosition: MessagePosition?) {
    messagePosition?.let { _messagePosition ->
        isVisible = _messagePosition == MessagePosition.SINGLE || _messagePosition == MessagePosition.BOTTOM
    }
}

@BindingAdapter("chatTextCall")
fun AppCompatTextView.chatTextCall(isVoice: Boolean) {
    text = if (isVoice) {
        resources.getString(R.string.text_message_voice_call)
    } else {
        resources.getString(R.string.text_message_video_call)
    }
}

@BindingAdapter("chatActionIconCall")
fun AppCompatImageView.chatActionIconCall(isVoice: Boolean?) {
    if (isVoice != null && isVoice) {
        setImageResource(R.drawable.ic_chat_call)
    } else {
        setImageResource(R.drawable.ic_chat_video_call)
    }
}

@BindingAdapter("chatIconCallOut")
fun AppCompatImageView.chatIconCallOut(isCallSuccess: Boolean) {
    if (isCallSuccess) {
        setImageResource(R.drawable.ic_call_out_success)
    } else {
        setImageResource(R.drawable.ic_call_out_fail)
    }
}

@BindingAdapter("chatIconCallIn")
fun AppCompatImageView.chatIconCallIn(isCallSuccess: Boolean) {
    if (isCallSuccess) {
        setImageResource(R.drawable.ic_call_in_success)
    } else {
        setImageResource(R.drawable.ic_call_in_fail)
    }
}

@BindingAdapter("setAudioTime")
fun AppCompatTextView.setAudioTime(message: NeMessage?) {
    message?.attachment?.audio?.let { neAudio ->
        this.text = AppUtils.setAudioTextViewTime(neAudio.duration ?: 0)
    }
}

@BindingAdapter(value = ["isCallSuccess", "isOwner", "callAttachment"], requireAll = false)
fun AppCompatTextView.setTextDuration(isCallSuccess: Boolean, isOwner: Boolean, callAttachment: CallAttachment?) {
    var duration = 0L
    if (isCallSuccess) {
        callAttachment?.let {
            duration = when {
                it.connectedAt != "0" -> (it.stoppedAt?.toLongOrNull() ?: 0L) - (it.connectedAt?.toLongOrNull() ?: 0L)
                it.acceptedAt != "0" -> (it.stoppedAt?.toLongOrNull() ?: 0L) - (it.acceptedAt?.toLongOrNull() ?: 0L)
                else -> 0
            }
        }
    }
    text = if (duration > 0) {
        TimeUtils.second2String(duration)
    } else {
        if (isOwner) {
            context.resources.getString(R.string.text_message_owner_miss_call)
        } else {
            context.resources.getString(R.string.text_message_partner_miss_call)
        }
    }
}

@BindingAdapter("replyUserName")
fun AppCompatTextView.replyUserName(neMessage: NeMessage) {
    text = if (neMessage.attachment?.replyMessage?.owner != null && neMessage.attachment?.replyMessage?.owner?.id != null) {
        String.format(
            context.resources.getString(R.string.message_reply_for_user),
            neMessage.attachment?.replyMessage?.owner?.getDisplayName
                ?: context.resources.getString(R.string.unknown_user)
        )
    } else {
        String.format(
            context.resources.getString(R.string.message_reply_for_user),
            context.resources.getString(R.string.unknown_user)
        )
    }
}

@BindingAdapter("setTime")
fun AppCompatTextView.setTime(time: Long) {
    text = DateTimeUtils.ParseDateTimeToString.parseJustTime(DateTimeUtils.getMillisFromNano(time))
}

@BindingAdapter("loadSticker")
fun AppCompatImageView.loadSticker(neMessage: NeMessage) {
    loadImage(FileUtils.getFileByPath(neMessage.attachment?.stickerUrl), R.drawable.ic_default_sticker)
}

@BindingAdapter("messageCopyType")
fun FrameLayout.messageCopy(messageType: Int) {
    when (messageType) {
        MESSAGE_TYPE_TEXT, MESSAGE_TYPE_REPLY, MESSAGE_TYPE_FORWARD -> {
            this.isVisible = true
        }
        MESSAGE_TYPE_STICKER, MESSAGE_TYPE_IMAGE, MESSAGE_TYPE_AUDIO, MESSAGE_TYPE_VIDEO, MESSAGE_TYPE_FILE, MESSAGE_TYPE_LOCATION -> {
            this.isGone = true
        }
    }
}

@BindingAdapter("messageForwardType")
fun FrameLayout.messageForward(messageType: Int) {
    when (messageType) {
        MESSAGE_TYPE_TEXT, MESSAGE_TYPE_AUDIO, MESSAGE_TYPE_REPLY, MESSAGE_TYPE_FORWARD, MESSAGE_TYPE_IMAGE, MESSAGE_TYPE_VIDEO, MESSAGE_TYPE_FILE, MESSAGE_TYPE_LOCATION -> {
            this.isVisible = true
        }
        else -> {
            this.isGone = true
        }
    }
}

@BindingAdapter("messageDownloadType")
fun FrameLayout.messageDownload(messageType: Int) {
    when (messageType) {
        MESSAGE_TYPE_IMAGE, MESSAGE_TYPE_AUDIO, MESSAGE_TYPE_VIDEO, MESSAGE_TYPE_FILE -> {
            this.isVisible = true
        }
        else -> {
            this.isGone = true
        }
    }
}

@SuppressLint("SetTextI18n")
@BindingAdapter("messageFileName")
fun AppCompatTextView.messageFileName(neMessage: NeMessage) {
    if (neMessage.status == MessageStatusType.MESSAGE_DELETED) {
        setTextColor(ContextCompat.getColor(context, R.color.color_grey_959595))
        setTypeface(null, Typeface.ITALIC)
        text = context.resources.getString(R.string.message_has_been_removed)
        setBackgroundResource(R.drawable.bg_message_deleted)
        backgroundTintList = null
    } else {
        neMessage.attachment?.file?.let {
            text = "${it.name}.${it.extension}"
        }
    }
}

@BindingAdapter("messageFileSize")
fun AppCompatTextView.messageFileSize(neMessage: NeMessage) {
    if (neMessage.status == MessageStatusType.MESSAGE_DELETED) {
        isGone = true
    } else {
        neMessage.attachment?.file?.let {
            text = it.sizeText
        }
    }
}

@BindingAdapter(value = ["groupType", "messagePosition"], requireAll = false)
fun AppCompatTextView.showName(groupType: Int, messagePosition: MessagePosition?) {
    when (messagePosition) {
        MessagePosition.SINGLE, MessagePosition.TOP -> {
            isVisible = GroupType.GROUP_TYPE_PUBLIC != groupType
        }
        MessagePosition.MID, MessagePosition.BOTTOM -> {
            isGone = true
        }
    }
}

@BindingAdapter("setVideoTime")
fun AppCompatTextView.setVideoTime(message: NeMessage?) {
    message?.attachment?.video?.let { neVideo ->
        this.text = TimeUtils.second2String(neVideo.duration ?: 0)
    }
}

@BindingAdapter("setVideoCancel")
fun AppCompatImageView.setVideoCancel(message: NeMessage?) {
    this.isVisible = message?.attachment?.video?.compress?.isNotNull == true || message?.attachment?.video?.progress?.isNotNull ?: false
}

@BindingAdapter("setVideoCompressProgress")
fun AppCompatTextView.setVideoCompressProgress(message: NeMessage?) {
    if (message?.type == MESSAGE_TYPE_VIDEO) {
        if (message.attachment?.video?.compress?.isNotNull == true) {
            this.text = "${message.attachment?.video?.compress} %"
            this.isVisible = true
        } else {
            this.text = ""
            this.isVisible = false
        }
    }
}

@BindingAdapter("setVideoUploadProgress")
fun AppCompatTextView.setVideoUploadProgress(message: NeMessage?) {
    if (message?.type == MESSAGE_TYPE_VIDEO) {
        if (message.attachment?.video?.progress?.isNotNull == true) {
            this.text = message.attachment?.video?.progress
            this.isVisible = true
        } else {
            this.text = ""
            this.isVisible = false
        }
    }
}

@BindingAdapter(value = ["neMessage", "index"], requireAll = false)
fun CircularProgressIndicator.setImageByIndexCompressProgress(message: NeMessage?, index: Int) {
    if (message?.type == MESSAGE_TYPE_IMAGE) {
        this.isVisible = message.attachment?.images.isNotNull && message.attachment?.images?.get(index)?.progress?.isNotNull == true
    }
}

@BindingAdapter(value = ["neMessage", "index"], requireAll = false)
fun AppCompatTextView.setImageByIndexUploadProgress(message: NeMessage, index: Int) {
    if (message.type == MESSAGE_TYPE_IMAGE) {
        if (message.attachment?.images.isNotNull && message.attachment?.images?.get(index)?.progress?.isNotNull == true) {
            this.text = message.attachment?.images?.get(index)?.progress
            this.isVisible = true
        } else {
            this.text = ""
            this.isVisible = false
        }
    }
}
